/*
 * @stratKGSFrame.java Version 1.1 03/30/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import horizon.strat.stratListStruct;
import horizon.strat.gui.stratTable;

/** Class stratKGSFrame
 *  <p> This Class will allow the user to load KGS Well/Outcrop Stratigraphic
 *      Units Picks from multiple sources.
 *
 *  @version 1.1 03/30/2009
 *  @author  John R. Victorine
 */

public class stratKGSFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable      notifier = null; // Observable Object
  private stratListStruct stList   = null; // User's Stratigraphic Units Data

  // Global Variables

  private stratListStruct stKGS    = null; // KGS Stratigraphic Units Data

  private stratTable      pUsers   = null; // User's Stratigraphic Units Table
  private stratTable      pKGS     = null; // KGS Stratigraphic Units Table

  private stratListStruct stKGSStandard = null; // Standard KGS Strat Units

  // Add KGS Data Variables

  private static final int _REPLACE = 0;
  private static final int _ADD     = 1;
  private static final int _MERGE   = 2;

  private int iMode = _REPLACE;

  // Source Name Variables

  private int          iSources  = 0;    // Total of Source Names
  private String       sources[] = null; // Array of Source Names

  private int          iTotal    = 0;
  private JPanel       pSource[] = null; // Panel to hold the Source Name Row
  private JRadioButton rb[]      = null; // Source Name Radiobutton

  // Global Frame Widgets

  // Move by Buttons

  private JRadioButton rbReplace = new JRadioButton();
  private JRadioButton rbAdd     = new JRadioButton();
  private JRadioButton rbMerge   = new JRadioButton();

  // Move KGS Stratigraphic Units to User's Stratigraphic Units

  private JButton btnMove      = new JButton();
  private JButton btnMoveAll   = new JButton();
  private JButton btnClearKGS  = new JButton();

  // Remove Stratigraphic Units from User's List

  private JButton btnClearUser = new JButton();

  private JButton btnRemove    = new JButton();
  private JButton btnRemoveAll = new JButton();

  private JButton btnLoad      = new JButton();
  private JButton btnClose     = new JButton();

  /** Constructor stratKGSFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stKGS    = KGS Stratigraphic Units Data List Structure
   *  @param stList   = User's Stratigraphic Units Data List Structure
   */

  public stratKGSFrame( Observable notifier,
                        stratListStruct stKGS,
                        stratListStruct stList )
  {
    try
    {
      this.notifier = notifier;
      this.stKGS    = stKGS;
      this.stList   = stList;

      this.stKGSStandard = kgs.io.kgsIORead.ReadKGSStratUnits(
                             horizon.strat.io.ReadStratUnitsXMLFile.URL );
      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlKGS          = new JPanel();
    JPanel pnlKGSTop       = new JPanel();
    JPanel pnlSources      = new JPanel();

    JScrollPane scrollSources = new JScrollPane();

    JPanel pnlKGSCenter    = new JPanel();
    JPanel pnlSelectMode   = new JPanel();
    JPanel pnlKGSList      = new JPanel();
    JPanel pnlKGSBase      = new JPanel();

    JScrollPane scrollKGS  = new JScrollPane();
    ButtonGroup groupMode  = new ButtonGroup();

    JPanel pnlUsers        = new JPanel();
    JPanel pnlUsersButtons = new JPanel();
    JPanel pnlSpace1       = new JPanel();
    JPanel pnlRemove       = new JPanel();
    JPanel pnlLoad         = new JPanel();

    JScrollPane scrollUsers = new JScrollPane();

    ButtonGroup group      = new ButtonGroup();

    // Build Frame

    TitledBorder titledBorderKGS = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "KGS Stratigraphic Units:");

    TitledBorder titledBorderStrat = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "User's Stratigraphic Units:");

    TitledBorder titledBorderAdd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Add to User's Stratigraphic Units List:");

    titledBorderKGS.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderStrat.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderAdd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setTitle("Move/Merge KGS Data.");
    this.getContentPane().setLayout(new GridLayout(2,1));

    // KGS Panel

    pnlKGS.setBorder(titledBorderKGS);
    pnlKGS.setLayout(new BorderLayout());

    // -- KGS Sources Panel

    pnlKGSTop.setPreferredSize(new Dimension(180, 10));
    pnlKGSTop.setLayout(new BorderLayout());

    // Add each source to the list

    sources = horizon.strat.stratUtility.getSourceNames( stKGS );
    if (sources != null) iSources = sources.length;

    iTotal = iSources;

    if (iTotal > 0)
    {
      if (iSources > 10)
      {
        pnlSources.setPreferredSize(new Dimension(150, iSources*25));
        pnlSources.setLayout(new GridLayout(iSources, 1));
      }
      else
      {
        pnlSources.setPreferredSize(new Dimension(150, 250));
        pnlSources.setLayout(new GridLayout(10,1));
      }

      pSource = new JPanel[iTotal];
      rb      = new JRadioButton[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        pSource[i]    = new JPanel();
        pSource[i].setLayout(new BorderLayout());
        pSource[i].setBorder(BorderFactory.createEtchedBorder());

        rb[i] = new JRadioButton();
        rb[i].setFont(new java.awt.Font("Dialog", 1, 11));
        rb[i].setHorizontalAlignment(SwingConstants.LEFT);
        if (i==0) rb[i].setSelected(true);
        rb[i].setText(sources[i]);
        rb[i].addActionListener(this);

        group.add(rb[i]);

        // Attach Source Name Row

        pnlSources.add(pSource[i], null);
        pSource[i].add(rb[i],      BorderLayout.WEST);
      }
    }

    // -- KGS Stratigraphic Units Panel

    pnlKGSCenter.setLayout(new BorderLayout());

    // .. Set Select Stratigraphic Units Mode Panel

    pnlSelectMode.setLayout(new GridLayout());
    pnlSelectMode.setBorder(titledBorderAdd);

    rbReplace.setFont(new java.awt.Font("Dialog", 1, 11));
    rbReplace.setHorizontalAlignment(SwingConstants.CENTER);
    rbReplace.setSelected(true);
    rbReplace.setText("Remove & Replace");
    rbReplace.addActionListener(this);

    rbAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    rbAdd.setHorizontalAlignment(SwingConstants.CENTER);
    rbAdd.setText("Add to List");
    rbAdd.addActionListener(this);

    rbMerge.setFont(new java.awt.Font("Dialog", 1, 11));
    rbMerge.setHorizontalAlignment(SwingConstants.CENTER);
    rbMerge.setText("Add New Units Only");
    rbMerge.addActionListener(this);

    // .. KGS Stratigraphic Units List Panel

    pnlKGSList.setLayout(new BorderLayout());

    pKGS      = new stratTable( stKGS );
    scrollKGS = pKGS.getScrollPane();

    // .. KGS Buttons Panel

    btnMove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnMove.setText("Add");
    btnMove.addActionListener(this);

    btnMoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnMoveAll.setText("Add All");
    btnMoveAll.addActionListener(this);

    btnClearKGS.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClearKGS.setText("Clear Selection");
    btnClearKGS.addActionListener(this);

    // User's Panel

    pnlUsers.setBorder(titledBorderStrat);
    pnlUsers.setLayout(new BorderLayout());

    // -- Stratigraphic Units Table

    pUsers      = new stratTable(stList);
    scrollUsers = pUsers.getScrollPane();
    pUsers.setFilter( cmn.cmnStruct.OFF );

    // -- Stratigraphic Units Buttons Panel

    pnlUsersButtons.setPreferredSize(new Dimension(125, 10));
    pnlUsersButtons.setLayout(new GridLayout(3,1));

    btnClearUser.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClearUser.setPreferredSize(new Dimension(120, 25));
    btnClearUser.setText("Clear Selection");
    btnClearUser.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setPreferredSize(new Dimension(120, 25));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setPreferredSize(new Dimension(120, 25));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    btnLoad.setFont(new java.awt.Font("Dialog", 1, 11));
    btnLoad.setPreferredSize(new Dimension(120, 25));
    btnLoad.setText("Load Data");
    btnLoad.addActionListener(this);

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setPreferredSize(new Dimension(120, 25));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // Attach Widgets to Frame

    this.getContentPane().add(pnlKGS,   null);
    pnlKGS.add(pnlKGSTop,               BorderLayout.WEST);
    pnlKGSTop.add(scrollSources,        BorderLayout.CENTER);
    scrollSources.getViewport().add(pnlSources, null);

    pnlKGS.add(pnlKGSCenter,            BorderLayout.CENTER);
    pnlKGSCenter.add(pnlSelectMode,     BorderLayout.NORTH);
    pnlSelectMode.add(rbReplace,        null);
    pnlSelectMode.add(rbAdd,            null);
    pnlSelectMode.add(rbMerge,          null);

    groupMode.add(rbReplace);
    groupMode.add(rbAdd);
    groupMode.add(rbMerge);

    pnlKGSCenter.add(pnlKGSList,        BorderLayout.CENTER);
    pnlKGSList.add(scrollKGS,           BorderLayout.CENTER);

    pnlKGSCenter.add(pnlKGSBase,        BorderLayout.SOUTH);
    pnlKGSBase.add(btnMove,             null);
    pnlKGSBase.add(btnMoveAll,          null);
    pnlKGSBase.add(btnClearKGS,         null);

    this.getContentPane().add(pnlUsers, null);
    pnlUsers.add(scrollUsers,           BorderLayout.CENTER);

    pnlUsers.add(pnlUsersButtons,       BorderLayout.EAST);
    pnlUsersButtons.add(pnlSpace1,      null);
    pnlSpace1.add(btnClearUser,         null);

    pnlUsersButtons.add(pnlRemove,      null);
    pnlRemove.add(btnRemove,            null);
    pnlRemove.add(btnRemoveAll,         null);

    pnlUsersButtons.add(pnlLoad,        null);
    pnlLoad.add(btnLoad,                null);
    pnlLoad.add(btnClose,               null);

    // Set Size & Display the Frame

    this.setSize(new Dimension(650, 650));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getDownloadData()
   * <p> This method will retrieve the selected data for download.
   * @return stList = List of stratigraphic units from the KGS database
   */

  public stratListStruct getDownloadData()
  {
    stList.sTOPS  = new String("Kansas Geological Survey Database");
    stList.sTOPDR = new String("Log Depth");

    return ( stList );
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add the selected stratigraphic units to the User's
   *     stratigraphic units list
   */

  private void add()
  {
    stratListStruct st    = null;
    String          sName = cmn.cmnString.UniqueName();

    st = pKGS.getSelected( horizon.strat.gui.stratTable._BY_TABLE );

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].depthStart > 0.0)
        {
          if (st.stItem[i].stratKEY.equals("0"))
          {
            st.stItem[i].stratKEY = new String(st.stItem[i].sKEY);
            st.stItem[i].sKEY     = new String(sName+"_"+i);
          }

          switch (iMode)
          {
            case _REPLACE:
              stList =
                horizon.strat.stratUtility.ReplaceWith(st.stItem[i], stList);
              stList =
                horizon.strat.stratUtility.fillBaseDepth(stList, stKGSStandard);
              break;
            case _ADD:
              stList = horizon.strat.stratUtility.add(st.stItem[i], stList);
              break;
            case _MERGE:
              stList =
                  horizon.strat.stratUtility.addNewOnly(st.stItem[i], stList);
              stList =
                horizon.strat.stratUtility.fillBaseDepth(stList, stKGSStandard);
              break;
          }
        }
      }

      pUsers.clear();
      pUsers.repopulateList( stList );
    }

    pKGS.clear();
  }

  /** Method addAll()
   * <p> This method will add the sources stratigraphic units to the User's
   *     stratigraphic units list
   */

  private void addAll()
  {
    stratListStruct st    = null;
    String          sName = cmn.cmnString.UniqueName();

    st = pKGS.getSelected( horizon.strat.gui.stratTable._BY_SOURCE );

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].depthStart > 0.0)
        {
          if (st.stItem[i].stratKEY.equals("0"))
          {
            st.stItem[i].stratKEY = new String(st.stItem[i].sKEY);
            st.stItem[i].sKEY     = new String(sName+"_"+i);
          }

          switch (iMode)
          {
            case _REPLACE:
              stList =
                horizon.strat.stratUtility.ReplaceWith(st.stItem[i], stList);
              stList =
                horizon.strat.stratUtility.fillBaseDepth(stList, stKGSStandard);
              break;
            case _ADD:
              stList = horizon.strat.stratUtility.add(st.stItem[i], stList);
              break;
            case _MERGE:
              stList =
                  horizon.strat.stratUtility.addNewOnly(st.stItem[i], stList);
              stList =
                horizon.strat.stratUtility.fillBaseDepth(stList, stKGSStandard);
              break;
          }
        }
      }

      pUsers.clear();
      pUsers.repopulateList( stList );
    }

    pKGS.clear();
  }

  /** Method removeUnit()
   * <p> This method will remove the selected units from User's List
   */

  private void removeUnit()
  {
    stratListStruct st    = null;

    st = pUsers.getSelected( horizon.strat.gui.stratTable._BY_TABLE );

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        stList = horizon.strat.stratUtility.remove( st.stItem[i].sKID,
                                                    st.stItem[i].sKEY,
                                                    stList);
      }
    }

    pUsers.clear();
    pUsers.repopulateList( stList );
  }

  /** Method removeAllUnits()
   * <p> This method will remove all units from User's List
   */

  private void removeAllUnits()
  {
    if (stList != null)
      stList.delete();
    stList = null;

    pUsers.repopulateList( stList );
  }

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier = null; // Observable Object
    stList   = null; // User's Stratigrpahic Units Data List Structure

    // Global Variables

    stKGS    = null; // KGS Stratigraphic Units Data List Structure

    pUsers   = null; // User's Stratigraphic Units Table
    pKGS     = null; // KGS Stratigraphic Units Table

    if (stKGSStandard != null)
      stKGSStandard.delete();
    stKGSStandard = null;

    // Global Frame Widgets

    // Move by Buttons

    rbReplace = null;
    rbAdd     = null;
    rbMerge   = null;

    // Move KGS Stratigraphic Units to User's Stratigraphic Units

    btnMove      = null;
    btnMoveAll   = null;
    btnClearKGS  = null;

    // Remove Stratigraphic Units from User's List

    btnClearUser = null;

    btnRemove    = null;
    btnRemoveAll = null;

    btnLoad      = null;
    btnClose     = null;

    // Source Name Variables

    sources  = null; // Array of Source Names
    pSource = null; // Panel to hold the Source Name Row
    rb      = null; // Source Name Radiobutton

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    for (int i=0; i<iTotal; i++)
    {
      if (event.getSource() == rb[i])
      {
        pKGS.clear();
        pKGS.setSource( sources[i] );
      }
    }

    // Move by Buttons

    if (event.getSource() == rbReplace) { iMode = _REPLACE; }
    if (event.getSource() == rbAdd)     { iMode = _ADD; }
    if (event.getSource() == rbMerge)   { iMode = _MERGE; }

    // KGS Stratigraphic Units List Actions

    if (event.getSource() == btnMove)     { add(); }
    if (event.getSource() == btnMoveAll)  { addAll(); }

    if (event.getSource() == btnClearKGS) { pKGS.clear(); }

    // User's Stratigraphic Units List Actions

    if (event.getSource() == btnClearUser) { pUsers.clear(); }

    if (event.getSource() == btnRemove)    { removeUnit(); }
    if (event.getSource() == btnRemoveAll) { removeAllUnits(); }

    if (event.getSource() == btnLoad)
    {
      notifier.notifyObservers(new String("KGS Source Selected"));
    }
    if (event.getSource() == btnClose) { close(); }
  }
}

/*
 *  @version 1.1 03/30/2009
 *  @author  John Victorine
 */
