/*
 * @stratDataEntryPanel.java Version 1.1 09/27/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observer;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import cmn.cmnStruct;
import iqstrat.iqstratStruct;
import iqstrat.iqstratHeadersStruct;

import horizon.strat.stratProjectStruct;
import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;
import horizon.strat.gui.stratListTable;
import horizon.strat.gui.stratPlotICSFrame;
import horizon.strat.gui.stratPlotStateFrame;
//import horizon.strat.gui.stratTemplateSaveFrame;
//import horizon.strat.gui.stratTemplateFrame;
//import horizon.strat.gui.stratUnitsSaveFrame;

/** Class stratDataEntryPanel
 *  <p> This Class will allow the user to enter the Formation Tops onto the
 *      stratigraphic Plot
 *
 *  @version 1.1 09/27/2007
 *  @author  John R. Victorine
 */

public class stratDataEntryPanel extends JPanel implements ActionListener, Observer
{
  // Input Variable

  private Observable      notifier = null; // Observable
  private iqstratStruct   stStruct = null; // Global IQSTRAT Data Structure

  // Global Variables

  private Observable      obsrvble = null; // Observable

  private cmnStruct       stCMN    = null; // Global Common Data Structure

  private stratListStruct stStrat  = null; // User Defined Stratigraphic Units
  private stratListStruct stKGS    = null; // KGS Stratigraphic Units
  private stratListStruct stICS    = null; // ICS Stratigraphic Units

  private iqstratHeadersStruct stInfo   = null; // Headers for Strat List
  private stratListStruct      stList   = null; // Stratigraphic Units List Structure
  private stratStruct          stRecord = null; // Stratigraphic Unit Data Structure

//  private stratTemplateSaveFrame pChart = null; // Save Stratigrahic Chart Frame
//  private stratTemplateFrame     pChange = null; // Change Stratigraphic Chart
//  private stratUnitsSaveFrame    pSaveUnits = null; // Save General

  // Stratigraphic Chart to Display

  private static final int _NONE         = -1;
  private static final int _DEFAULT      = 0;
  private static final int _USER_DEFINED = 1;

  private int    iChart     = _DEFAULT;

  // Action

  private static final int _ADD    = 0;
  private static final int _MODIFY = 1;

  private int    iAction    = _ADD;

  private String sKID       = "0"; // KGS Primary KEY of Record
  private String sKEY       = "0"; // unique KEY of Record
  private String sid        = "0"; // unique order id
//  private int    id         = 0;   // unique order id

  private int    iLevel     = horizon.strat.stratStruct._POOR; // Confidence
  private int    iRank      = horizon.strat.stratStruct._NONE; // Rank

  private String sName      = "";  // Strigraphic Name
  private String sName2     = "";  // Alternate Name

  private String sEon       = "";  // Eon Stratigraphic Unit Name
  private String sEra       = "";  // Era Stratigraphic Unit Name
  private String system     = "";  // System Stratigraphic Unit Name
  private String series     = "";  // Series Stratigraphic Unit Name
  private String subSystem  = "";  // Substage Stratigraphic Unit Name
  private String subSeries  = "";  // Subseries Stratigraphic Unit Name

  private String stage      = "";  // Stage Stratigraphic Unit Name
  private String sprGroup   = "";  // Super Group Stratigraphic Unit Name
  private String sGroup     = "";  // Group Stratigraphic Unit Name
  private String subGroup   = "";  // Subgroup Stratigraphic Unit Name
  private String sFormation = "";  // Formation Stratigraphic Unit Name

  private double dAgeStart  = 0.0; // Starting Age
  private double dStartCorr = 0.0; // Correction

  private double dAgeEnd    = 0.0; // Ending Age
  private double dEndCorr   = 0.0; // Correction

  private double depthStart = 0.0; // Starting depth
  private double depthEnd   = 0.0; // Ending depth

  private stratListTable      pTable = null;
  private stratPlotICSFrame   pICS   = null;
  private stratPlotStateFrame pState = null;

  // Stratigraphic Depth Range

  private JTextField txtTop  = null;
  private JTextField txtBase = null;

  // Global Panel Widgets

  // Stratigraphic Unit List Buttons

  private JButton btnAdd            = new JButton();       // Add Record
  private JButton btnClear          = new JButton();       // Clear Data

  private JButton btnModify         = new JButton();       // Modify Record
  private JButton btnRemove         = new JButton();       // Delete Record
  private JButton btnRemoveAll      = new JButton();       // Delete All Records
  private JButton btnSaveAs         = new JButton();       // Save as General
  private JButton btnTemplate       = new JButton();       // Save as Template

  private JButton btnICS            = new JButton();       // ICS Strat Chart
  private JButton btnSelect         = new JButton();       // Select State Chart
  private JButton btnState          = new JButton();       // State Strat Chart

  // Level of Confidence

  private JRadioButton rbGold       = new JRadioButton();  // Excellent
  private JRadioButton rbSilver     = new JRadioButton();  // Good
  private JRadioButton rbCopper     = new JRadioButton();  // Fair
  private JRadioButton rbLead       = new JRadioButton();  // Poor

  // Rank of Stratigraphic Unit

  private JRadioButton rbUnknown    = new JRadioButton();  // Unknown
  private JRadioButton rbEon        = new JRadioButton();  // Eon
  private JRadioButton rbEra        = new JRadioButton();  // Era
  private JRadioButton rbSystem     = new JRadioButton();  // System
  private JRadioButton rbSubsystem  = new JRadioButton();  // Subsystem
  private JRadioButton rbSeries     = new JRadioButton();  // Series
  private JRadioButton rbSubseries  = new JRadioButton();  // Subseries
  private JRadioButton rbStage      = new JRadioButton();  // Stage
  private JRadioButton rbSuperGroup = new JRadioButton();  // Super group
  private JRadioButton rbGroup      = new JRadioButton();  // Group
  private JRadioButton rbSubgroup   = new JRadioButton();  // Subgroup
  private JRadioButton rbFormation  = new JRadioButton();  // Formation
  private JRadioButton rbMember     = new JRadioButton();  // Member
  private JRadioButton rbBed        = new JRadioButton();  // Bed

  // Strigraphic Unit Names

  private JTextField txtEon         = new JTextField();    // Eon
  private JTextField txtEra         = new JTextField();    // Era

  private JTextField txtSystem      = new JTextField();    // System
  private JTextField txtSubsystem   = new JTextField();    // Subsystem
  private JTextField txtSeries      = new JTextField();    // Series
  private JTextField txtStage       = new JTextField();    // Stage
  private JTextField txtSprGroup    = new JTextField();    // Super Group
  private JTextField txtGroup       = new JTextField();    // Group
  private JTextField txtSubgroup    = new JTextField();    // Subgroup

  private JTextField txtFormation   = new JTextField();    // Formation

  private JTextField txtName        = new JTextField();    // Stratigraphic Name
  private JTextField txtName2       = new JTextField();    // Alternate Name

  // Begining Age

  private JTextField txtAgeBegin        = new JTextField();
  private JTextField txtCorrectionBegin = new JTextField();

  // Ending Age

  private JTextField txtAgeEnd          = new JTextField();
  private JTextField txtCorrectionEnd   = new JTextField();

  /** Constructor stratDataEntryPanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public stratDataEntryPanel( Observable notifier, iqstratStruct stStruct)
  {
    try
    {
      this.notifier = notifier;
      this.stStruct = stStruct;

      this.stICS    = stStruct.stICS;
      this.stKGS    = stStruct.stKGS;
      this.stCMN    = stStruct.stCMN;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop         = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlListButtons = new JPanel();
    JPanel pnlButtons     = new JPanel();
    JPanel pnlUnitNames   = new JPanel();
    JPanel pnlUnits       = new JPanel();
    JPanel pnlRank        = new JPanel();
    JPanel pnlLevel       = new JPanel();
    JPanel pnlEon         = new JPanel();
    JPanel pnlEra         = new JPanel();
    JPanel pnlSystem      = new JPanel();
    JPanel pnlSubsystem   = new JPanel();
    JPanel pnlSeries      = new JPanel();
    JPanel pnlStage       = new JPanel();
    JPanel pnlSprGroup    = new JPanel();
    JPanel pnlGroup       = new JPanel();
    JPanel pnlSubgroup    = new JPanel();
    JPanel pnlFormation   = new JPanel();
    JPanel pnlNames       = new JPanel();
    JPanel pnlAltName     = new JPanel();
    JPanel pnlTextButtons = new JPanel();
    JPanel pnlName        = new JPanel();
    JPanel pnlBeginAge    = new JPanel();
    JPanel pnlAgeRange    = new JPanel();
    JPanel pnlEndAge      = new JPanel();

    JPanel pnlICSUnits    = new JPanel();
    JPanel pnlICSButton   = new JPanel();
    JPanel pnlStratUnits  = new JPanel();
    JPanel pnlWest        = new JPanel();
    JPanel pnlKSButtons   = new JPanel();

    JLabel lblPMBegin     = new JLabel();
    JLabel lblPMEnd       = new JLabel();

    JScrollPane scroll    = new JScrollPane();
    JScrollPane scrollUnitNames = new JScrollPane();

    ButtonGroup groupLevel = new ButtonGroup();
    ButtonGroup groupRank = new ButtonGroup();

    TitledBorder titledBorderNames = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Stratigraphic Name is apart of, ");
//        "Stratigraphic Unit Names");

    TitledBorder titledBorderICS = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "International Commission on Stratigraphy");

    TitledBorder titledBorderList = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151)),
        "Stratigraphic Units Selected:");

    TitledBorder titledBorderRank = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Rank:");

    TitledBorder titledBorderLevel = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Confidence:");

    TitledBorder titledBorderEon = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Eon:");

    TitledBorder titledBorderEra = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Era:");

    TitledBorder titledBorderSystem = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "System:");

    TitledBorder titledBorderSubsystem = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Subsystem:");

    TitledBorder titledBorderSeries = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Series:");

    TitledBorder titledBorderStage = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Stage:");

    TitledBorder titledBorderSprGroup = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Super Group:");

    TitledBorder titledBorderGroup = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Group:");

    TitledBorder titledBorderSubgroup = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Subgroup:");

    TitledBorder titledBorderFormation = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Formation:");

    TitledBorder titledBorderName = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Stratigraphic Name:");

    TitledBorder titledBorderName2 = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Alternate Name:");

    TitledBorder titledBorderBeginAge = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Begin Age (Ma):");

    TitledBorder titledBorderEndAge = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "End Age (Ma):");

    titledBorderList.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderRank.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderLevel.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderNames.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderICS.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEon.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEra.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSystem.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSubsystem.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSeries.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderStage.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSprGroup.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderGroup.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSubgroup.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderFormation.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderName.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderName2.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderBeginAge.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEndAge.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Create an Observable

    obsrvble = new stratDataEntryPanelObservable();

    // Add this Dialog to the notifier as observer

    obsrvble.addObserver(this);

    // Build Panel

    this.setLayout(new BorderLayout());
    pnlTop.setPreferredSize(new Dimension(10, 400));
    pnlTop.setLayout(new BorderLayout());

    // Buttons Panel

    pnlTextButtons.setLayout(new GridLayout());

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    // Data Panels

    pnlWest.setLayout(new BorderLayout());

    // -- Radio Button Panels

    pnlButtons.setLayout(new BorderLayout());

    // .. Level of Confidence Panel

    pnlLevel.setBorder(titledBorderLevel);
    pnlLevel.setLayout(new GridLayout(2,2));

    rbGold.setBackground(new Color(217, 217, 25));
    rbGold.setFont(new java.awt.Font("Dialog", 1, 11));
    rbGold.setSelected(false);
    rbGold.setText("Excellent");
    rbGold.addActionListener(this);

    rbSilver.setBackground(new Color(230, 232, 250));
    rbSilver.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSilver.setText("Good");
    rbSilver.addActionListener(this);

    rbCopper.setBackground(new Color(217, 135, 25));
    rbCopper.setFont(new java.awt.Font("Dialog", 1, 11));
    rbCopper.setText("Fair");
    rbCopper.addActionListener(this);

    rbLead.setBackground(new Color(100, 100, 100));
    rbLead.setFont(new java.awt.Font("Dialog", 1, 11));
    rbLead.setSelected(true);
    rbLead.setText("Poor");
    rbLead.addActionListener(this);

    // .. Stratigraphic Rank Panel

    pnlRank.setBorder(titledBorderRank);
    pnlRank.setLayout(new GridLayout(8,2));

    rbUnknown.setFont(new java.awt.Font("Dialog", 1, 11));
    rbUnknown.setSelected(true);
    rbUnknown.setText("Unknown");
    rbUnknown.addActionListener(this);

    rbEon.setFont(new java.awt.Font("Dialog", 1, 11));
    rbEon.setText("Eon");
    rbEon.addActionListener(this);

    rbEra.setFont(new java.awt.Font("Dialog", 1, 11));
    rbEra.setText("Era");
    rbEra.addActionListener(this);

    rbSystem.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSystem.setText("System");
    rbSystem.addActionListener(this);

    rbSubsystem.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSubsystem.setText("Subsystem");
    rbSubsystem.addActionListener(this);

    rbSeries.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSeries.setText("Series");
    rbSeries.addActionListener(this);

    rbSubseries.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSubseries.setText("Subseries");

    rbStage.setFont(new java.awt.Font("Dialog", 1, 11));
    rbStage.setText("Stage");
    rbStage.addActionListener(this);

    rbSuperGroup.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSuperGroup.setText("Super Group");
    rbSuperGroup.addActionListener(this);

    rbGroup.setFont(new java.awt.Font("Dialog", 1, 11));
    rbGroup.setText("Group");
    rbGroup.addActionListener(this);

    rbSubgroup.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSubgroup.setText("Subgroup");
    rbSubgroup.addActionListener(this);

    rbFormation.setFont(new java.awt.Font("Dialog", 1, 11));
    rbFormation.setText("Formation");
    rbFormation.addActionListener(this);

    rbMember.setFont(new java.awt.Font("Dialog", 1, 11));
    rbMember.setText("Member");
    rbMember.addActionListener(this);

    rbBed.setFont(new java.awt.Font("Dialog", 1, 11));
    rbBed.setText("Bed");
    rbBed.addActionListener(this);

    // -- Stratigraphic Names Panel

    pnlNames.setLayout(new GridLayout(2,1));

    // .. Stratigraphic Name Panel

    pnlName.setBorder(titledBorderName);
    pnlName.setLayout(new BorderLayout());

    txtName.setText("");
    txtName.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // .. Alternate Name Panel

    pnlAltName.setBorder(titledBorderName2);
    pnlAltName.setLayout(new BorderLayout());

    txtName2.setText("");
    txtName2.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // Stratigraphic Unit Names Panel

//    pnlUnitNames.setBorder(titledBorderNames);
    pnlUnitNames.setLayout(new BorderLayout());

    // International Commission on Stratigraphy Panel

    pnlICSUnits.setBorder(titledBorderICS);
    pnlICSUnits.setLayout(new GridLayout(3,2));

    // -- Stratigraphic Eon Panel

    pnlEon.setBorder(titledBorderEon);
    pnlEon.setLayout(new BorderLayout());

    txtEon.setEditable(false);
    txtEon.setText("");
    txtEon.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic System Panel

    pnlEra.setBorder(titledBorderEra);
    pnlEra.setLayout(new BorderLayout());

    txtEra.setEditable(false);
    txtEra.setText("");
    txtEra.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic System Panel

    pnlSystem.setBorder(titledBorderSystem);
    pnlSystem.setLayout(new BorderLayout());

    txtSystem.setEditable(false);
    txtSystem.setText("");
    txtSystem.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Subsystem Panel

    pnlSubsystem.setLayout(new BorderLayout());
    pnlSubsystem.setBorder(titledBorderSubsystem);

    txtSubsystem.setEditable(false);
    txtSubsystem.setText("");
    txtSubsystem.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Series Panel

    pnlSeries.setLayout(new BorderLayout());
    pnlSeries.setBorder(titledBorderSeries);

    txtSeries.setEditable(false);
    txtSeries.setText("");
    txtSeries.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- ICS Chart Button

    pnlICSButton.setLayout(new BorderLayout());

    btnICS.setFont(new java.awt.Font("Dialog", 1, 11));
    btnICS.setText("ICS Chart");
    btnICS.addActionListener(this);

    // Stratigraphic Units User Input Panel

    pnlStratUnits.setBorder(titledBorderNames);
    pnlStratUnits.setLayout(new GridLayout(3,2));
//    pnlStratUnits.setLayout(new BorderLayout());

    // -- Stratigraphic Units Container Panel

    pnlUnits.setLayout(new GridLayout(10,1));

    // -- Stratigraphic Stage Panel

    pnlStage.setLayout(new BorderLayout());
    pnlStage.setBorder(titledBorderStage);

    txtStage.setText("");
    txtStage.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Super Group Panel

    pnlSprGroup.setLayout(new BorderLayout());
    pnlSprGroup.setBorder(titledBorderSprGroup);

    txtSprGroup.setText("");
    txtSprGroup.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Group Panel

    pnlGroup.setLayout(new BorderLayout());
    pnlGroup.setBorder(titledBorderGroup);

    txtGroup.setText("");
    txtGroup.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Subgroup Panel

    pnlSubgroup.setLayout(new BorderLayout());
    pnlSubgroup.setBorder(titledBorderSubgroup);

    txtSubgroup.setText("");
    txtSubgroup.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Stratigraphic Formation Panel

    pnlFormation.setLayout(new BorderLayout());
    pnlFormation.setBorder(titledBorderFormation);

    txtFormation.setText("");
    txtFormation.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    // -- Select Stratigraphic Unit Panel

    pnlKSButtons.setLayout(new GridLayout(2,1));

    btnState.setFont(new java.awt.Font("Dialog", 1, 11));
    btnState.setText("1968 Kansas Chart");
    btnState.addActionListener(this);

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setText("Change Chart");
    btnSelect.addActionListener(this);

    // Age of formation Panel

    pnlAgeRange.setLayout(new GridLayout());

    // -- Age From Panel

    pnlBeginAge.setBorder(titledBorderBeginAge);
    pnlBeginAge.setLayout(new GridLayout());

    txtAgeBegin.setText("0.0");
    txtAgeBegin.setColumns(4);
    txtAgeBegin.setHorizontalAlignment(SwingConstants.TRAILING);
    txtAgeBegin.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    txtCorrectionBegin.setText("0.0");
    txtCorrectionBegin.setColumns(4);
    txtCorrectionBegin.setHorizontalAlignment(SwingConstants.TRAILING);
    txtCorrectionBegin.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    lblPMBegin.setFont(new java.awt.Font("Dialog", 1, 12));
    lblPMBegin.setHorizontalAlignment(SwingConstants.CENTER);
    lblPMBegin.setText("+/-");

    // -- Age  To Panel

    pnlEndAge.setBorder(titledBorderEndAge);
    pnlEndAge.setLayout(new GridLayout());

    txtAgeEnd.setText("0.0");
    txtAgeEnd.setColumns(4);
    txtAgeEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtAgeEnd.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    txtCorrectionEnd.setText("0.0");
    txtCorrectionEnd.setColumns(4);
    txtCorrectionEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtCorrectionEnd.addFocusListener(new stratDataEntryPanelFocusAdapter(this));

    lblPMEnd.setFont(new java.awt.Font("Dialog", 1, 12));
    lblPMEnd.setHorizontalAlignment(SwingConstants.CENTER);
    lblPMEnd.setText("+/-");

    // Selected Stratigraphic Units Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBorder(titledBorderList);

    pTable = new stratListTable();
    scroll = pTable.getScrollPane();

    // Stratigraphic Units Button Panel

//    pnlListButtons.setLayout(new GridLayout(5,1));

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setText("Modify");
    btnModify.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    btnSaveAs.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSaveAs.setText("Save for Access");
    btnSaveAs.addActionListener(this);

    btnTemplate.setFont(new java.awt.Font("Dialog", 1, 11));
    btnTemplate.setText("Save As Template");
    btnTemplate.addActionListener(this);

    // Enable or disable buttons depending on the data

    setButtons();

    // Attach Widgets to Frame

    this.add( pnlTop,                    BorderLayout.NORTH );
    pnlTop.add( pnlWest,                 BorderLayout.WEST );
    pnlWest.add( pnlButtons,             BorderLayout.NORTH );
    pnlButtons.add( pnlLevel,            BorderLayout.NORTH );
    pnlLevel.add( rbGold,                null );
    pnlLevel.add( rbCopper,              null );
    pnlLevel.add( rbSilver,              null );
    pnlLevel.add( rbLead,                null );

    groupLevel.add( rbGold );
    groupLevel.add( rbSilver );
    groupLevel.add( rbCopper );
    groupLevel.add( rbLead );

    pnlButtons.add( pnlRank,             BorderLayout.CENTER );
    pnlRank.add( rbUnknown,              null );
    pnlRank.add( rbEon,                  null );
    pnlRank.add( rbEra,                  null );
    pnlRank.add( rbSuperGroup,           null );
    pnlRank.add( rbSystem,               null );
    pnlRank.add( rbGroup,                null );
    pnlRank.add( rbSubsystem,            null );
    pnlRank.add( rbSubgroup,             null );
    pnlRank.add( rbSeries,               null );
    pnlRank.add( rbFormation,            null );
    pnlRank.add( rbSubseries,            null );
    pnlRank.add( rbMember,               null );
    pnlRank.add( rbStage,                null );
    pnlRank.add( rbBed,                  null );

    groupRank.add( rbUnknown );
    groupRank.add( rbEon );
    groupRank.add( rbEra );
    groupRank.add( rbSystem );
    groupRank.add( rbSubsystem );
    groupRank.add( rbSeries );
    groupRank.add( rbSubseries );
    groupRank.add( rbStage );
    groupRank.add( rbSuperGroup );
    groupRank.add( rbGroup );
    groupRank.add( rbSubgroup );
    groupRank.add( rbFormation );
    groupRank.add( rbMember );
    groupRank.add( rbBed );

    pnlWest.add( pnlNames,               BorderLayout.SOUTH );
    pnlNames.add( pnlName,               null );
    pnlName.add( txtName,                BorderLayout.CENTER );

    pnlNames.add( pnlAltName,            null );
    pnlAltName.add( txtName2,            BorderLayout.CENTER );

    pnlTop.add( pnlUnitNames,            BorderLayout.CENTER );
    pnlUnitNames.add( pnlICSUnits,       BorderLayout.NORTH );

    pnlICSUnits.add( pnlEon,             null );
    pnlEon.add( txtEon,                  BorderLayout.SOUTH );

    pnlICSUnits.add( pnlEra,             null );
    pnlEra.add( txtEra,                  BorderLayout.SOUTH );

    pnlICSUnits.add( pnlSystem,          null );
    pnlSystem.add( txtSystem,            BorderLayout.SOUTH );

    pnlICSUnits.add( pnlSubsystem,       null );
    pnlSubsystem.add( txtSubsystem,      BorderLayout.SOUTH );

    pnlICSUnits.add( pnlSeries,          null );
    pnlSeries.add( txtSeries,            BorderLayout.SOUTH );

    pnlICSUnits.add( pnlICSButton,       null );
    pnlICSButton.add( btnICS,            BorderLayout.SOUTH );

    pnlUnitNames.add( pnlAgeRange,       BorderLayout.SOUTH );
    pnlAgeRange.add( pnlBeginAge,        null );
    pnlBeginAge.add( txtAgeBegin,        null );
    pnlBeginAge.add( lblPMBegin,         null );
    pnlBeginAge.add( txtCorrectionBegin, null );

    pnlAgeRange.add( pnlEndAge,          null );
    pnlEndAge.add( txtAgeEnd,            null );
    pnlEndAge.add( lblPMEnd,             null );
    pnlEndAge.add( txtCorrectionEnd,     null );

    pnlUnitNames.add( pnlStratUnits,     BorderLayout.CENTER );

    pnlStratUnits.add( pnlStage,         null );
    pnlStage.add( txtStage,              BorderLayout.SOUTH );

    pnlStratUnits.add( pnlSprGroup,      null );
    pnlSprGroup.add( txtSprGroup,        BorderLayout.SOUTH );

    pnlStratUnits.add( pnlGroup,         null );
    pnlGroup.add( txtGroup,              BorderLayout.SOUTH );

    pnlStratUnits.add( pnlSubgroup,      null );
    pnlSubgroup.add( txtSubgroup,        BorderLayout.SOUTH );

    pnlStratUnits.add( pnlFormation,     null );
    pnlFormation.add( txtFormation,      BorderLayout.SOUTH );

    pnlStratUnits.add( pnlKSButtons,     null );
    pnlKSButtons.add( btnState,          null );
//    pnlKSButtons.add( btnSelect,         null );

    pnlTop.add( pnlTextButtons,          BorderLayout.SOUTH );
    pnlTextButtons.add( btnAdd,          null );
    pnlTextButtons.add( btnClear,        null );

    this.add( pnlCenter,                 BorderLayout.CENTER );
    pnlCenter.add( scroll,               BorderLayout.CENTER );
//    pnlCenter.add( pnlListButtons,       BorderLayout.EAST );
    pnlCenter.add( pnlListButtons,       BorderLayout.SOUTH );
    pnlListButtons.add( btnModify,       null );
    pnlListButtons.add( btnRemove,       null );
    pnlListButtons.add( btnRemoveAll,    null );
//    pnlListButtons.add( btnSaveAs,       null );
//    pnlListButtons.add( btnTemplate,     null );
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getListData()
   * <p> This method will retrieve the Stratigraphic Unit data list structure
   * @return stList = the Stratigraphic Unit data list structure
   */

  public stratListStruct getListData() { return (stList); }

  /** Method getData()
   * <p> This method will get the array of sedimentary structures selected
   * @return st = sedimentary data Structure
   */

  public stratStruct getData()
  {
//    if (stRecord == null)
    if (iAction == _ADD)
    {
      stRecord      = new stratStruct();
      stRecord.sKEY = new String(cmn.cmnString.UniqueName());
    }

    if (!sKID.equals("0"))
      stRecord.sKID = new String(sKID);
    else
      stRecord.sKID = new String(stRecord.sKEY);

    stRecord.sid        = new String(sid);

//    stRecord.id         = id;
    stRecord.iLevel     = iLevel;
    stRecord.iRank      = iRank;

    stRecord.sName      = new String(sName);
    stRecord.sName2     = new String(sName2);

    if (depthEnd == 0.0) depthEnd = depthStart;

    stRecord.dAgeStart  = dAgeStart;
    stRecord.dStartCorr = dStartCorr;

    stRecord.dAgeEnd    = dAgeEnd;
    stRecord.dEndCorr   = dEndCorr;

    stRecord.depthStart = depthStart;
    stRecord.depthEnd   = depthEnd;

    stRecord.sEon       = new String(sEon);
    stRecord.sEra       = new String(sEra);

    stRecord.system     = new String(system);
    stRecord.series     = new String(series);
    stRecord.subSystem  = new String(subSystem);
    stRecord.subSeries  = new String(subSeries);

    stRecord.stage      = new String(stage);
    stRecord.sprGroup   = new String(sprGroup);
    stRecord.sGroup     = new String(sGroup);
    stRecord.subGroup   = new String(subGroup);
    stRecord.sFormation = new String(sFormation);

    if (sName.length() > 0)
    {
      switch (iRank)
      {
        case horizon.strat.stratStruct._STAGE:
          if (stage.length() == 0) stRecord.stage = new String(sName);
          break;
        case horizon.strat.stratStruct._SUPER_GROUP:
          if (sprGroup.length() == 0) stRecord.sprGroup = new String(sName);
          break;
        case horizon.strat.stratStruct._GROUP:
          if (sGroup.length() == 0) stRecord.sGroup = new String(sName);
          break;
        case horizon.strat.stratStruct._SUBGROUP:
          if (subGroup.length() == 0) stRecord.subGroup = new String(sName);
          break;
        case horizon.strat.stratStruct._FORMATION:
          if (sFormation.length() == 0) stRecord.sFormation = new String(sName);
          break;
      }
    }

    return (stRecord);
  }

  /** Method getICSData()
   * <p> This method will set the ICS data.
   */

  public void getICSData()
  {
    String sID = "";

    if (pICS != null)
    {
      sID = new String( pICS.getData() );

      if (stICS != null)
      {
        for (int i=0; i<stICS.iCount; i++)
        {
          if (sID.equals(stICS.stItem[i].sKEY))
          {
            // Force Subseries to Series JRV 10/28/2014
            if (stICS.stItem[i].iRank == horizon.strat.stratStruct._SUBSERIES)
            {
			  stICS.stItem[i].iRank     = horizon.strat.stratStruct._SERIES;
              stICS.stItem[i].series    = new String(stICS.stItem[i].subSeries);
              stICS.stItem[i].subSeries = new String("");
			}

            this.sEon       = new String(stICS.stItem[i].sEon);
            this.sEra       = new String(stICS.stItem[i].sEra);
            this.system     = new String(stICS.stItem[i].system);
            this.series     = new String(stICS.stItem[i].series);
            this.subSystem  = new String(stICS.stItem[i].subSystem);
            this.subSeries  = new String(stICS.stItem[i].subSeries);

            txtEon.setText(stICS.stItem[i].sEon);
            txtEra.setText(stICS.stItem[i].sEra);
            txtSystem.setText(stICS.stItem[i].system);
            txtSubsystem.setText(stICS.stItem[i].subSystem);

            if (stICS.stItem[i].iRank == horizon.strat.stratStruct._SERIES)
              txtSeries.setText(stICS.stItem[i].series);
            else if (stICS.stItem[i].iRank == horizon.strat.stratStruct._SUBSERIES)
              txtSeries.setText(stICS.stItem[i].subSeries);

            if (sName.length() == 0)
            {
              if (this.sEon.length() > 0)      sName = new String(this.sEon);
              if (this.sEra.length() > 0)      sName = new String(this.sEra);
              if (this.system.length() > 0)    sName = new String(this.system);
              if (this.series.length() > 0)    sName = new String(this.series);
              if (this.subSystem.length() > 0) sName = new String(this.subSystem);
              if (this.subSeries.length() > 0) sName = new String(this.subSeries);

              txtName.setText(sName);
            }

            if (sName2.length() == 0)
            {
              sName2 = new String(sName);
              txtName2.setText(sName2);
            }

            if (iRank == horizon.strat.stratStruct._NONE)
              iRank = stICS.stItem[i].iRank;
          }
        }
      }

      setRank();   // Set the formation Rank

      pICS.close();
      pICS = null;
    }
  }

  /** Method getSelection()
   * <p> This method will retrieve and load the State Stratigraphic data
   *     structure
   */

  public void getSelection()
  {
    if (pState != null)
    {
      stratStruct st = pState.getData();
      this.stRecord  = horizon.strat.stratUtility.copy( st );

      if (st != null)
      {
        if ( iAction == _ADD )
        {
          this.sKID       = new String(st.sKID); // KGS Primary KEY of Record
          this.sKEY       = new String(st.sKEY); // unique KEY of Record
          this.sid        = new String(st.sid);  // unique order id
//          this.id         = st.id;               // unique order id
        }
        else
        {
          this.stRecord.sKID = new String(this.sKID);
          this.stRecord.sKEY = new String(this.sKEY);
          this.stRecord.sid  = new String(this.sid);
//          this.stRecord.id   = this.id;
        }

        this.iLevel     = st.iLevel;
        this.iRank      = st.iRank;

        this.sName      = new String(st.sName);
        this.sName2     = new String(st.sName2);

        this.sEon       = new String(st.sEon);
        this.sEra       = new String(st.sEra);

        this.system     = new String(st.system);
        this.series     = new String(st.series);
        this.subSystem  = new String(st.subSystem);
        this.subSeries  = new String(st.subSeries);

        this.stage      = new String(st.stage);
        this.sprGroup   = new String(st.sprGroup);
        this.sGroup     = new String(st.sGroup);
        this.subGroup   = new String(st.subGroup);
        this.sFormation = new String(st.sFormation);

        this.dAgeStart  = dAgeStart;
        this.dStartCorr = dStartCorr;

        this.dAgeEnd    = dAgeEnd;
        this.dEndCorr   = dEndCorr;

        this.depthStart = depthStart;
        this.depthEnd   = depthEnd;

        if (txtTop  != null) txtTop.setText( ""  + depthStart );
        if (txtBase != null) txtBase.setText( "" + depthEnd );

        // Strigraphic Unit Names

        txtName.setText(st.sName);           // Stratigraphic Name
        txtName2.setText(st.sName2);         // Alternate Name

        txtEon.setText(st.sEon);             // Eon
        txtEra.setText(st.sEra);             // Era

        txtSystem.setText(st.system);        // System
        txtSubsystem.setText(st.subSystem);  // Subsystem
        if (st.series.length() > 0)
          txtSeries.setText(st.series);      // Series
        else
          txtSeries.setText(st.subSeries);   // subseries

        txtStage.setText(st.stage);          // Stage
        txtSprGroup.setText(st.sprGroup);    // Super Group
        txtGroup.setText(st.sGroup);         // Group
        txtSubgroup.setText(st.subGroup);    // Subgroup
        txtFormation.setText(st.sFormation); // Formation

        if (sName.length() == 0)
        {
          if (this.system.length() > 0)     sName = new String(this.system);
          if (this.series.length() > 0)     sName = new String(this.series);
          if (this.subSystem.length() > 0)  sName = new String(this.subSystem);
          if (this.subSeries.length() > 0)  sName = new String(this.subSeries);
          if (this.stage.length() > 0)      sName = new String(this.stage);
          if (this.sprGroup.length() > 0)   sName = new String(this.sprGroup);
          if (this.sGroup.length() > 0)     sName = new String(this.sGroup);
          if (this.subGroup.length() > 0)   sName = new String(this.subGroup);
          if (this.sFormation.length() > 0) sName = new String(this.sFormation);

          txtName.setText(sName);
        }

        if (sName2.length() == 0)
        {
          sName2 = new String(sName);
          txtName2.setText(sName2);
        }

        // Begining Age

        txtAgeBegin.setText(""        + st.dAgeStart);
        txtCorrectionBegin.setText("" + st.dStartCorr);

        // Ending Age

        txtAgeEnd.setText(""        + st.dAgeEnd);
        txtCorrectionEnd.setText("" + st.dEndCorr);

        setLevel();  // Set the Formation Level
        setRank();   // Set the formation Rank

        pState.setVisible(false);
      }
    }
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setDepthTextfields()
   *  <p> This method will set the Depth Textfield
   * @param txtTop  = the depth top textfield
   * @param txtBase = the depth base textfield
   */

  public void setDepthTextfields(JTextField txtTop, JTextField txtBase)
  {
    this.txtTop  = txtTop;
    this.txtBase = txtBase;
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.depthStart = depthStart;
    this.depthEnd   = depthEnd;

    if (txtTop  != null) txtTop.setText( ""  + depthStart );
    if (txtBase != null) txtBase.setText( "" + depthEnd );
  }

  /** Method setData()
   * <p> This method will load the Well/Outcrop header data structure
   *     into this panel
   * @param stInfo = the Well/Outcrop header data structure
   */

  public void setData(iqstratHeadersStruct stInfo) { this.stInfo = stInfo; }

  /** Method setListData()
   * <p> This method will load the Stratigraphic Unit data list structure
   *     into this panel
   * @param stList = the Stratigraphic Unit data list structure
   */

  public void setListData(stratListStruct stList)
  {
    this.stList = stList;
    pTable.repopulateList(stList);
    setButtons();
  }

  /** Method setData()
   * <p> This method will set the array of Stratigraphic Unit selected
   * @param st = Stratigraphic Unit data Structure
   */

  public void setData( stratStruct st )
  {
    this.stRecord = horizon.strat.stratUtility.copy( st );

    if (st != null)
    {
      this.sKID       = new String(st.sKID); // KGS Primary KEY of Record
      this.sKEY       = new String(st.sKEY); // unique KEY of Record
      this.sid        = new String(st.sid);  // unique order id
//      this.id         = st.id;               // unique order id

      this.iLevel     = st.iLevel;
      this.iRank      = st.iRank;

      this.sName      = new String(st.sName);
      this.sName2     = new String(st.sName2);

      this.sEon       = new String(st.sEon);
      this.sEra       = new String(st.sEra);

      this.system     = new String(st.system);
      this.series     = new String(st.series);
      this.subSystem  = new String(st.subSystem);
      this.subSeries  = new String(st.subSeries);

      this.stage      = new String(st.stage);
      this.sprGroup   = new String(st.sprGroup);
      this.sGroup     = new String(st.sGroup);
      this.subGroup   = new String(st.subGroup);
      this.sFormation = new String(st.sFormation);

      this.dAgeStart  = st.dAgeStart;
      this.dStartCorr = st.dStartCorr;

      this.dAgeEnd    = st.dAgeEnd;
      this.dEndCorr   = st.dEndCorr;

      this.depthStart = st.depthStart;
      this.depthEnd   = st.depthEnd;

      if (txtTop  != null) txtTop.setText( ""  + depthStart );
      if (txtBase != null) txtBase.setText( "" + depthEnd );

      // Strigraphic Unit Names

      txtName.setText(st.sName);           // Stratigraphic Name
      txtName2.setText(st.sName2);         // Alternate Name

      txtEon.setText(st.sEon);             // Eon
      txtEra.setText(st.sEra);             // Era

      txtSystem.setText(st.system);        // System
      txtSubsystem.setText(st.subSystem);  // Subsystem
      if (st.series.length() > 0)
        txtSeries.setText(st.series);      // Series
      else
        txtSeries.setText(st.subSeries);   // subseries

      txtStage.setText(st.stage);          // Stage
      txtSprGroup.setText(st.sprGroup);    // Super Group
      txtGroup.setText(st.sGroup);         // Group
      txtSubgroup.setText(st.subGroup);    // Subgroup
      txtFormation.setText(st.sFormation); // Formation

      // Begining Age

      txtAgeBegin.setText(""+st.dAgeStart);
      txtCorrectionBegin.setText(""+st.dStartCorr);

      // Ending Age

      txtAgeEnd.setText(""+st.dAgeEnd);
      txtCorrectionEnd.setText(""+st.dEndCorr);

      setLevel();  // Set the Formation Level
      setRank();   // Set the formation Rank
    }

    iAction = _MODIFY;
    btnAdd.setText("Modify");
  }

  /** Method setLevel()
   * <p> This method will set the level of confidence of the formation
   */

  private void setLevel()
  {
    rbGold.setSelected(false);    // Excellent
    rbSilver.setSelected(false);  // Good
    rbCopper.setSelected(false);  // Fair
    rbLead.setSelected(false);    // Poor

    switch (iLevel)
    {
      case horizon.strat.stratStruct._EXCELLENT:  // Gold   - Excellent
        rbGold.setSelected(true);    // Excellent
        break;
      case horizon.strat.stratStruct._GOOD:       // Silver - Good
        rbSilver.setSelected(true);  // Good
        break;
      case horizon.strat.stratStruct._FAIR:       // Copper - Fair
        rbCopper.setSelected(true);  // Fair
        break;
      case horizon.strat.stratStruct._POOR:       // Lead   - Poor
        rbLead.setSelected(true);    // Poor
        break;
    }
  }

  /** Method setRank()
   * <p> This method will set the rank of the formation
   */

  private void setRank()
  {
    rbUnknown.setSelected(false);    // Unknown
    rbEon.setSelected(false);        // Eon
    rbEra.setSelected(false);        // Era
    rbSystem.setSelected(false);     // System
    rbSubsystem.setSelected(false);  // Subsystem
    rbSeries.setSelected(false);     // Series
    rbSubseries.setSelected(false);  // Subseries
    rbStage.setSelected(false);      // Stage
    rbSuperGroup.setSelected(false); // Super group
    rbGroup.setSelected(false);      // Group
    rbSubgroup.setSelected(false);   // Subgroup
    rbFormation.setSelected(false);  // Formation
    rbMember.setSelected(false);     // Member
    rbBed.setSelected(false);        // Bed

    switch (iRank)
    {
      case horizon.strat.stratStruct._NONE:
        rbUnknown.setSelected(true);    // Unknown
        break;
      case horizon.strat.stratStruct._EON:
        rbEon.setSelected(true);        // Eon
        break;
      case horizon.strat.stratStruct._ERA:
        rbEra.setSelected(true);        // Era
        break;
      case horizon.strat.stratStruct._SYSTEM:
        rbSystem.setSelected(true);     // System
        break;
      case horizon.strat.stratStruct._SERIES:
        rbSeries.setSelected(true);     // Series
        break;
      case horizon.strat.stratStruct._SUBSYSTEM:
        rbSubsystem.setSelected(true);  // Subsystem
        break;
      case horizon.strat.stratStruct._SUBSERIES:
        rbSubseries.setSelected(true);  // Subseries
        break;
      case horizon.strat.stratStruct._STAGE:
        rbStage.setSelected(true);      // Stage
        break;
      case horizon.strat.stratStruct._SUPER_GROUP:
        rbSuperGroup.setSelected(true); // Super group
        break;
      case horizon.strat.stratStruct._GROUP:
        rbGroup.setSelected(true);      // Group
        break;
      case horizon.strat.stratStruct._SUBGROUP:
        rbSubgroup.setSelected(true);   // Subgroup
        break;
      case horizon.strat.stratStruct._FORMATION:
        rbFormation.setSelected(true);  // Formation
        break;
      case horizon.strat.stratStruct._MEMBER:
        rbMember.setSelected(true);     // Member
        break;
      case horizon.strat.stratStruct._BED:
        rbBed.setSelected(true);        // Bed
        break;
    }
  }

  /** Method setChart(int iChart)
   * <p> This method will set the user defined chart to display
   * @param iChart = the type of chart to display
   */

  public void setChart(int iChart)
  {
    switch (iChart)
    {
      case _NONE:
        iChart = this.iChart;
        break;

      case _DEFAULT:
        if (stStrat != null)
          stStrat.delete();
        stStrat = null;
        break;

      case _USER_DEFINED:
/*
        if (pChange != null)
        {
          int iSource = pChange.getSource();
          stratProjectStruct st = pChange.getData();

          switch (iSource)
          {
            case horizon.strat.io.stratIOProjects.USER_PC:
              stStrat = horizon.strat.io.stratIOChart.read(
                          iSource, st.sKEY, st.sUserKID );
              break;

            case horizon.strat.io.stratIOProjects.KGS_DB:
              stStrat = horizon.strat.io.stratIOChart.read(
                          iSource, st.sKID, st.sUserKID );
              break;
          }

          pChange.close();
          pChange = null;
        }
*/
        break;
    }

//    if (pChange != null)
//      pChange.close();
//    pChange = null;

    this.iChart = iChart;

    setButtons();
    displayChart();
  }

  /** Method setButtons()
   * <p> This method will set the buttons depending on the data
   */

  private void setButtons()
  {
    int iContinue = 0;

    btnModify.setEnabled(false);       // Modify Record
    btnRemove.setEnabled(false);       // Delete Record
    btnRemoveAll.setEnabled(false);    // Delete All Records
    btnSaveAs.setEnabled(false);       // Save for general Acess
    btnTemplate.setEnabled(false);     // Save as Template

    if (pTable.getTotalRows() > 0)
      iContinue = 1;

    switch (iAction)
    {
      case _ADD:
        iContinue = 1;
        break;
      case _MODIFY:
        iContinue = 0;
        break;
    }

    if (iContinue == 1)
    {
      btnModify.setEnabled(true);       // Modify Record
      btnRemove.setEnabled(true);       // Delete Record
      btnRemoveAll.setEnabled(true);    // Delete All Records
      btnSaveAs.setEnabled(true);       // Save for general Acess
      btnTemplate.setEnabled(true);     // Save as Template
    }

    switch (iChart)
    {
      case _DEFAULT:
        btnState.setText("1968 Kansas Chart");
        break;

      case _USER_DEFINED:
        btnState.setText("User Defined Chart");
        break;
    }
  }

  /* ===================================================================== *
   * ------------------ STRATIGRAPHIC CHART ACTIONS ---------------------- *
   * ===================================================================== */

  /** Method clearReplace()
   * <p> This method will clear the Replace action
   *

  public void clearReplace() { if (pChart != null) { pChart.clearReplace(); } }
*/
  /** Method setProject()
   * <p> This method will set the Project Data
   *

  public void setProject()
  {
    if (pChange != null)
    {
      setChart( pChange.getChart() );
    }

    if (pChart != null)
    {
      pChart.setProject();
    }
  }
*/
  /* ===================================================================== *
   * ---------------- WELL STRATIGRAPHIC UNITS ACTIONS ------------------- *
   * ===================================================================== */

  /** Method clearWellReplace()
   * <p> This method will clear the Replace action
   *

  public void clearWellReplace()
  {
    if (pSaveUnits != null) pSaveUnits.clearReplace();
  }
*/
  /** Method setWellUnits()
   * <p> This method will set the Project Data
   *

  public void setWellUnits()
  {
    if (pSaveUnits != null) pSaveUnits.setProject();
  }
*/
  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add or modify the stratigraphic data in the list
   */

  private void add()
  {
    stratStruct st = getData();

    switch (iAction)
    {
      case _ADD:
        stList = horizon.strat.stratUtility.add( st, stList );
        break;

      case _MODIFY:
        stList = horizon.strat.stratUtility.modify(st.sKID, st.sKEY, st, stList);
        break;
    }

    iAction = _ADD;
    btnAdd.setText("Add");

    clear();

    pTable.repopulateList(stList);
    setButtons();

    if (notifier != null)
      notifier.notifyObservers(new String("Stratigraphic Units modified"));
  }

  /** Method modify()
   * <p> This method will allow the user to modify the stratigraphic
   *     data in the list
   */

  private void modify() { setData( pTable.getRowData() ); }

  /** Method removeData()
   * <p> This method will remove the stratigraphic data from the list
   */

  private void removeData()
  {
    stratStruct st = pTable.getRowData();

    stList = horizon.strat.stratUtility.remove(st.sKID, st.sKEY, stList);

    pTable.repopulateList(stList);
    setButtons();

    if (notifier != null)
      notifier.notifyObservers(new String("Stratigraphic Units modified"));
  }

  /** Method remove()
   * <p> This method will remove all the stratigraphic data from the list
   */

  private void removeAllData()
  {
    if (stList != null)
      stList.delete();
    stList = null;

    pTable.repopulateList(stList);
    setButtons();

    if (notifier != null)
      notifier.notifyObservers(new String("Stratigraphic Units modified"));
  }

  /** Method displayICS()
   * <p> This method will show the International Stratigraphic Chart
   */

  private void displayICS()
  {
    closeICS();
//    pICS = new stratPlotICSFrame( notifier, 0, 0, 0, 0, stICS );
    pICS = new stratPlotICSFrame( obsrvble, 0, 0, 0, 0, stICS );
  }

  /** Method displayChart()
   * <p> This method will display either the Kansas or another states
   *     Stratigraphic Chart
   */

  private void displayChart()
  {
    if (pState != null)
      pState.close();
    pState = null;

    if (iChart == _DEFAULT)
    {
//      pState = new stratPlotStateFrame( notifier, stICS, stKGS );
      pState = new stratPlotStateFrame( obsrvble, stICS, stKGS );
    }
    else
    {
      if (stStrat != null)
//        pState = new stratPlotStateFrame( notifier, stICS, stStrat );
        pState = new stratPlotStateFrame( obsrvble, stICS, stStrat );
      else
      {
        JOptionPane.showMessageDialog((Component) null,
                                       "No Chart Available",
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /** Method changeChart()
   * <p> This method will change the Stratigraphic unit Chart.
   *

  private void changeChart()
  {
    if (pChange != null)
      pChange.close();
    pChange = null; // Change Stratigraphic Chart

    pChange = new stratTemplateFrame( notifier, stStruct, _USER_DEFINED );
  }
*/
  /** Method saveChart()
   * <p> This method will save the Stratigrpahic Unit Chart as a Template
   *

  private void saveChart()
  {
    if (pChart != null)
      pChart.close();
    pChart = null; // Save Stratigrahic Chart Frame

    pChart = new stratTemplateSaveFrame( notifier, stStruct, stList );
  }
*/
  /** Method saveAsGeneral()
   * <p> This method will save the Stratigraphic Units to be retrieve by other
   *     programs.
   *

  private void saveAsGeneral()
  {
    if (pSaveUnits != null)
      pSaveUnits.close();
    pSaveUnits = null;

    pSaveUnits = new stratUnitsSaveFrame( notifier, stStruct, stInfo, stList );
  }
*/
  /** Method clear()
   * <p> This method will clear the settings on the panel
   */

  public void clear()
  {
    if (stRecord != null)
      stRecord.delete();
    stRecord = null;

    sKID       = new String("0"); // KGS Primary KEY of Record
    sKEY       = new String("0"); // unique KEY of Record
    sid        = new String("0"); // unique order id
//    id         = 0;               // unique order id

    iLevel     = horizon.strat.stratStruct._POOR; // Confidence
    iRank      = horizon.strat.stratStruct._NONE; // Rank

    sName      = new String("");
    sName2     = new String("");

    sEon       = new String("");
    sEra       = new String("");
    system     = new String("");
    series     = new String("");
    subSystem  = new String("");
    subSeries  = new String("");

    stage      = new String("");
    sprGroup   = new String("");
    sGroup     = new String("");
    subGroup   = new String("");
    sFormation = new String("");

    dAgeStart  = 0.0;
    dStartCorr = 0.0;

    dAgeEnd    = 0.0;
    dEndCorr   = 0.0;

    // Strigraphic Unit Names

    txtEon.setText("");        // Eon
    txtEra.setText("");        // Era
    txtSystem.setText("");     // System
    txtSubsystem.setText("");  // Subsystem
    txtSeries.setText("");     // series
    txtStage.setText("");      // Stage
    txtSprGroup.setText("");   // Super Group
    txtGroup.setText("");      // Group
    txtSubgroup.setText("");   // Subgroup
    txtFormation.setText("");  // Formation

    txtName.setText("");       // Stratigraphic Name
    txtName2.setText("");      // Alternate Name

    // Begining Age

    txtAgeBegin.setText("");
    txtCorrectionBegin.setText("");

    // Ending Age

    txtAgeEnd.setText("");
    txtCorrectionEnd.setText("");

    setLevel();  // Set the Formation Level
    setRank();   // Set the formation Rank

    iAction = _ADD;
    btnAdd.setText("Add");
  }

  /** Method closeICS()
   * <p> This method will close the ICS Dialog)
   */

  public void closeICS()
  {
    if (pICS != null)
      pICS.close();
    pICS = null;
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier   = null;
    stStruct   = null;
    stStrat    = null;
    stKGS      = null; // State Stratigraphic Units
    stICS      = null; // ICS Stratigraphic Units
    stList     = null;
    stCMN      = null;

    stInfo     = null;

    sKID       = null;  // KGS Primary KEY of Record
    sKEY       = null;  // unique KEY of Record
    sid        = null;  // unique order number

    sEon       = null;  // Eon Stratigraphic Unit Names
    sEra       = null;  // Era Stratigraphic Unit Names
    system     = null;  // System Stratigraphic Unit Names
    series     = null;  // Series Stratigraphic Unit Names
    subSystem  = null;  // Substage Stratigraphic Unit Names
    subSeries  = null;  // Subseries Stratigraphic Unit Names

    stage      = null;
    sprGroup   = null;
    sGroup     = null;
    subGroup   = null;
    sFormation = null;
    sName      = null;
    sName2     = null;

    if (pTable != null)
      pTable.close();
    pTable   = null;

    closeICS();

    if (pState != null)
      pState.close();
    pState   = null;

//    if (pChart != null)
//      pChart.close();
//    pChart = null; // Save Stratigrahic Chart Frame

//    if (pChange != null)
//      pChange.close();
//    pChange = null; // Change Stratigraphic Chart

//    if (pSaveUnits != null)
//      pSaveUnits.close();
//    pSaveUnits = null;

    // Global Panel Widgets

    // Stratigraphic Unit List Buttons

    btnAdd       = null;       // Add Record

    btnModify    = null;       // Modify Record
    btnRemove    = null;       // Delete Record
    btnRemoveAll = null;       // Delete All Records
    btnSaveAs    = null;       // Save for general Acess
    btnTemplate  = null;       // Save as Template

    btnICS       = null;       // ICS Strat Chart
    btnSelect    = null;       // Select State Chart
    btnState     = null;       // State Strat Chart

    // Level of Confidence

    rbGold       = null;  // Excellent
    rbSilver     = null;  // Good
    rbCopper     = null;  // Fair
    rbLead       = null;  // Poor

    // Rank of Stratigraphic Unit

    rbUnknown    = null;  // Unknown
    rbEon        = null;  // Eon
    rbEra        = null;  // Era
    rbSystem     = null;  // System
    rbSubsystem  = null;  // Subsystem
    rbSeries     = null;  // Series
    rbSubseries  = null;  // Subseries
    rbStage      = null;  // Stage
    rbSuperGroup = null;  // Super group
    rbGroup      = null;  // Group
    rbSubgroup   = null;  // Subgroup
    rbFormation  = null;  // Formation
    rbMember     = null;  // Member
    rbBed        = null;  // Bed

    // Strigraphic Unit Names

    txtEon       = null;    // Eon
    txtEra       = null;    // Era
    txtSystem    = null;    // System
    txtSubsystem = null;    // Subsystem
    txtSeries    = null;    // Series
    txtStage     = null;    // Stage
    txtSprGroup  = null;    // Super Group
    txtGroup     = null;    // Group
    txtSubgroup  = null;    // Subgroup
    txtFormation = null;    // Formation

    txtName      = null;    // Stratigraphic Name
    txtName2     = null;    // Alternate Name

    // Begining Age

    txtAgeBegin        = null;
    txtCorrectionBegin = null;

    // Ending Age

    txtAgeEnd          = null;
    txtCorrectionEnd   = null;
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // Stratigraphic Unit List Buttons

    if (event.getSource() == btnAdd)   { add(); }         // Add Record
    if (event.getSource() == btnClear) { clear(); }       // Clear Record

    if (event.getSource() == btnModify)    { modify(); }     // Modify Record
    if (event.getSource() == btnRemove)    { removeData(); } // Delete Record
    if (event.getSource() == btnRemoveAll) { removeAllData(); } // Delete All
//    if (event.getSource() == btnSaveAs)    { saveAsGeneral(); } // Save General
//    if (event.getSource() == btnTemplate)  { saveChart(); }  // Save as Template

    if (event.getSource() == btnICS)    { displayICS(); }   // ICS Strat Chart
//    if (event.getSource() == btnSelect) { changeChart(); }  // Select Chart
    if (event.getSource() == btnState)  { displayChart(); } // State Strat Chart

    // Level of Confidence

    if (event.getSource() == rbGold)
    {
      iLevel = horizon.strat.stratStruct._EXCELLENT;
    }
    if (event.getSource() == rbSilver)
    {
      iLevel = horizon.strat.stratStruct._GOOD;
    }
    if (event.getSource() == rbCopper)
    {
      iLevel = horizon.strat.stratStruct._FAIR;
    }
    if (event.getSource() == rbLead)
    {
      iLevel = horizon.strat.stratStruct._POOR;
    }

    // Rank of Stratigraphic Unit

    if (event.getSource() == rbUnknown)
    {
      iRank = horizon.strat.stratStruct._NONE;
    }
    if (event.getSource() == rbEon)
    {
      iRank = horizon.strat.stratStruct._EON;
    }
    if (event.getSource() == rbEra)
    {
      iRank = horizon.strat.stratStruct._ERA;
    }
    if (event.getSource() == rbSystem)
    {
      iRank = horizon.strat.stratStruct._SYSTEM;
    }
    if (event.getSource() == rbSubsystem)
    {
      iRank = horizon.strat.stratStruct._SUBSYSTEM;
    }
    if (event.getSource() == rbSeries)
    {
      iRank = horizon.strat.stratStruct._SERIES;
    }
    if (event.getSource() == rbSubseries)
    {
      iRank = horizon.strat.stratStruct._SUBSERIES;
    }
    if (event.getSource() == rbStage)
    {
      iRank = horizon.strat.stratStruct._STAGE;
    }
    if (event.getSource() == rbSuperGroup)
    {
      iRank = horizon.strat.stratStruct._SUPER_GROUP;
    }
    if (event.getSource() == rbGroup)
    {
      iRank = horizon.strat.stratStruct._GROUP;
    }
    if (event.getSource() == rbSubgroup)
    {
      iRank = horizon.strat.stratStruct._SUBGROUP;
    }
    if (event.getSource() == rbFormation)
    {
      iRank = horizon.strat.stratStruct._FORMATION;
    }
    if (event.getSource() == rbMember)
    {
      iRank = horizon.strat.stratStruct._MEMBER;
    }
    if (event.getSource() == rbBed)
    {
      iRank = horizon.strat.stratStruct._BED;
    }

    // Enable or disable buttons depending on the data

    setButtons();
    if (notifier != null) notifier.notifyObservers(new String("MEMORY CHANGE"));
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtName)  { sName  = new String(txtName.getText());  }
    if (e.getSource() == txtName2) { sName2 = new String(txtName2.getText()); }

    if (e.getSource() == txtStage) { stage  = new String(txtStage.getText()); }
    if (e.getSource() == txtSprGroup) { sprGroup = new String(txtSprGroup.getText()); }
    if (e.getSource() == txtGroup) { sGroup = new String(txtGroup.getText()); }
    if (e.getSource() == txtSubgroup)
    {
      subGroup = new String(txtSubgroup.getText());
    }
    if (e.getSource() == txtFormation)
    {
      sFormation = new String(txtFormation.getText());
    }

    if (e.getSource() == txtAgeBegin)
    {
      iNumeric = 1;
      sTemp = txtAgeBegin.getText();
      sMessage = new String("Begin Age Value is a Numeric Field");
    }

    if (e.getSource() == txtCorrectionBegin)
    {
      iNumeric = 1;
      sTemp = txtCorrectionBegin.getText();
      sMessage = new String("Begin Correction Value is a Numeric Field");
    }

    if (e.getSource() == txtAgeEnd)
    {
      iNumeric = 1;
      sTemp = txtAgeEnd.getText();
      sMessage = new String("End Age Value is a Numeric Field");
    }

    if (e.getSource() == txtCorrectionEnd)
    {
      iNumeric = 1;
      sTemp = txtCorrectionEnd.getText();
      sMessage = new String("End Correction Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtAgeBegin)
          dAgeStart = cmn.cmnString.stringToDouble(txtAgeBegin.getText());

        if (e.getSource() == txtCorrectionBegin)
          dStartCorr = cmn.cmnString.stringToDouble(txtCorrectionBegin.getText());

        if (e.getSource() == txtAgeEnd)
          dAgeEnd = cmn.cmnString.stringToDouble(txtAgeEnd.getText());

        if (e.getSource() == txtCorrectionEnd)
          dEndCorr = cmn.cmnString.stringToDouble(txtCorrectionEnd.getText());
      }
    }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("ICS Stratigraphic Unit Selected"))   { getICSData(); }
    if (sArg.equals("Close ICS Dialog"))                  { closeICS(); }
    if (sArg.equals("State Stratigraphic Unit Selected")) { getSelection(); }

    if (notifier != null) notifier.notifyObservers(new String("MEMORY CHANGE"));
  }
}

/** CLASS stratDataEntryPanelFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class stratDataEntryPanelFocusAdapter extends java.awt.event.FocusAdapter
{
  stratDataEntryPanel adaptee;

  stratDataEntryPanelFocusAdapter(stratDataEntryPanel adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/** CLASS stratDataEntryPanelObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class stratDataEntryPanelObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 08/20/2008
 *  @author  John R. Victorine
 */
