/*
 * @seqUtility.java Version 1.1 09/28/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.seq;

import horizon.seq.seqListStruct;
import horizon.seq.seqStruct;

import iqstrat.iqstratHeadersStruct;

/** Class seqUtility
 *  <p> This Class will provide basic utilities for the Regions
 *      data structures.
 *
 *  @version 1.1 09/28/2007
 *  @author  John R. Victorine
 */

public class seqUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the sequence List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Sequence List Data Structure
   * @return st       = The Sequence List Data Structure.
   */

  public static seqListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                seqListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one sequence stratigraphic unit list
   *     structure to another
   * @param  stOld = Old sequence stratigraphic unit list data structure
   * @return stNew = New sequence stratigraphic unit list data structure
   */

  public static seqListStruct copyList( seqListStruct stOld )
  {
    seqListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new seqListStruct();

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID);   // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
        stNew.sName  = new String(stOld.sName);  // Name
        stNew.status = new String(stOld.status); // Status

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude;  // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL;   // Ground Level

        // LAS 3 Dependent Variable

        stNew.sKGS     = new String( stOld.sKGS ); // KGS Saved Data Indicator
        stNew.source   = new String( stOld.source );    // Source of Data
        stNew.sCreated = new String( stOld.sCreated );  // Created Date

        // Data List

        stNew.iCount = stOld.iCount;
        stNew.stItem = new seqStruct[stOld.iCount];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static seqListStruct transfer(seqListStruct stOld)
  {
    seqListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method add()
   * <p> This method will add the Sequences to an existing list
   * @param  stNew = The Sequence Data Structure
   * @param  st    = The Old Sequences List Data Structure
   * @return st    = The new Sequences List Data Structure.
   */

  public static seqListStruct add( seqStruct stNew, seqListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    seqListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new seqListStruct();
    stTemp.stItem = new seqStruct[iRecords];

    if (st != null)
    {
	  iSource  = st.iSource;

      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new seqStruct();
    stTemp.stItem[iCount] = copy(stNew);
    iCount++;

    stTemp.iCount = iCount;

    st        = new seqListStruct();
    st.stItem = new seqStruct[iCount];
    st.iCount = iCount;

    st.iSource  = iSource;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Sequence in an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  stNew  = Sequence data structure
   * @param  st     = Sequence data list structure
   * @return st     = New Sequence data list structure
   */

  public static seqListStruct modify( String sKEY,
                                      seqStruct stNew,
                                      seqListStruct st )
  {
    int i=0;
    int    iSource  = -1;
    seqListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
	  iSource  = st.iSource;

      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp        = new seqListStruct();
      stTemp.stItem = new seqStruct[st.iCount];
      stTemp.iCount = st.iCount;

      for (i=0; i<st.iCount; i++)
      {
        if (sKEY.equals(st.stItem[i].sKEY))
          stTemp.stItem[i] = copy(stNew);
        else
          stTemp.stItem[i] = copy(st.stItem[i]);
      }

      st.delete();

      st        = new seqListStruct();
      st.stItem = new seqStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.iSource  = iSource;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Sequence from an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  st     = Sequence data list structure
   * @return st     = New Sequence data list structure
   */

  public static seqListStruct remove( String sKEY, seqListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    int    iSource  = -1;
    seqListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      if (st.iCount == 1)
      {
        st.delete();
        st = null;
      }
      else
      {
        iRecords = st.iCount-1;

        stTemp        = new seqListStruct();
        stTemp.stItem = new seqStruct[iRecords];

        iSource  = st.iSource;

        sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
        source   = new String( st.source );    // Source of Data
        sCreated = new String( st.sCreated );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          if ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0")))
          {
            iFound = i;
          }
        }

        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();

        stTemp.iCount = iCount;

        st        = new seqListStruct();
        st.stItem = new seqStruct[iCount];
        st.iCount = iCount;

        st.iSource  = iSource;

        st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
        st.source   = new String( source );    // Source of Data
        st.sCreated = new String( sCreated );  // Created Date

        for (i=0; i<stTemp.iCount; i++)
        {
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        stTemp.delete();
      }
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one region structure to another
   * @param  stOld = Old Sequence data structure
   * @return stNew = New Sequence data structure
   */

  public static seqStruct copy( seqStruct stOld )
  {
    seqStruct stNew = new seqStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY); // User defined KEY of Record

      stNew.depthStart = stOld.depthStart;       // Starting Depth.
      stNew.depthEnd   = stOld.depthEnd;         // Ending Depth.

      stNew.sequence   = new String(stOld.sequence);  // Sequence Description
      stNew.sMnemonic  = new String(stOld.sMnemonic); // Stratigraphic Mnemonic

      stNew.abbrev     = new String(stOld.abbrev);   // Abbreviation
    }

    return (stNew);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = Sequence data list structure
   * @return st = sorted Sequence data list structure
   */

  public static seqListStruct bubbleSort( seqListStruct st )
  {
    boolean   swappedOnPrevRun = true;
    seqStruct stTemp           = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print( seqListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(i +" "+
                           st.stItem[i].sKEY       +" "+
                           st.stItem[i].depthStart +" "+
                           st.stItem[i].depthEnd   +" "+
                           st.stItem[i].sMnemonic  +" "+
                           st.stItem[i].sequence   +" "+
                           st.stItem[i].abbrev);
      }
      System.out.println(" ------------------------------------------------- ");
    }
  }
}