/*
 * @seqPlotTrack.java Version 1.1 06/05/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.seq.plot;

import java.awt.*;
import java.awt.geom.*;

import horizon.seq.seqListStruct;

/** CLASS seqPlotTrack
 *  <p> This Class will create the sequence stratigraphic Track.
 *
 *  @version 1.1 06/05/2008
 *  @author  John R. Victorine
 */

public class seqPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private seqListStruct stSeq = null; // Sequence Stratigraphy Data List Struct

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR seqPlotTrack( int iDataType,
   *                            double depthStart,
   *                            double depthEnd,
   *                            int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public seqPlotTrack( int iDataType,
                       double depthStart,
                       double depthEnd,
                       int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close() { stSeq = null; }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getSequenceData()
   * <p> This method will get an existing Rock Color List Data Structure
   * @return stSeq = Rock Color List Data Structure
   */

  public seqListStruct getSequenceData() { return (stSeq); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight(int iScale, double dMaximum, double dMinimum)
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setSequenceData( seqListStruct stSeq )
   * <p> This method will set an existing Rock Color List Data Structure
   * @param stSeq = Rock Color List Data Structure
   */

  public void setSequenceData( seqListStruct stSeq ) { this.stSeq = stSeq; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/6;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawSequenceTrack( Graphics g, int iStartTrack, int iWidthTrack )
   *   <p> This track will plot the rock color track data
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawSequenceTrack( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iDepth1     = 0;
    int    iDepth2     = 0;
    int    iDepth      = 0;
    double dRatio      = 0.0;
    double depth1      = 0.0;
    double depth2      = 0.0;
    int    iStart      = iStartTrack;
    int    iEnd        = iStartTrack+iWidthTrack;
    int    iWide       = iWidthTrack;
    int    iIncrement  = iWidthTrack/5;
    int    iColor[]    = { 0, 0, 0 };
    int    iFound      = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g.setColor(Color.black);

    if (stSeq != null)
    {
      for (int i=0; i<stSeq.iCount; i++)
      {
        if (stSeq.stItem[i] != null)
        {
          depth1 = stSeq.stItem[i].depthStart;
          depth2 = stSeq.stItem[i].depthEnd;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio = (double) iLogHeight * (depthStart - depth2) /
                                         (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs(iDepth2 - iDepth1);

          if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 < iHeight))
          {
            iFound = -1;
            for (int k=0; k<horizon.seq.seqTypesStruct._TOTAL; k++)
            {
              if (stSeq.stItem[i].sMnemonic.equals(
                    horizon.seq.seqTypesStruct.NAME[k][2]))
              {
                iFound = k;
              }
            }

            if (iFound > -1)
            {
              if (iFound < horizon.seq.seqTypesStruct._SB)
              {
                iColor = horizon.seq.seqTypesStruct.getColors(iFound);

                g.setColor(new Color(iColor[0], iColor[1], iColor[2]));
                g.fillRect(iStart, iDepth1, iWidthTrack, iDepth);

                g.setColor(Color.black);
                g.drawString(stSeq.stItem[i].abbrev,
                             iStartTrack+2, iDepth1+iDepth/2);
              }
            }
          }
        }
      }

      for (int i=0; i<stSeq.iCount; i++)
      {
        if (stSeq.stItem[i] != null)
        {
          depth1 = stSeq.stItem[i].depthStart;
          depth2 = stSeq.stItem[i].depthEnd;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio = (double) iLogHeight * (depthStart - depth2) /
                                         (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth2 = iHeight - (int) dRatio;

          iDepth = Math.abs(iDepth2 - iDepth1);

          if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 < iHeight))
          {
            iFound = -1;
            for (int k=0; k<horizon.seq.seqTypesStruct._TOTAL; k++)
            {
              if (stSeq.stItem[i].sMnemonic.equals(
                    horizon.seq.seqTypesStruct.NAME[k][2]))
              {
                iFound = k;
              }
            }

            if (iFound > -1)
            {
              if (iFound > horizon.seq.seqTypesStruct._LW)
              {
                iColor = horizon.seq.seqTypesStruct.getForeColors(iFound);

                g.setColor(new Color(iColor[0], iColor[1], iColor[2]));
                g.drawLine(iStartTrack, iDepth1,
                           iStartTrack+iWidthTrack, iDepth1);
                g.drawLine(iStartTrack, iDepth1+1,
                           iStartTrack+iWidthTrack, iDepth1+1);

                g.setColor(Color.black);
                g.drawString(stSeq.stItem[i].abbrev,
                             iStartTrack+2, iDepth1+4);
              }
            }
          }
        }
      }
    }

    g.setColor(Color.black);

    fsb  = new Font("Serif", Font.BOLD, 12);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Sequence",     LABELSTART+14, -1*(iStart+2*iWide/3));
    g2.drawString("Stratigraphy", LABELSTART+7,  -1*(iStart+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStart, LABELSTART,  iEnd,   LABELSTART);
    g.drawLine(iStart, PLOT_TITLES, iEnd,   PLOT_TITLES);
    g.drawLine(iStart, LABELSTART,  iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART,  iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Porosity Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_TOPS;

    if (iSelected == iqstrat.iqstratTracksStruct._SEQUENCE_STRAT)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawSequenceTrack( g, iStartTrack, iWidthTrack );
      drawGrid( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}