/*
 * @regionsUtility.java Version 1.1 09/28/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.regions;

import horizon.regions.regionsListStruct;
import horizon.regions.regionsStruct;

import iqstrat.iqstratHeadersStruct;

/** Class regionsUtility
 *  <p> This Class will provide basic utilities for the Regions
 *      data structures.
 *
 *  @version 1.1 09/28/2007
 *  @author  John R. Victorine
 */

public class regionsUtility
{
  public static final int TOTAL_SOURCES  = 20;

  /** Method addHeaderData()
   * <p> This method will add the header data to the regions List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Regions List Data Structure
   * @return st       = The Regions List Data Structure.
   */

  public static regionsListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                regionsListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one region list structure to another
   * @param  stOld = Old region list data structure
   * @return stNew = New region list data structure
   */

  public static regionsListStruct copyList(regionsListStruct stOld)
  {
    int i,j;
    regionsListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new regionsListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new regionsStruct[stOld.iCount];

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID); // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY); // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI); // API-Number of Location
        stNew.sName  = new String(stOld.sName); // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL; // Ground Level

        stNew.sKGS  = new String( stOld.sKGS );      // KGS Saved Data Indicator

        // LAS 3 Dependent Variable

        stNew.sPERFTYPE = new String(stOld.sPERFTYPE); // Perforation Charge Type

        // LAS 3 File Parameters

        stNew.iParamRows = stOld.iParamRows;    // Total number of rows
        stNew.iParamCols = stOld.iParamCols;    // Total number of columns
        stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

        for (i=0; i<stNew.iParamRows; i++)
        {
          for (j=0; j<stNew.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
          }
        }

        for (i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static regionsListStruct transfer(regionsListStruct stOld)
  {
    regionsListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method add()
   * <p> This method will add the Regions to an existing list
   * @param  stRegion = The Region Data Structure
   * @param  st       = The Old Regions List Data Structure
   * @return st       = The new Regions List Data Structure.
   */

  public static regionsListStruct add(regionsStruct stRegion,
                                      regionsListStruct st)
  {
    int    i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    regionsListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new regionsListStruct();
    stTemp.stItem = new regionsStruct[iRecords];

    if (st != null)
    {
	  iSource      = st.iSource;
      stTemp.sKGS  = new String( st.sKGS );      // KGS Saved Data Indicator

      // LAS 3 Dependent Variable

      stTemp.sPERFTYPE = new String(st.sPERFTYPE); // Perforation Charge Type

      // LAS 3 File Parameters

      stTemp.iParamRows = st.iParamRows;    // Total number of rows
      stTemp.iParamCols = st.iParamCols;    // Total number of columns
      stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

      for (i=0; i<st.iParamRows; i++)
      {
        for (j=0; j<st.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stTemp.sParams[i][j] = new String( st.sParams[i][j] );
        }
      }

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new regionsStruct();
    stTemp.stItem[iCount] = copy(stRegion);
    iCount++;

    stTemp.iCount = iCount;

    st        = new regionsListStruct();
    st.stItem = new regionsStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;
    st.sKGS  = new String( stTemp.sKGS );      // KGS Saved Data Indicator

    // LAS 3 Dependent Variable

    st.sPERFTYPE = new String(stTemp.sPERFTYPE); // Perforation Charge Type

    // LAS 3 File Parameters

    st.iParamRows = stTemp.iParamRows;    // Total number of rows
    st.iParamCols = stTemp.iParamCols;    // Total number of columns
    st.sParams    = new String[st.iParamRows][st.iParamCols];

    for (i=0; i<st.iParamRows; i++)
    {
      for (j=0; j<st.iParamCols; j++)
      {
        // Array holding the Parameter Definitions
        st.sParams[i][j] = new String( stTemp.sParams[i][j] );
      }
    }

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a region in an existing list
   * @param  sKID   = The Primary Key from DB
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  stRegion = region data structure
   * @param  st     = region data list structure
   * @return st     = New region data list structure
   */

  public static regionsListStruct modify(String sKID,
                                         String sKEY,
                                         regionsStruct stRegion,
                                         regionsListStruct st)
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if (((sKID.equals(st.stItem[i].sKID)) && (!sKID.equals("0"))) ||
            ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0"))))
        {
          st.stItem[i] = copy(stRegion);
        }
      }
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Rock fossil from an existing list
   * @param  sKID   = The Primary Key from DB
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  st     = region data list structure
   * @return st     = New region data list structure
   */

  public static regionsListStruct remove(String sKID,
                                         String sKEY,
                                         regionsListStruct st)
  {
    int i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    int    iSource  = -1;
    regionsListStruct stTemp = null;

    if (st != null)
    {
      if (st.iCount == 1)
      {
        st.delete();
        st = null;
      }
      else
      {
        iRecords = st.iCount-1;

        stTemp        = new regionsListStruct();
        stTemp.stItem = new regionsStruct[iRecords];

        iSource      = st.iSource;

        stTemp.sKGS  = new String( st.sKGS );      // KGS Saved Data Indicator

        // LAS 3 Dependent Variable

        stTemp.sPERFTYPE = new String(st.sPERFTYPE); // Perforation Charge Type

        // LAS 3 File Parameters

        stTemp.iParamRows = st.iParamRows;    // Total number of rows
        stTemp.iParamCols = st.iParamCols;    // Total number of columns
        stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            stTemp.sParams[i][j] = new String( st.sParams[i][j] );
          }
        }

        for (i=0; i<st.iCount; i++)
        {
          if (((sKID.equals(st.stItem[i].sKID)) && (!sKID.equals("0"))) ||
              ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0"))))
          {
            iFound = i;
          }
        }

        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();

        stTemp.iCount = iCount;

        st        = new regionsListStruct();
        st.stItem = new regionsStruct[iCount];
        st.iCount = iCount;

        st.iSource = iSource;

        st.sKGS  = new String( stTemp.sKGS );      // KGS Saved Data Indicator

        // LAS 3 Dependent Variable

        st.sPERFTYPE = new String(stTemp.sPERFTYPE); // Perforation Charge Type

        // LAS 3 File Parameters

        st.iParamRows = stTemp.iParamRows;    // Total number of rows
        st.iParamCols = stTemp.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stTemp.sParams[i][j] );
          }
        }

        for (i=0; i<stTemp.iCount; i++)
        {
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        stTemp.delete();
      }
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one region structure to another
   * @param  stOld = Old region data structure
   * @return stNew = New region data structure
   */

  public static regionsStruct copy(regionsStruct stOld)
  {
    regionsStruct stNew = new regionsStruct();

    if (stOld != null)
    {
      stNew.sKID       = new String(stOld.sKID);   // Primary KEY of Record
      stNew.sKEY       = new String(stOld.sKEY);   // User defined KEY of Record
      stNew.sID        = new String(stOld.sID);    // ID of the Regions
      stNew.id         = stOld.id; // The Numeric equivalent to the Region Text.

      stNew.sequence   = new String(stOld.sequence); // Sequence Number
      stNew.sRank      = new String(stOld.sRank);  // Stratigraphic Unit Rank
      stNew.iLevel     = stOld.iLevel;         // Confidence level of selection

      stNew.sEon       = new String(stOld.sEon);   // Eon of this formation
      stNew.sEra       = new String(stOld.sEra);   // Era of this formation
      stNew.system     = new String(stOld.system); // System of this Formation
      stNew.systemName = new String(stOld.systemName); // System Name
      stNew.series     = new String(stOld.series);     // Series id of Formation
      stNew.seriesName = new String(stOld.seriesName); // Series Name

      stNew.sName      = new String(stOld.sName);  // Region Name
      stNew.sName2     = new String(stOld.sName2); // Original Region Name
      stNew.source     = new String(stOld.source); // Source Name

      stNew.depth_top  = stOld.depth_top;          // Starting Depth.
      stNew.depth_base = stOld.depth_base;         // Ending Depth.
      stNew.dTime_top  = stOld.dTime_top;          // Starting Time.
      stNew.dTime_base = stOld.dTime_base;         // Ending Time.
      stNew.shots      = stOld.shots;              // Number of shots / foot
    }

    return (stNew);
  }

  /** Method getSourceNames()
   * <p> This method will retrieve the list of source names
   * @param  st      = regions List Data Structure
   * @return sources = array of source names
   */

  public static String[] getSourceNames(regionsListStruct st)
  {
    String sources[] = null;
    int    iSources  = 0;
    int    iFound    = 0;
    int    i = 0;
    int    j = 0;

    sources = new String[TOTAL_SOURCES];

    for (i=0; i<TOTAL_SOURCES; i++)
      sources[i] = new String("");

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].id == horizon.regions.regionsStruct._TOPS)
        {
          iFound = 0;
          for (j=0; j<iSources; j++)
          {
            if (st.stItem[i].source.equals(sources[j]))
            {
              iFound = 1;
            }
          }

          if (iFound == 0)
          {
            if (iSources < TOTAL_SOURCES)
            {
              sources[iSources] = new String(st.stItem[i].source);
              iSources++;
            }
          }
        }
      }
    }

    return (sources);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = region data list structure
   * @return st = sorted region data list structure
   */

  public static regionsListStruct bubbleSort(regionsListStruct st)
  {
    boolean swappedOnPrevRun  = true;
    regionsStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depth_top > st.stItem[i+1].depth_top)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print(regionsListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(i +" "+
                           st.stItem[i].sKID +" "+
                           st.stItem[i].sKEY +" "+
                           st.stItem[i].sID  +" "+
                           st.stItem[i].id);
//        System.out.println("----"+st.stItem[i].sequence+" "+
//                                  st.stItem[i].sRank+" "+
//                                  st.stItem[i].iLevel+" "+
//                                  st.stItem[i].system+" "+
//                                  st.stItem[i].systemName+" "+
//                                  st.stItem[i].sEra+" "+
//                                  st.stItem[i].sName);
//        System.out.println("----"+st.stItem[i].sName2+" "+
//                                  st.stItem[i].source +" "+
//                                  st.stItem[i].depth_top+" "+
//                                  st.stItem[i].depth_base +" "+
//                                  st.stItem[i].shots );
        System.out.println(" ---- "+st.stItem[i].depth_top+" "+
                                    st.stItem[i].depth_base +" "+
                                    st.stItem[i].shots );
      }
      System.out.println(" ------------------------------------------------- ");
    }
  }
}