/*
 * @ReadRegionsXMLFile.java Version 1.0 01/29/2004
 *
 * Copyright (c) 2001 John R. Victorine
 * All Rights Reserved.
 */

package horizon.regions.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import horizon.regions.regionsListStruct;
import horizon.regions.regionsStruct;

/** Class ReadRegionsXMLFile()
 * <p> This Class will parse the Regions XML Data Stream
 *     directly to the Regions List Data Structure.
 *
 *  @version 1.1 01/29/2004
 *  @author  John R. Victorine
 */

public class ReadRegionsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  public static final String ROOTNAME    = "regions";   // Root Name of XML
  public static final String RECORDS     = "records";   // Record Count

  public static final String WELL        = "well";      // ELEMENT info

  public static final String KID         = "kid";       // Well Primary key
  public static final String KEY         = "key";       // Primary Key
  public static final String TYPE        = "type";      // type of data
  public static final String API_NUMBER  = "api";       // API-Number
  public static final String WELL_STATUS = "status";    // Well Status
  public static final String ERROR_VAL   = "error";     // DB Error
  public static final String LATITUDE    = "latitude";  // Latitude
  public static final String LONGITUDE   = "longitude"; // Longitude
  public static final String DEPTH       = "depth";     // Depth or Height
  public static final String GROUND_LEVEL = "gl";       // Ground Level

  public static final String REGION      = "region";    // ELEMENT Region
  public static final String ID          = "id";        // TOPS, PERF, DST
  public static final String RANK        = "rank";      // Rank of formation
  public static final String SEQUENCE    = "sequence";  // Sequence ID

  public static final String SYSTEM      = "system";    // System ID
  public static final String SERIES      = "series";    // Series ID

  public static final String NAME        = "name";      // Region Name
  public static final String NAME2       = "alt_name";  // Alternate Region Name
  public static final String SOURCE      = "source";    // Source Name

  public static final String DEPTH_START = "top";       // Starting Depth
  public static final String DEPTH_END   = "base";      // Ending Depth
  public static final String SHOTS       = "shots";     // Number of Shots/foot

  private static final int    SYSTEM_TOTAL = 12;
  private static final String SYSTEM_ID[][] = {
  //     KGS KID,      Name,          KGS ID,       ID
      {"1029095988","Quaternary",    "100500", "10101000"},
      {"1027710578","Cretaceous",    "101000", "10201000"},
      {"1027710665","Jurassic",      "120000", "10202000"},
      {"1027710667","Triassic",      "122000", "10203000"},
      {"1027710669","Permian",       "124000", "10301000"},
      {"1027710965","Pennsylvanian", "194000", "10303000"},
      {"1027711178","Mississippian", "358000", "10304000"},
      {"1027711206","Devonian",      "387000", "10305000"},
      {"1027711209","Silurian",      "390000", "10306000"},
      {"1027711210","Ordovician",    "391000", "10307000"},
      {"1027711222","Cambrian",      "405000", "10308000"},
      {"1027711226","Precambrian",   "409000", "20000000"}};

  private int               iType  = FILE;
  private int               iRows  =  0;
  private int               iCount = -1;
  private int               iError = 0;
  private String            sError = "";
  private regionsListStruct stList = new regionsListStruct();

  /** Method ReadRegionsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadRegionsXMLFile() { iType = FILE; }

  /** Method ReadRegionsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadRegionsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public regionsListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);
    stList.iCount = iCount+1;

    if (stList != null)
    {
      for (int i=0; i<stList.iCount; i++)
      {
        if ((stList.stItem[i].sKEY.equals("0")) &&
            (!stList.stItem[i].sKID.equals("0")))
          stList.stItem[i].sKEY = new String(stList.stItem[i].sKID);
      }
    }

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(REGION))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new regionsStruct();
      }
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i<attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new regionsStruct[iRows];
          }
        }

        // Parse each attribute and add it to the structure

        if (eName.equals(WELL))
        {
          ParseData(aName, sOut.trim());
        }

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            if (eName.equals(REGION))
            {
              stList.stItem[iCount] =
                  ParseData(aName, sOut.trim(), stList.stItem[iCount]);
            }
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** Method ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Regions String
   * @param  st         = The Regions Data Structure
   * @return st         = The Regions Data Structure
   */

  private regionsStruct ParseData(String sIdentifier,
                                  String sData,
                                  regionsStruct st)
  {
    int    i=0;
    int    j=0;
    int    iFound = -1;
    int    iValue = -1;
    String sEonID = "0";
    String sEraID = "0";

    if (sIdentifier.equals(KID)) { st.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { st.sKEY = new String(sData); }

    if (sIdentifier.equals(ID))       // ID of the Region
    {
      st.sID = new String(sData);

      if (st.sID.equals(horizon.regions.regionsStruct.TOPS))        // Formation Tops
        st.id = horizon.regions.regionsStruct._TOPS;

      if (st.sID.equals(horizon.regions.regionsStruct.PERFORATION)) // Perforation
        st.id = horizon.regions.regionsStruct._PERFORATION;

      if (st.sID.equals(horizon.regions.regionsStruct.DST))         // DST
        st.id = horizon.regions.regionsStruct._DST;
    }

    if (sIdentifier.equals(SYSTEM))
    {
      st.system = new String(sData);

      for (i=0; i<SYSTEM_TOTAL; i++)
      {
        if (st.system.equals(SYSTEM_ID[i][2]))
        {
          st.system = new String(SYSTEM_ID[i][3]);
        }
      }

      iFound = -1;
      for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
      {
        if (st.system.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
          iFound = i;
      }

//      for (i=0; i<horizon.stratigraphicUnitsStruct.COLORS_TOTAL; i++)
//      {
//        if (st.system.equals(horizon.stratigraphicUnitsStruct.STRAT_COLORS[i][3]))
//          iFound = i;
//      }

      if (iFound > -1)
      {
        st.systemName = new String(
            horizon.stratigraphicChartStruct.CHART[iFound][2]);
//            horizon.stratigraphicUnitsStruct.STRAT_COLORS[iFound][1]);
        iFound = -1;
        for (j=0; j<horizon.stratigraphicChartStruct.ISC_TOTAL; j++)
        {
          if (st.system.equals(horizon.stratigraphicChartStruct.ISC[j][2]))
            iFound = j;
        }

        if (iFound > -1)
        {
          sEonID = new String(horizon.stratigraphicChartStruct.ISC[iFound][0]);
          sEraID = new String(horizon.stratigraphicChartStruct.ISC[iFound][1]);

          if (!sEonID.equals("0"))
          {
            for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
            {
              if (sEonID.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
                st.sEon = new String(horizon.stratigraphicChartStruct.CHART[i][2]);
            }
          }

          if (!sEraID.equals("0"))
          {
            for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
            {
              if (sEraID.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
                st.sEra = new String(horizon.stratigraphicChartStruct.CHART[i][2]);
            }
          }
        }
//        iValue = cmn.cmnString.stringToInt(
//                     horizon.stratigraphicUnitsStruct.STRAT_COLORS[iFound][2]);

//        if (iValue > -1)
//          st.sEra = new String(horizon.stratigraphicUnitsStruct.EON[iValue][1]);
      }
    }

    if (sIdentifier.equals(SERIES))
    {
      st.series = new String(sData);

      iFound = -1;
      for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
      {
        if (st.series.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
          iFound = i;
      }

      if (iFound > -1)
      {
        st.seriesName = new String(
            horizon.stratigraphicChartStruct.CHART[iFound][2]);

        iFound = -1;
        for (j=0; j<horizon.stratigraphicChartStruct.ISC_TOTAL; j++)
        {
          if (st.system.equals(horizon.stratigraphicChartStruct.ISC[j][3]))
            iFound = j;
        }

        if (iFound > -1)
        {
          sEonID = new String(horizon.stratigraphicChartStruct.ISC[iFound][0]);
          sEraID = new String(horizon.stratigraphicChartStruct.ISC[iFound][1]);
          if (st.system.length() < 2)
            st.system =
                new String(horizon.stratigraphicChartStruct.ISC[iFound][2]);

          if (!sEonID.equals("0"))
          {
            for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
            {
              if (sEonID.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
                st.sEon = new String(horizon.stratigraphicChartStruct.CHART[i][2]);
            }
          }

          if (!sEraID.equals("0"))
          {
            for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
            {
              if (sEraID.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
                st.sEra = new String(horizon.stratigraphicChartStruct.CHART[i][2]);
            }
          }

          if (st.system.length() > 1)
          {
            for (i=0; i<horizon.stratigraphicChartStruct.TOTAL; i++)
            {
              if (st.system.equals(horizon.stratigraphicChartStruct.CHART[i][0]))
                st.systemName =
                    new String(horizon.stratigraphicChartStruct.CHART[i][2]);
            }
          }
        }
      }
    }

    if (sIdentifier.equals(SEQUENCE)) { st.sequence = new String(sData); }
    if (sIdentifier.equals(RANK))     { st.sRank    = new String(sData); }

    if (sIdentifier.equals(NAME))     { st.sName    = new String(sData); }
    if (sIdentifier.equals(NAME2))    { st.sName2   = new String(sData); }
    if (sIdentifier.equals(SOURCE))   { st.source   = new String(sData); }
    if (sIdentifier.equals(DEPTH_START))
    {
      st.depth_top = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depth_top = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(DEPTH_END))   // End Depth
    {
      st.depth_base = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depth_base = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(SHOTS))   //Shots / Foot
    {
      st.shots = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.shots = cmn.cmnString.stringToInt(sData);
    }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Well Header Information String
   */

  private void ParseData(String sIdentifier, String sData)
  {
    // Identification Information

    if (sIdentifier.equals(KID)) { stList.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { stList.sKEY = new String(sData); }
    if (sIdentifier.equals(TYPE))
    {
      if (sData.equals("WELL"))    stList.iType = iqstrat.iqstratHeadersStruct.WELL;
      if (sData.equals("OUTCROP")) stList.iType = iqstrat.iqstratHeadersStruct.OUTCROP;
    }
    if (sIdentifier.equals(API_NUMBER))  { stList.sAPI = new String(sData); }
    if (sIdentifier.equals(NAME))
    {
      stList.sName = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(WELL_STATUS))
    {
      stList.status = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = -1;
      sError    = new String(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      stList.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      stList.dLongitude = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(DEPTH))
    {
      stList.depth = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      stList.dGL = cmn.cmnString.stringToDouble(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}