/*
 * @regionsPerfFrame.java Version 1.1 03/16/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.regions.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import horizon.regions.regionsListStruct;
import horizon.regions.regionsStruct;
import horizon.regions.gui.regionsPerfTable;

/** Class regionsPerfFrame
 *  <p> This Class will allow the user to enter the rock data into the profile
 *
 *  @version 1.1 03/15/2011
 *  @author  John R. Victorine
 */

public class regionsPerfFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable        notifier = null;
  private regionsListStruct stList   = null;

  // Global Variables

  private String sKEY   = "0";
  private double dStart = 0.0;
  private double dEnd   = 0.0;
  private int    iShots = 0;

  private regionsPerfTable pTable = null;

  private static final int _ADD    = 0;
  private static final int _MODIFY = 1;

  private int iAction = _ADD;

  // Global Frame Widgets

  private JTextField txtStart = new JTextField();
  private JTextField txtEnd   = new JTextField();
  private JTextField txtShots = new JTextField();

  // Modify Data Structure Buttons

  private JButton      btnAdd            = new JButton();
  private JButton      btnClear          = new JButton();

  // Modify Data List Structure Buttons

  private JButton      btnModify         = new JButton();
  private JButton      btnRemove         = new JButton();
  private JButton      btnRemoveAll      = new JButton();

  // Menus

  // File Button Menu

  private JMenuItem mExit  = null;

  /** Constructor rockDataEntryFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stList   = Perforation List Data Structure
   */

  public regionsPerfFrame( Observable notifier, regionsListStruct stList )
  {
    try
    {
      this.notifier = notifier;
      this.stList   = stList;

      jbInit();
      addWindowListener( new regionsPerfFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel   pnlTop    = new JPanel();
    JPanel   pnlStart  = new JPanel();
    JPanel   pnlEnd    = new JPanel();
    JPanel   pnlShots  = new JPanel();
    JPanel   pnlAdd    = new JPanel();
    JPanel   pnlCenter = new JPanel();
    JPanel   pnlBase   = new JPanel();

    JScrollPane scroll = new JScrollPane();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Start Depth:");

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "End Depth:");

    TitledBorder titledBorderShots = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Shots/Foot:");

    JMenuBar menuBar   = new JMenuBar();
    JMenu    menuFile  = new JMenu("Close Dialog");

    // Create Menu Options

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add(mExit);
    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Create Frame Widgets

    this.setTitle("Perforation Data Entry");
    this.getContentPane().setLayout(new BorderLayout());

    // Top Panel

    pnlTop.setLayout( new GridLayout() );

    // .. Starting Depth Panel

    pnlStart.setBorder( titledBorderStart );
    pnlStart.setLayout( new BorderLayout( ));

    txtStart.setText( "" + dStart );
    txtStart.setHorizontalAlignment( SwingConstants.TRAILING );
    txtStart.addFocusListener( new regionsPerfFrameFocusAdapter(this) );

    // .. Endind Depth Panel

    pnlEnd.setBorder( titledBorderEnd );
    pnlEnd.setLayout( new BorderLayout() );

    txtEnd.setText( "" + dEnd );
    txtEnd.setHorizontalAlignment( SwingConstants.TRAILING );
    txtEnd.addFocusListener( new regionsPerfFrameFocusAdapter(this) );

    // .. Endind Depth Panel

    pnlShots.setBorder( titledBorderShots );
    pnlShots.setLayout( new BorderLayout() );

    txtShots.setText( "" + iShots );
    txtShots.setHorizontalAlignment( SwingConstants.TRAILING );
    txtShots.addFocusListener( new regionsPerfFrameFocusAdapter(this) );

    // -- Add/Clear Panel

    pnlAdd.setLayout( new GridLayout(2,1) );

    btnAdd.setFont( new java.awt.Font("Dialog", 1, 11) );
    btnAdd.setText( "Add" );
    btnAdd.addActionListener( this );

    btnClear.setFont( new java.awt.Font("Dialog", 1, 11) );
    btnClear.setText( "Clear" );
    btnClear.addActionListener( this );

    // Center Panel

    pnlCenter.setLayout( new BorderLayout() );

    // -- Perforations Table

    pTable = new regionsPerfTable( stList );
    scroll = pTable.getScrollPane();

    // Bottom Panel

    btnModify.setFont( new java.awt.Font("Dialog", 1, 11) );
    btnModify.setPreferredSize( new Dimension(98, 25) );
    btnModify.setText( "Modify");
    btnModify.addActionListener( this );

    btnRemove.setFont( new java.awt.Font("Dialog", 1, 11) );
    btnRemove.setPreferredSize( new Dimension(98, 25) );
    btnRemove.setText( "Remove");
    btnRemove.addActionListener( this );

    btnRemoveAll.setFont( new java.awt.Font("Dialog", 1, 11) );
    btnRemoveAll.setPreferredSize( new Dimension(98, 25) );
    btnRemoveAll.setText( "Remove All");
    btnRemoveAll.addActionListener( this );

    // Attach Widgets to Frame

    this.getContentPane().add( pnlTop,      BorderLayout.NORTH );
    pnlTop.add( pnlStart,                   null );
    pnlStart.add( txtStart,                 BorderLayout.CENTER );
    pnlTop.add( pnlEnd,                     null );
    pnlEnd.add( txtEnd,                     BorderLayout.CENTER );
    pnlTop.add( pnlShots,                   null );
    pnlShots.add( txtShots,                 BorderLayout.CENTER );
    pnlTop.add( pnlAdd,                     null );
    pnlAdd.add( btnAdd,                     null );
    pnlAdd.add( btnClear,                   null );

    this.getContentPane().add( pnlCenter,   BorderLayout.CENTER );
    pnlCenter.add( scroll,                  BorderLayout.CENTER );

    this.getContentPane().add( pnlBase,     BorderLayout.SOUTH );
    pnlBase.add( btnModify,                 null );
    pnlBase.add( btnRemove,                 null );
    pnlBase.add( btnRemoveAll,              null );

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    this.setSize( new Dimension(400, 300) );
    this.setLocation( (d.width - this.getSize().width) / 2,
                      (d.height - this.getSize().height) / 2 );

    this.setResizable( false );
    this.setVisible( true );
  }

  /** METHOD regionsPerfFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class regionsPerfFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve the Perforations Data List Structure
   * @param stList = the Perforations Data List Structure
   */

  public regionsListStruct getData() { return (stList); }

  /** Method getScreen()
   * <p> This method will retrieve data from the screen and load into data
   *     structure
   */

  public regionsStruct getScreen()
  {
	regionsStruct st = new regionsStruct();

	if (sKEY.equals("0"))
	{
	  st.sKEY     = new String( cmn.cmnString.UniqueName() );
    }
	else
	{
	  st.sKEY     = new String( sKEY );
    }

    st.sKID       = new String( st.sKEY );
    st.sID        = new String( horizon.regions.regionsStruct.PERFORATION );
    st.id         = horizon.regions.regionsStruct._PERFORATION;
    st.depth_top  = dStart;
    st.depth_base = dEnd;
    st.shots      = iShots;

    return (st);
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setButtons()
   * <p> This method will enable & disable buttons depending on the data
   */

  private void setButtons()
  {
    btnModify.setEnabled( false );
    btnRemove.setEnabled( false );
    btnRemoveAll.setEnabled( false );

    if ((pTable.getTotalRows() > 0) && (iAction == _ADD))
    {
      btnModify.setEnabled( true );
      btnRemove.setEnabled( true );
      btnRemoveAll.setEnabled( true );
	}

	if (iAction == _ADD)
	{
	  btnAdd.setText("Add");
	}
	else
	{
	  btnAdd.setText("Modify");
	}
  }

  /** Method clear()
   * <p> This method will clear the text fields
   */

  private void clear()
  {
	sKEY   = new String( "0" );
    dStart = 0.0;
    dEnd   = 0.0;
    iShots = 0;

    txtStart.setText( "" + dStart );
    txtEnd.setText( "" + dEnd );
    txtShots.setText( "" + iShots );

    iAction = _ADD;
  }

  /** Method setScreen()
   * <p> This method will retrieve data from the screen and load into data
   *     structure
   */

  public void setScreen( regionsStruct st )
  {
    if (st != null)
    {
	  sKEY   = new String( st.sKEY );
      dStart = st.depth_top;
      dEnd   = st.depth_base;
      iShots = st.shots;

      txtStart.setText( "" + dStart );
      txtEnd.setText( "" + dEnd );
      txtShots.setText( "" + iShots );
	}
  }

  /* ===================================================================== *
   * -------------------------- DATA ACTIONS ----------------------------- *
   * ===================================================================== */

  /** Method addData()
   * <p> This method will add/modify the data in the list.
   */

  public void addData()
  {
	regionsStruct st = null;

    st = getScreen();
    if (st != null)
    {
  	  if (iAction == _ADD)
	  {
        stList = horizon.regions.regionsUtility.add( st, stList );
      }
      else
      {
        stList = horizon.regions.regionsUtility.modify( st.sKID, st.sKEY, st, stList );
	  }

      if (pTable != null)
        pTable.setData( stList );

      clear();

  	  if (notifier != null)
        notifier.notifyObservers(new String("Peforation Data Changed"));
    }
  }

  /** Method modifyData()
   * <p> This method will retrieve the row from the list and load into the
   *     data entry dialog.
   */

  public void modifyData()
  {
	regionsStruct st = null;

    if ( pTable != null )
    {
	  st = pTable.getRowData();

	  if (st != null)
	  {
		setScreen( st );

        iAction = _MODIFY;
      }
    }
  }

  /** Method removeRow()
   * <p> This method will remove a row from the List.
   */

  public void removeRow()
  {
	regionsStruct st = null;

    if ( pTable != null )
    {
	  st = pTable.getRowData();

	  if (st != null)
	  {
		stList = horizon.regions.regionsUtility.remove( st.sKID, st.sKEY, stList );

        pTable.setData( stList );

  	    if (notifier != null)
          notifier.notifyObservers(new String("Peforation Data Changed"));
      }
    }
  }

  /** Method removeList()
   * <p> This method will remove the List.
   */

  public void removeList()
  {
	if (stList != null)
	  stList.delete();
	stList = null;

    if ( pTable != null )
      pTable.setData( stList );

	if (notifier != null)
      notifier.notifyObservers(new String("Peforation Data Changed"));
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close & destroy the Rock Dialog
   */

  public void close()
  {
    // Input Variables

    notifier = null;
    stList   = null;

    sKEY     = null;

    if (pTable != null)
      pTable.close();
    pTable = null;

    // Global Frame Widgets

    txtStart     = null;
    txtEnd       = null;
    txtShots     = null;

    // Modify Data Structure Buttons

    btnAdd       = null;;
    btnClear     = null;;

    // Modify Data List Structure Buttons

    btnModify    = null;;
    btnRemove    = null;;
    btnRemoveAll = null;;

    // File Button Menu

    mExit        = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnAdd) { addData(); }
    if (event.getSource() == btnClear) { clear(); }

    // Modify Data List Structure Buttons

    if (event.getSource() == btnModify) { modifyData(); }
    if (event.getSource() == btnRemove) { removeRow(); }
    if (event.getSource() == btnRemoveAll) { removeList(); }

    setButtons();

    if (event.getSource() == mExit) { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp    = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp    = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtShots)
    {
      iNumeric = 1;
      sTemp    = txtShots.getText();
      sMessage = new String("Shots/Foot Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());

        if (e.getSource() == txtShots)
          iShots = cmn.cmnString.stringToInt(txtShots.getText());
      }
    }
  }
}

/** CLASS regionsPerfFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class regionsPerfFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  regionsPerfFrame adaptee;

  regionsPerfFrameFocusAdapter(regionsPerfFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 03/16/2011
 *  @author  John R. Victorine
 */
