/*
 * @topsCSVFile.java Version 1.1 11/27/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.io;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import util.utilFileIO;
import cmn.cmnString;
import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

/** Class topsCSVFile
 *  <p> This Class is designed to open an ascii text tops CSV file and read
 *      the contents and parse the data into the regions Structure.
 *
 *  @version 1.1 11/27/2007
 *  @author  John R. Victorine
 */

public class topsCSVFile
{
  private utilFileIO        pFILE  = null;

  private String            sLine1 = "";
  private String            sLine2 = "";
  private stratListStruct stList = null;

  private int         iRows    = 0;
  private int         iColumns = 0;

  private int         iError   = util.utilFileIO.NOERROR;
  private String      sError   = new String("");

  public static final String TOP         = "Depth Top";
  public static final String BASE        = "Depth Base";
  public static final String RANK        =
      "Stratigraphic Unit Rank [ SYSTEM, GROUP, etc. ]";
  public static final String NAME        = "Stratigraphic Name";
  public static final String NAME2       = "Alternate Name";
  public static final String ERA         = "Era";
  public static final String SYSTEM      = "System";
  public static final String SUBSYSTEM   = "Subsystem";
  public static final String SERIES      = "Series";
  public static final String SUBSERIES   =
      "Subseries { Pennsylvanian & Mississippian Series }";
  public static final String STAGE       = "Stage";
  public static final String GROUP       = "Group";
  public static final String SUBGROUP    = "Subgroup";
  public static final String FORMATION   = "Formation";
  public static final String AGE_START   = "Start Age (Ma)";
  public static final String AGE_END     = "End Age (Ma)";

  private static final int    TOTAL = 16;
  private static final String COLUMNS[] =
  {
    TOP,
    BASE,
    RANK,
    NAME,
    NAME2,
    ERA,
    SYSTEM,
    SUBSYSTEM,
    SERIES,
    SUBSERIES,
    STAGE,
    GROUP,
    SUBGROUP,
    FORMATION,
    AGE_START,
    AGE_END
  };

  /** Construct topsCSVFile()
   *  <p> This is the Constructor for this class.
   */

  public topsCSVFile() { }

  /* ====================================================================== *
   * -------------------------- LAS FILE METHODS -------------------------- *
   * ====================================================================== */

  /** Method Open()
    * <p> This is the Constructor for this class.
    * @param iFileID    = To access the file as URL or FILE.
    * @param sDirectory = File URL or Directory Path
    * @param sFilename  = File Name
    */

  public void Open(int iFileID, String sDirectory, String sFilename)
  {
    pFILE   = new utilFileIO();
    pFILE.Open(iFileID, util.utilFileIO.READ, sDirectory, sFilename);
    iError = pFILE.getErrorID();
    sError = pFILE.getError();

    iRows             = 0;
    iColumns          = 0;
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close() { pFILE.Close(); }

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    sLine1   = null;
    sLine2   = null;
    sError   = null;

    if (pFILE != null)
      pFILE.delete();
    pFILE = null;
  }

  /** Method get2Rows()
   * <p> This method will read and parse the Tops File and get the first 2 lines.
   */

  public void get2Rows()
  {
    int             iCount  = 0;
    String          sTemp   = "";
    StringTokenizer stToken = null;

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        sTemp = pFILE.getString();
        if (iCount == 0)
        {
          sLine1   = new String(sTemp);
          stToken  = new StringTokenizer(sTemp, ",");
          iColumns = stToken.countTokens();
        }

        if (iCount == 1)
          sLine2 = new String(sTemp);

        iCount++;
        iRows++;
      }
    }
  }

  /** Method getData()
   * <p> This method will read and parse the Tops File into its data structures.
   * @param iStart  = the starting row number
   * @param iColumn = Array of file column mapping to the regions data structure
   */

  public void getData(int iStart, int iColumn[])
  {
    int    iCount = 1;
    String sTemp  = "";
    stratStruct st = null;
    String sKEY = cmn.cmnString.UniqueName();
    String sKID = "0";

    stList = new stratListStruct();

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (iCount >= iStart)
          {
            sTemp = pFILE.getString();
            sKID  = new String(sKEY + iCount);
            st    = parseRow(sKID, sTemp, iColumn);

            if ((st.sName.length() > 0) && (st.sName2.length() == 0))
              st.sName2 = new String(st.sName);

            if ((st.sName2.length() > 0) && (st.sName.length() == 0))
              st.sName = new String(st.sName2);

            stList = horizon.strat.stratUtility.add(st, stList);
          }

          iCount++;
        }
      }
    }
//horizon.strat.stratUtility.print(stList);
  }

  /** Method parseRow()
   * <p> This method will parse a Tops File row into the regions structure.
   * @param  sKEY    = Unique key for formation name
   * @param  sData   = Data Row String
   * @param  iColumn = The mapped columns
   * @return st      = the regions Data Structure
   */

  private stratStruct parseRow(String sKEY, String sData, int iColumn[])
  {
    stratStruct st = new stratStruct();
    int    iCount = 0;
    int    iFound = 0;
    int    iNext  = 0;
    String sTemp  = "";
    StringTokenizer stToken = null;

    st.sKEY = new String(sKEY);

    sTemp    = new String(sData.replace('\t', ' '));
    stToken  = new StringTokenizer(sTemp, ",");
    iColumns = stToken.countTokens();

    for (int i=0; i<iColumns; i++)
    {
      sTemp = stToken.nextToken();
      sTemp = new String(sTemp.trim());

      iNext  = i+1;
      iFound = -1;
      for (int j=0; j<TOTAL; j++)
      {
        if (iNext == iColumn[j])
          iFound = j;
      }

      if (iFound > -1)
        st = mapData(sTemp, COLUMNS[iFound], st);
    }

    return (st);
  }

  /** Method mapData()
   * <p> This method will parse a Tops File row into the regions structure.
   * @param  sData   = Data Row String
   * @param  sColumn = the column idenfier
   * @param  st      = the regions Data Structure
   * @return st      = the regions Data Structure
   */

  private stratStruct mapData(String sData, String sColumn, stratStruct st)
  {
    if (sColumn.equals(TOP))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.depthStart = cmn.cmnString.stringToDouble(sData);
    }
    if (sColumn.equals(BASE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.depthEnd = cmn.cmnString.stringToDouble(sData);
    }

    if (sColumn.equals(RANK))
    {
      String sTemp =  new String(sData.toUpperCase());

      st.iRank = horizon.strat.stratStruct._NONE;
      for (int i=0; i<horizon.strat.stratStruct._TOTAL_RANK; i++)
      {
        if (sTemp.equals(horizon.strat.stratStruct.RANK[i]))
          st.iRank = i;
      }
    }

    if (sColumn.equals(NAME))      { st.sName = new String(sData); }
    if (sColumn.equals(NAME2))     { st.sName2 = new String(sData); }

    if (sColumn.equals(ERA))       { st.sEra = new String(sData); }
    if (sColumn.equals(SYSTEM))    { st.system = new String(sData); }
    if (sColumn.equals(SUBSYSTEM)) { st.subSystem = new String(sData); }
    if (sColumn.equals(SERIES))    { st.series = new String(sData); }
    if (sColumn.equals(SUBSERIES)) { st.subSeries = new String(sData); }
    if (sColumn.equals(STAGE))     { st.stage = new String(sData); }
    if (sColumn.equals(GROUP))     { st.sGroup = new String(sData); }
    if (sColumn.equals(SUBGROUP))  { st.subGroup = new String(sData); }
    if (sColumn.equals(FORMATION)) { st.sFormation = new String(sData); }

    if (sColumn.equals(AGE_START))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dAgeStart = cmn.cmnString.stringToDouble(sData);
    }

    if (sColumn.equals(AGE_END))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dAgeEnd = cmn.cmnString.stringToDouble(sData);
    }

    return (st);
  }

  /* ====================================================================== *
   * ----------------------------- GET METHODS ---------------------------- *
   * ====================================================================== */

  /** Method getLine1()
   * <p> This method will return the 1st Line of the delimited file
   * @return sLine1 = 1st line of delimited file
   */

  public String getLine1() { return (sLine1); }

  /** Method getLine2()
   * <p> This method will return the 2nd Line of the delimited file
   * @return sLine2 = 2nd line of delimited file
   */

  public String getLine2() { return (sLine2); }

  /** Method getTops()
   * <p> This method will return tops list
   * @return stList = regions data list structure
   */

  public stratListStruct getTops() { return (stList); }

  /* ====================================================================== *
   * -------------------------- GET ERROR METHODS ------------------------- *
   * ====================================================================== */

  /** METHOD getErrorID()
   *  <p> This method will return the error number.
   *  @return iError = Error Indicator
   */

  public int getErrorID() { return (iError); }

  /** METHOD getError()
   *  <p> This method will return the error string.
   *  @return sError = Error String
   */

  public String getError() { return (sError); }
}