/*
 * @horizonEnterDataFrame.java Version 1.1 06/04/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;
import iqstrat.iqstratHeadersStruct;

import horizon.seq.seqListStruct;
import horizon.seq.seqStruct;
import horizon.seq.gui.seqPanel;

import horizon.strat.gui.stratDataEntryPanel;
import horizon.strat.stratStruct;
import horizon.strat.stratListStruct;

import horizon.env.gui.envDataEntryPanel;
import horizon.env.envListStruct;
import horizon.env.envStruct;

/** Class horizonEnterDataFrame
 *  <p> This Class will allow the user to enter the horizon data into the profile
 *
 *  @version 1.1 06/04/2008
 *  @author  John R. Victorine
 */

public class horizonEnterDataFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable    notifier = null;
  private iqstratStruct stStruct = null;

  // Depth range variables

  private double dStart       = 0.0;
  private double dEnd         = 0.0;

  // Data Entry Panels

  private seqPanel            pSequence = null;  // Sequence Stratigraphy Panel
  private stratDataEntryPanel pRegions  = null;  // Stratigraphic Units Panel
  private envDataEntryPanel   pEnv      = null;  // Depostional environment panel

  public static final int _TOPS = 0;
  public static final int _SEQ  = 1;
  public static final int _ENV  = 2;

  private int iPanel = _TOPS;

  // Global Frame Widges

  // Type of Data Buttons

  private JRadioButton rbTops = new JRadioButton();
  private JRadioButton rbSeq  = new JRadioButton();
  private JRadioButton rbEnv  = new JRadioButton();

  // Frame Buttons

  private JButton btnClose     = new JButton();           // Close Frame Button
  private JButton btnHelp      = new JButton();           // Frame Help Button

  // Change List Border

  private TitledBorder titledBorderData  = null;
  private TitledBorder titledBorderEntry = null;

  // Enter Data Panels

  private CardLayout cardLayout = new CardLayout();
  private JPanel     pnlData     = new JPanel();

  // Panels

  // Modify Depth Range

  private JTextField txtStart = new JTextField();  // Starting Depth
  private JTextField txtEnd   = new JTextField();  // Ending Depth

//  JPanel pnlSeq = new JPanel();
  JPanel pnlTops = new JPanel();

  /** Constructor horizonEnterDataFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public horizonEnterDataFrame(Observable notifier, iqstratStruct stStruct)
  {
    try
    {
      this.notifier = notifier;
      this.stStruct = stStruct;

      jbInit();
      addWindowListener( new horizonEnterDataFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop          = new JPanel();
    JPanel pnlCenter       = new JPanel();
    JPanel pnlBase         = new JPanel();
    JPanel pnlDepthRange   = new JPanel();
    JPanel pnlEnd          = new JPanel();
    JPanel pnlStart        = new JPanel();

    ButtonGroup group  = new ButtonGroup();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Starting Depth:");

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Ending Depth:");

    titledBorderEntry = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Data Entry Panel: ");

    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEntry.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Enter Horizon Data:");

    // Type of Data being modified panel

    pnlTop.setLayout(new GridLayout());
    pnlTop.setBorder(BorderFactory.createEtchedBorder());

    rbTops.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTops.setHorizontalAlignment(SwingConstants.CENTER);
    rbTops.setSelected(true);
    rbTops.setText("Stratigraphic Units");
    rbTops.addActionListener(this);

    rbSeq.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSeq.setHorizontalAlignment(SwingConstants.CENTER);
    rbSeq.setSelected(false);
    rbSeq.setText("Sequence Stratigraphy");
    rbSeq.addActionListener(this);

    rbEnv.setFont(new java.awt.Font("Dialog", 1, 11));
    rbEnv.setHorizontalAlignment(SwingConstants.CENTER);
    rbEnv.setSelected(false);
    rbEnv.setText("Depositional Environment");
    rbEnv.addActionListener(this);

    // Frame Button Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    // -- Close Button

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setMaximumSize(new Dimension(71, 25));
    btnClose.setPreferredSize(new Dimension(75, 25));
    btnClose.setText("Cancel");
    btnClose.addActionListener(this);

    // -- Help Button

    btnHelp.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHelp.setPreferredSize(new Dimension(75, 25));
    btnHelp.setText("Help");
    btnHelp.addActionListener(this);

    pnlCenter.setLayout(new BorderLayout());

    // -- Data Entry Panels

    pnlData.setLayout(cardLayout);
    pnlData.setBorder(titledBorderEntry);

    // Depth Range Panel

    pnlDepthRange.setLayout(new GridLayout());

    // -- Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("" + dStart);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new horizonEnterDataFrameFocusAdapter(this));

    // -- Ending Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("" + dEnd);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new horizonEnterDataFrameFocusAdapter(this));

    // Load Data Entry Panels

    pSequence = new seqPanel(notifier);
    pSequence.setStartText(txtStart);
    pSequence.setEndText(txtEnd);

    pRegions = new stratDataEntryPanel(notifier, stStruct);
    pRegions.setDepthTextfields(txtStart, txtEnd);

    pEnv = new envDataEntryPanel(notifier, stStruct);
    pEnv.setStartText(txtStart);
    pEnv.setEndText(txtEnd);

    // Attach Widgets to Frame

    this.getContentPane().add(pnlTop,    BorderLayout.NORTH);
    if (stStruct.bLAS[las3.las3Constants._IQ_SEQ_STRAT])
      pnlTop.add(rbSeq,                  null);
    pnlTop.add(rbTops,                   null);
    if (stStruct.bLAS[las3.las3Constants._IQ_ENV])
      pnlTop.add(rbEnv,                  null);

    group.add(rbSeq);
    group.add(rbTops);
    group.add(rbEnv);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);

    pnlCenter.add(pnlData,               BorderLayout.CENTER);
    pnlData.add(pSequence,               "SEQ");
    pnlData.add(pRegions,                "TOPS");
    pnlData.add(pEnv,                    "ENV");

    pnlCenter.add(pnlDepthRange,         BorderLayout.NORTH);
    pnlDepthRange.add(pnlStart,       null);
    pnlStart.add(txtStart,            BorderLayout.CENTER);

    pnlDepthRange.add(pnlEnd,         null);
    pnlEnd.add(txtEnd,                BorderLayout.CENTER);


    this.getContentPane().add(pnlBase,   BorderLayout.SOUTH);
    pnlBase.add(btnClose,                null);
    pnlBase.add(btnHelp,                 null);

    setBorders();

    // Display the Frame
/*
//    this.setSize(new Dimension(525, 750));
    this.setSize(new Dimension(600, 750));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
//    this.setVisible(false);
*/
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(600, d.height-50));
    this.setLocation((d.width - this.getSize().width) / 2, 10);
//    this.setLocation((d.width - this.getSize().width) / 2,
//                     (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);

  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getSeqList()
   * <p> This method will return the Sequence List Data Structure
   * @return st = Sequence List Data Structure
   */

  public seqListStruct getSeqList() { return ( pSequence.getList() ); }

  /** Method getSelection()
   * <p> This method will get the KGS Standard Stratigraphic Unit Name Choice.
   *

  public void getSelection() {  pRegions.getSelection(); }
*/
  /** Method getTopsData()
   * <p> This method will get an existing Formation Tops List Data Structure
   * @return stTops = Formation Tops List Data Structure
   */

  public stratListStruct getTopsData() { return (pRegions.getListData()); }

  /** Method getStratigraphicUnits()
   * <p> This method will retrieve the stratigraphic unit id and find the
   *     standard unit from the the standard stratigrpahic chart structure.
   *

  public void getStratigraphicUnits() { pRegions.getICSData(); }
*/

  /** Method getEnvList()
   * <p> This method will return the Depositonal Environment List Data Structure
   * @return st = Depositonal Environmen List Data Structure
   */

  public envListStruct getEnvList() { return ( pEnv.getList() ); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setDataType()
   * <p> This method will set the radio buttons to match the selected plot track
   * @param iPanel = The Data entry panel displayed
   */

  public void setDataType(int iPanel)
  {
    this.iPanel = iPanel;

    rbTops.setSelected(false);  // Formation Tops
    rbSeq.setSelected(false);   // Sequence Stratigraphy

    switch (iPanel)
    {
      case _TOPS:
        rbTops.setSelected(true);  // Formation Tops
        cardLayout.show(pnlData, "TOPS");
        break;

      case _SEQ:
        rbSeq.setSelected(true);   // Sequence Stratigraphy
        cardLayout.show(pnlData, "SEQ");
        break;

      case _ENV:
        rbEnv.setSelected(true);   // Depositional Environment
        cardLayout.show(pnlData, "ENV");
        break;
    }

    setBorders();
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    txtStart.setText("" + this.dStart);

    this.dEnd   = depthEnd;
    txtEnd.setText(""   + this.dEnd);

    pSequence.setDepthRange( depthStart, depthEnd );
    pRegions.setDepthRange( depthStart, depthEnd );
    pEnv.setDepthRange( depthStart, depthEnd );
  }

  /** Method setData()
   * <p> This method will load the Well/Outcrop header data structure
   *     into this panel
   * @param st = the Well/Outcrop header data structure
   */

  public void setData(iqstratHeadersStruct st) { pRegions.setData(st); }

  /* --------------------------------------------------------------------- *
   * -------------- Stratigraphic Units Set Methods ---------------------- *
   * --------------------------------------------------------------------- */

  /** Method setTopsData()
   * <p> This method will set an existing Formation Tops List Data Structure
   * @param stTops = Formation Tops List Data Structure
   */

  public void setTopsData(stratListStruct stTops)
  {
    pRegions.setListData(stTops);
  }

  /* --------------------------------------------------------------------- *
   * -------------------- Sequence Set Methods --------------------------- *
   * --------------------------------------------------------------------- */

  /** Method setSeqList()
   * <p> This method will set the Sequence List Data Structure
   * @param st = Sequence List Data Structure
   */

  public void setSeqList(seqListStruct st) { pSequence.setData(st); }

  /* --------------------------------------------------------------------- *
   * ------------------- Environment Set Methods ------------------------- *
   * --------------------------------------------------------------------- */

  /** Method setEnvList()
   * <p> This method will set the Environment List Data Structure
   * @param st = Environment List Data Structure
   */

  public void setEnvList(envListStruct st) { pEnv.setData(st); }

  /* --------------------------------------------------------------------- *
   * ---------------------- Misc Set Methods ----------------------------- *
   * --------------------------------------------------------------------- */

  /** Method setBorders()
   * <p> This method will set the borders of the List and Data Entry Panel
   */

  private void setBorders()
  {
    switch (iPanel)
    {
      case _TOPS:
        titledBorderEntry.setTitle("Data Entry Panel: Formation Tops");
        break;

      case _SEQ:
        titledBorderEntry.setTitle("Data Entry Panel: Sequence Stratigraphy");
        break;

      case _ENV:
        titledBorderEntry.setTitle("Data Entry Panel: Depositional Environment");
        break;
    }

    pnlData.updateUI();
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */


  /** METHOD horizonEnterDataFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class horizonEnterDataFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /** Method close()
    * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    // Input Variables

    notifier = null;
    stStruct = null;

    // Data Structures


    // Data Entry Panels

    if (pSequence != null)
      pSequence.close();
    pSequence = null;

    if (pEnv != null)
      pEnv.close();
    pEnv = null;

    // Type of Data Buttons

    rbTops   = null;
    rbSeq    = null;
    rbEnv    = null;

    // Frame Buttons

    btnClose = null;           // Close Frame Button
    btnHelp  = null;           // Frame Help Button

    // Change List Border

    titledBorderData = null;

    // Enter Data Panels

    cardLayout = null;
    pnlData    = null;

    // Panels

    // Modify Depth Range

    txtStart = null;  // Starting Depth
    txtEnd   = null;  // Ending Depth

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == rbTops)
    {
      iPanel = _TOPS;
      cardLayout.show(pnlData, "TOPS");
    }

    if (event.getSource() == rbSeq)
    {
      iPanel = _SEQ;
      cardLayout.show(pnlData, "SEQ");
    }

    if (event.getSource() == rbEnv)
    {
      iPanel = _ENV;
      cardLayout.show(pnlData, "ENV");
    }

    // Frame Buttons

    if (event.getSource() == btnHelp)    // Frame Help Button
    {
      switch (iPanel)
      {
        case _TOPS:
          util.BrowserControl.displayURL( cmn.cmnStruct.TOPS );
          break;
        case _SEQ:
          util.BrowserControl.displayURL( cmn.cmnStruct.SEQ_STRAT );
          break;
        case _ENV:
//          util.BrowserControl.displayURL( cmn.cmnStruct.ENV );
          break;
      }
    }

    setBorders();

    if (event.getSource() == btnClose) { close(); } //this.setVisible(false); }
  }


  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
//        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

//        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());

        setDepthRange( dStart, dEnd );
      }
    }
  }
}

/** CLASS horizonEnterDataFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class horizonEnterDataFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  horizonEnterDataFrame adaptee;

  horizonEnterDataFrameFocusAdapter(horizonEnterDataFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 06/04/2008
 *  @author  John R. Victorine
 */
