/*
 * @envPlotTrack.java Version 1.1 11/07/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.env.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.net.*;
import javax.swing.*;

import horizon.env.envListStruct;
import horizon.env.envStruct;

/** CLASS envPlotTrack
 *  <p> This Class will create the Depositional Environment Plot Track.
 *
 *  @version 1.1 11/07/2011
 *  @author  John R. Victorine
 */

public class envPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private envListStruct stDepEnv  = null; // Dep Environment List Data Structure

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR envPlotTrack( int iDataType,
   *                               double depthStart,
   *                               double depthEnd,
   *                               int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   */

  public envPlotTrack( int    iDataType,
                       double depthStart,
                       double depthEnd,
                       int    iScale )
  {
    this.iDataType = iDataType;

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close() { stDepEnv  = null; }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getFossilData()
   * <p> This method will get an existing Fossil List Data Structure
   * @return stDepEnv = Fossil List Data Structure
   */

  public envListStruct getFossilData() { return (stDepEnv); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setData( envListStruct stDepEnv )
   * <p> This method will set an existing Depositional Environment
   *     List Data Structure
   * @param stDepEnv = Depositional Environment List Data Structure
   */

  public void setData( envListStruct stDepEnv ) { this.stDepEnv = stDepEnv; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawTextureGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the LAS Texture Grid
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
	int    i,j;
    int    iY1        = 0;
    int    iY2        = 0;
    int    jXinc      = 0;
    int    iIncrement = iWidthTrack/14;
    int    iEndTrack  = iStartTrack+iWidthTrack;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawString("Environment", iStartTrack + iIncrement, LABELSTART+10);

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(new Color( 165, 42, 42 ));

    Graphics2D g2 = (Graphics2D) g;
    g2.translate(iStartTrack+3, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Continental", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+3), -1.0*(LABELSTART+20));

    g.drawLine( iStartTrack+1*iIncrement, LABELSTART+20,
                iStartTrack+1*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

//    g.drawLine( iStartTrack+1*iIncrement, PLOT_TITLES-5,
//                iStartTrack+1*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.setColor(new Color( 160, 82, 45 ));

    g2.translate(iStartTrack+iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Coastal", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+iIncrement+2), -1.0*(LABELSTART+20));

    g.setColor(new Color( 128, 128, 255 ));

    g2.translate(iStartTrack+2*iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Fresh", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+2*iIncrement+2), -1.0*(LABELSTART+20));

    g.setColor(new Color( 102, 205, 170 ));

    g2.translate(iStartTrack+3*iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Brackish", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+3*iIncrement+2), -1.0*(LABELSTART+20));

    g.setColor(new Color( 102, 205, 170 ));

    g2.translate(iStartTrack+4*iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Hypersaline", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+4*iIncrement+2), -1.0*(LABELSTART+20));

    g.setColor(new Color( 0, 0, 180 ));

    g2.drawString("Neritic", iStartTrack+6*iIncrement, LABELSTART+28);

    g.drawLine( iStartTrack+5*iIncrement, LABELSTART+20,
                iStartTrack+5*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.setColor(Color.gray);

    g2.translate(iStartTrack+5*iIncrement+2, LABELSTART+30);
    g2.rotate( Math.PI / 2.0);
    g2.drawString(" 0-2 m", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+5*iIncrement+2), -1.0*(LABELSTART+30));

    g2.translate(iStartTrack+6*iIncrement+2, LABELSTART+30);
    g2.rotate( Math.PI / 2.0);
    g2.drawString(" 2-10 m", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+6*iIncrement+2), -1.0*(LABELSTART+30));

    g2.translate(iStartTrack+7*iIncrement+2, LABELSTART+30);
    g2.rotate( Math.PI / 2.0);
    g2.drawString(" 10-30 m", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+7*iIncrement+2), -1.0*(LABELSTART+30));

    g.setColor(new Color( 0, 0, 180 ));

    g2.drawString("M", iStartTrack+8*iIncrement+2,  PLOT_TITLES-7);
    g2.drawString("O", iStartTrack+9*iIncrement+2,  PLOT_TITLES-7);

    g.setColor(new Color( 0, 0, 130 ));

    g2.translate(iStartTrack+10*iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Bathyal", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+10*iIncrement+2), -1.0*(LABELSTART+20));

    g2.drawString("U", iStartTrack+10*iIncrement+2, PLOT_TITLES-7);
    g2.drawString("M", iStartTrack+11*iIncrement,   PLOT_TITLES-7);
    g2.drawString("L", iStartTrack+12*iIncrement+2, PLOT_TITLES-7);

    g.drawLine( iStartTrack+10*iIncrement, LABELSTART+20,
                iStartTrack+10*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.setColor(new Color( 0, 0, 100 ));

    g2.translate(iStartTrack+13*iIncrement+2, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Abyssal", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+13*iIncrement+2), -1.0*(LABELSTART+20));

    g.drawLine( iStartTrack+13*iIncrement, LABELSTART+20,
                iStartTrack+13*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.setColor(new Color( 160, 82, 45 ));
    g.fillRect( iStartTrack+0*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(new Color( 210, 180, 140 ));
    g.fillRect( iStartTrack+1*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 124, 252, 0 ));
    g.fillRect( iStartTrack+2*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 102, 205, 170 ));
    g.fillRect( iStartTrack+3*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 102, 205, 170 ));
    g.fillRect( iStartTrack+4*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(new Color( 127, 255, 212 ));
    g.fillRect( iStartTrack+5*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 0, 255, 255 ));
    g.fillRect( iStartTrack+6*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 135, 206, 235 ));
    g.fillRect( iStartTrack+7*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(new Color( 0, 191, 255 ));
    g.fillRect( iStartTrack+8*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 0, 178, 238 ));
    g.fillRect( iStartTrack+9*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(new Color( 128, 128, 255 ));
    g.fillRect( iStartTrack+10*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 65, 128, 255 ));
    g.fillRect( iStartTrack+11*iIncrement, PLOT_TITLES-5, 10, 5 );
    g.setColor(new Color( 0, 0, 255 ));
    g.fillRect( iStartTrack+12*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(new Color( 0, 0, 180 ));
    g.fillRect( iStartTrack+13*iIncrement, PLOT_TITLES-5, 10, 5 );

    g.setColor(Color.black);

//    g2.drawString("Shelf", iStartTrack+5*iIncrement,     PLOT_TITLES); //LABELSTART+50);
//    g2.drawString("Slope", iStartTrack+19*iIncrement/2,  PLOT_TITLES); //LABELSTART+50);
//    g2.drawString("Basin", iStartTrack+25*iIncrement/2,  PLOT_TITLES); //LABELSTART+50);

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawData( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the data
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawData( Graphics g, int iStartTrack, int iWidthTrack )
  {
	int    i,j;
    int    iDepth1   = 0;
    int    iDepth2   = 0;
    int    iDepth    = 0;
    int    iX1       = 0;
    int    iX        = 0;
    double dRatio    = 0.0;
    double depth1    = 0.0;
    double depth2    = 0.0;
    int    iRed      = 0;
    int    iGreen    = 0;
    int    iBlue     = 0;
    int    itmp[]    = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int    icont     = 0;
    int    igray     = 0;

	if (stDepEnv != null)
	{
      for (i=0; i<stDepEnv.iCount; i++)
      {
		for (j=0; j<itmp.length; j++)
		  itmp[j] = 0;

		itmp[0]  = stDepEnv.stItem[i].icontl;    // Continental
		itmp[1]  = stDepEnv.stItem[i].icstl;     // Coastal
		itmp[2]  = stDepEnv.stItem[i].ifrsh;     // Fresh Water
		itmp[3]  = stDepEnv.stItem[i].ibrcksh;   // Brackish Water
		itmp[4]  = stDepEnv.stItem[i].ihyper;    // Hyper saline Water
		itmp[5]  = stDepEnv.stItem[i].ishore;    // Shoreface
		itmp[6]  = stDepEnv.stItem[i].ioff_tran; // Off shore trasition
		itmp[7]  = stDepEnv.stItem[i].ioffshr;   // Off Shore
		itmp[8]  = stDepEnv.stItem[i].ishlf_m;   // Middle Neritic
		itmp[9]  = stDepEnv.stItem[i].ishlf_o;   // Outer Neritic
		itmp[10] = stDepEnv.stItem[i].ibthyl_u;  // Upper Bathyal
		itmp[11] = stDepEnv.stItem[i].ibthyl_m;  // Middle Bathyal
		itmp[12] = stDepEnv.stItem[i].ibthyl_l;  // Lower Bathyal
		itmp[13] = stDepEnv.stItem[i].iabyssl;   // Abyssal

        depth1 = stDepEnv.stItem[i].depthStart;
        depth2 = stDepEnv.stItem[i].depthEnd;

        dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        iDepth = Math.abs(iDepth2 - iDepth1);
        if (iDepth == 0) { iDepth = 1; }

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight)      && (iDepth2 <= iHeight))
        {
  		  for (j=0; j<itmp.length; j++)
  		  {
            if (itmp[j] > 4)
            {
              switch(j)
              {
			    case 0: // Continental
                  // "sienna"
                  iRed   = 160;
                  iGreen =  82;
                  iBlue  =  45;
			      break;
			    case 1: // Coastal
			      // "tan"
                  iRed   = 210;
                  iGreen = 180;
                  iBlue  = 140;
			      break;
			    case 2: // Fresh Water
			      // "middle cyan"
                  iRed   = 124;
                  iGreen = 252;
                  iBlue  =   0;
			      break;
			    case 3: // Brackish Water
			      // "aquamarine 3"
                  iRed   = 102;
                  iGreen = 205;
                  iBlue  = 170;
			      break;
			    case 4: // Hyper Saline Water
			      // "aquamarine 3"
                  iRed   = 102;
                  iGreen = 205;
                  iBlue  = 170;
			      break;
			    case 5: // Shoreface
			      // "aquamarine 1"
                  iRed   = 127;
                  iGreen = 255;
                  iBlue  = 212;
			      break;
			    case 6: // offshore transition
			      // "cyan"
                  iRed   =   0;
                  iGreen = 255;
                  iBlue  = 255;
			      break;
			    case 7: // offshore
			      // "sky-blue"
                  iRed   = 135;
                  iGreen = 206;
                  iBlue  = 235;
			      break;
			    case 8: // middle neritic
			      // "deep sky blue 1"
                  iRed   =   0;
                  iGreen = 191;
                  iBlue  = 255;
			      break;
			    case 9: // outer neritic
			      // "deep sky blue 2"
                  iRed   =   0;
                  iGreen = 178;
                  iBlue  = 238;
			      break;
			    case 10: // upper bathyal
			      // "middle blue"
                  iRed   = 128;
                  iGreen = 128;
                  iBlue  = 255;
			      break;
			    case 11: // middle bathyal
			      // "royal blue"
                  iRed   =  65;
                  iGreen = 128;
                  iBlue  = 255;
			      break;
			    case 12: // lower bathyal
			      // "blue"
                  iRed   =   0;
                  iGreen =   0;
                  iBlue  = 255;
			      break;
			    case 13: // abyssal
			      // "dark blue"
                  iRed   =   0;
                  iGreen =   0;
                  iBlue  = 180;
			      break;
			  }

              if (itmp[j] < 5)
              {
				igray  = 255 - 20 * itmp[j];
                iRed   = igray;
                iGreen = igray;
                iBlue  = igray;
			  }

              g.setColor(new Color( iRed, iGreen, iBlue ));

	          iX1 = iStartTrack + j * 10;
	          iX  = 10;

              g.fillRect( iX1, iDepth1, iX, iDepth );
		    }
		  }
		}
	  }
	}
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_TOPS;

    if (iSelected == iqstrat.iqstratTracksStruct._DEP_ENVIRONMENT)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iStartTrack, iWidthTrack );
      drawData( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 11/07/2011
 *  @author  John Victorine
 */
