/*
 * @ReadDepEnvXMLFile.java Version 1.0 01/26/2012
 *
 * Copyright (c) 2012 John R. Victorine
 * All Rights Reserved.
 */

package horizon.env.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import horizon.env.envListStruct;
import horizon.env.envStruct;

/** Class ReadDepEnvXMLFile()
 * <p> This Class will parse the Depositional Environment XML Data Stream
 *     directly to the Depositional Environment List Data Structure.
 *
 *  @version 1.1 01/26/2012
 *  @author  John R. Victorine
 */

public class ReadDepEnvXMLFile extends DefaultHandler
{
  // File Type

  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType  = FILE;

  // XML Element & Attribute Labels

  public static final String ROOTNAME = "dep_environment";  // Root Name of XML
  public static final String RECORDS  = "records";          // ATT: Record Count
  public static final String DATA     = "data";             // ELEMENT data
  public static final String VALUE    = "value";    // String equivalent to Dep Env Plot Track

  public static final String ENV      = "env";      // ELEMENT environment
  public static final String NAME     = "name";     // Level Name, Envrionment, Data Name
  public static final String SUBGROUP = "subgroup"; // Environment Subgroup
  public static final String GROUP    = "group";    // Environment Group
  public static final String TYPE     = "type";     // Environment Type
  public static final String MODEL    = "model";    // Sedimentary Model
  public static final String LEVEL    = "level";    // ELEMENT & Dep Env Plot Track

  public static final String PHRASE   = "phrase";   // ELEMENT phrase
  public static final String ID       = "id";       // Type of Data

  // ELEMENT Identifier

  public static final int _DATA   = 0;  // Starting point of data
  public static final int _LEVEL  = 1;  // Environment Level Element
  public static final int _ENV    = 2;  // Environment Element
  public static final int _PHRASE = 3;  // Phrase to determine Environment

  private int iData  = _DATA;

  // Global Variables

//  private int           iRows  =  0;
//  private int           iCount = -1;
  private int           iError = 0;
  private String        sError = "";
  private String        sValue = "";
  private String        sID    = "";
  private envListStruct stList = new envListStruct();
  private envStruct     st     = null;

  /** Method ReadDepEnvXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadDepEnvXMLFile() { iType = FILE; }

  /** Method ReadDepEnvXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadDepEnvXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename  = The File to be opened.
   *  @return stList    = The XML List Data Structure (Array List).
   */

  public envListStruct Process( String filename )
  {
//    stList.iCount = 0;
    Read(filename);
//    stList.iCount = iCount+1;

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String    eName = lName; // element name
    String    aName = "";    // attribute name
    String    sTemp = "";    // attribute value
    String    sOut  = "";    // attribute value striped of excess white spaces

    if ("".equals(eName))
      eName = qName; // namespaceAware = false


    if (eName.equals(LEVEL))    // ELEMENT & Dep Env Plot Track
    {
	  iData = _LEVEL;
	}

    if (eName.equals(ENV))      // ELEMENT environment
    {
	  iData = _ENV;
	  st = new envStruct();
	}

    if (eName.equals(PHRASE))   // ELEMENT phrase
    {
	  iData  = _PHRASE;
      sValue = new String( "" );
      sID    = new String( "" );
	}


    if (attrs != null)
    {
      for (int i=0; i<attrs.getLength(); i++)
      {
        aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        sTemp = new String(attrs.getValue(i));
        sOut  = removeExcess(sTemp);

        switch (iData)
        {
          case _DATA:    // Starting point of data
          case _LEVEL:   // Environment Level Element
		    break;
          case _ENV:     // Environment Element
            st = ParseData( aName, sOut.trim(), st );
		    break;
          case _PHRASE:  // Phrase to determine Environment
            ParsePhraseData( aName, sOut.trim() );
		    break;
		}
      }
    }  // end attribute section
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    String eName = qName; // element name
    int    id    = -1;
    int    in    = 0;

    if (eName.equals(ENV))      // ELEMENT environment
    {
	  if (st != null)
	  {
		stList = horizon.env.envUtility.add( st, stList );
		st.delete();
		st = null;
	  }
	}

    if (eName.equals(PHRASE))   // ELEMENT phrase
    {
	  if (st != null)
	  {
  	    if (sID.equals("Lt"))
	      id = horizon.env.envStruct._Lt;
	    else if (sID.equals("L"))
	      id = horizon.env.envStruct._L;
	    else if (sID.equals("C"))
	      id = horizon.env.envStruct._C;
	    else if (sID.equals("F"))
	      id = horizon.env.envStruct._F;
	    else if (sID.equals("Gt"))
	      id = horizon.env.envStruct._Gt;
	    else if (sID.equals("G"))
	      id = horizon.env.envStruct._G;

	    if (id != -1)
	    {
		  if (st.iData[id] == -1)
            st.iData[id] = 1;
          else
            st.iData[id]++;

	      switch (id)
	      {
            case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
              if (st.sLith_T != null)
                in = st.sLith_T.length;
              break;
            case horizon.env.envStruct._L:  // Total number of Lithologies found
              if (st.sLith != null)
                in = st.sLith.length;
              break;
            case horizon.env.envStruct._C:  // Total number of Colors found
              if (st.sColor != null)
                in = st.sColor.length;
              break;
            case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
              if (st.sFossil != null)
                in = st.sFossil.length;
              break;
            case horizon.env.envStruct._Gt: // Total number of Unique Genera found
              if (st.sGenus_T != null)
                in = st.sGenus_T.length;
              break;
            case horizon.env.envStruct._G:  // Total number of Genera found
              if (st.sGenus != null)
                in = st.sGenus.length;
              break;
          }

	      st = horizon.env.envUtility.addDataItem( in, sValue, id, st );
	    }
	  }
	}
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** Method ParseData()
   * <p> This method will parse the Data Stream
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = String
   * @param  st         = The Data Structure
   * @return st         = The Data Structure
   */

  private envStruct ParseData( String sIdentifier,
                               String sData,
                               envStruct st )
  {
    char c[] = null;

    if (sIdentifier.equals(NAME))     { st.sDepEnv = new String(sData); }
    if (sIdentifier.equals(SUBGROUP)) { st.sGroup3 = new String(sData); }
    if (sIdentifier.equals(GROUP))    { st.sGroup2 = new String(sData); }
    if (sIdentifier.equals(TYPE))     { st.sGroup  = new String(sData); }
    if (sIdentifier.equals(MODEL))
    {
	  st.sType  = new String(sData);
	  if (st.sType.toLowerCase().equals("continental"))
	    st.iType = 0;
	  else if (st.sType.toLowerCase().equals("transitional"))
	    st.iType = 1;
	  else if (st.sType.toLowerCase().equals("marine"))
	    st.iType = 2;
	}

    if (sIdentifier.equals(LEVEL))
    {
      c = sData.toCharArray();

      for (int j=0; j<horizon.env.envStruct._MAXIMUM; j++)
      {
		if (c[j] == 'X')
		{
		  switch (j)
		  {
            case horizon.env.envStruct._CONTINENTAL:  // Continental
              st.icontl = 5;   // Continental
              break;
                                                     // Costal
            case horizon.env.envStruct._COASTAL:     // -- Upper
              st.icstl = 5;
              break;
            case horizon.env.envStruct._COASTAL_F:   // .. Fresh Water
              st.ifrsh = 5;
              break;
            case horizon.env.envStruct._COASTAL_B:   // .. Brackish Water
              st.ibrcksh = 5;
              break;
            case horizon.env.envStruct._COASTAL_H:   // .. Hyper saline Water
              st.ihyper = 5;
              break;
                                                     // Neritic
                                                     // -- Inner
            case horizon.env.envStruct._NERITIC_I_0: // .. Shoreface (upper and lower)
              st.ishore = 5;
              break;
            case horizon.env.envStruct._NERITIC_I_1: // .. Offshore transition
              st.ioff_tran = 5;
              break;
            case horizon.env.envStruct._NERITIC_I_2: // .. Offshore
              st.ioffshr = 5;
              break;
            case horizon.env.envStruct._NERITIC_M:   // -- Middle
              st.ishlf_m = 5;
              break;
            case horizon.env.envStruct._NERITIC_O:   // -- Outer
              st.ishlf_o = 5;
              break;
                                                     // Bathyal
            case horizon.env.envStruct._BATHYAL_U:   // -- Upper
              st.ibthyl_u = 5;
              break;
            case horizon.env.envStruct._BATHYAL_M:   // -- Middle
              st.ibthyl_m = 5;
              break;
            case horizon.env.envStruct._BATHYAL_L:   // -- Lower
              st.ibthyl_l = 5;
              break;
            case horizon.env.envStruct._ABYSSAL:     // Abyssal
              st.iabyssl = 5;
              break;
	      }
	    }
	  }
	}

    return (st);
  }

  /** Method ParsePhraseData()
   * <p> This method will parse the Data Stream
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = String
   */

  private void ParsePhraseData( String sIdentifier, String sData )
  {
    if (sIdentifier.equals(NAME)) { sValue = new String(sData); }
    if (sIdentifier.equals(ID))   { sID    = new String(sData); }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}

/*
 *  @version 1.1 01/26/2012
 *  @author  John Victorine
 */
