/*
 * @envDataEntryPanel.java Version 1.1 11/15/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.env.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;

import horizon.env.envListStruct;
import horizon.env.envStruct;
import horizon.env.gui.envTable;
import horizon.env.gui.envMatrixTable;

import util.utilBrowser;

/** Class envDataEntryPanel
 *  <p> This Class will allow the user to enter the sequence stratigraphy
 *      data into the profile program
 *
 *  @version 1.1 11/15/2011
 *  @author  John R. Victorine
 */

public class envDataEntryPanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable    notifier = null;
  private iqstratStruct stStruct = null;

  // Global Variables

  // Action Variables

  public static final int _ADD    = 0;
  public static final int _MODIFY = 1;

  private int iAction = _ADD;

  // Data Panel Type

  public static final int    _CONTI  = 0;
  public static final int    _TRANS  = 1;
  public static final int    _MARINE = 2;
  public static final String TYPE[]  = { "Continental", "Transitional", "Marine" };

  private int iPanel = _CONTI;

  private utilBrowser          pBrowser = null;

  // Depth range variables

  private double dStart       = 0.0;
  private double dEnd         = 0.0;

  // Modify Depth Range

  private JTextField txtStart = null;  // Starting Depth
  private JTextField txtEnd   = null;  // Ending Depth

  // Sequence List Table

  private envTable       pTable  = null;
  private envMatrixTable pXTable = null;

  // Sequence Structures

  private envListStruct stList   = null;
  private envStruct     stModify = null;

  // Temporary Edit Variables

  private String sKEY      = "0";
  private String sEnv      = "";  // Environment
  private String sDesc     = "";  // User Comments / Description

  private String sModel    = "";  // Sedimentary Model, i.e. continental, trasitional or marine
  private String sType     = "";  // Type of Environment  (Group 1 Variable)
  private String sGroup    = "";  // Environment Group    (Group 2 Variable)
  private String subgroup  = "";  // Environment Subgroup (Group 3 Variable)

  private int    iLevel[]  = { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };  // Level: 0=white, 1=set color

  // Global Frame Widges
  // -- Global Panel Buttons

  private JButton btnSelect    = new JButton();
  private JButton btnLevel     = new JButton();

  private JButton btnAdd       = new JButton();
  private JButton btnClear     = new JButton();

  private JButton btnModify    = new JButton();
  private JButton btnRemove    = new JButton();
  private JButton btnRemoveAll = new JButton();

  private JButton btnColors[]  = null;

  // Edit Text Variables

  private JTextField txtEnv      = new JTextField();  // Environment
  private JTextArea  txtRemarks  = new JTextArea();   // User Comments / Description

  private JTextField txtModel    = new JTextField();  // Sedimentary Model
  private JTextField txtType     = new JTextField();  // Type of Enviornment
  private JTextField txtGroup    = new JTextField();  // Environment Group
  private JTextField txtSubgroup = new JTextField();  // Environment Subgroup

  // Enter Data Panels

  private JRadioButton rbConti  = new JRadioButton();
  private JRadioButton rbTrans  = new JRadioButton();
  private JRadioButton rbMarine = new JRadioButton();

  private TitledBorder titledBorderEntry = null;
  private CardLayout cardLayout = new CardLayout();
  private JPanel     pnlData    = new JPanel();

  /** Constructor horizonEnterDataFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public envDataEntryPanel(Observable notifier, iqstratStruct stStruct)
  {
    try
    {
      this.notifier = notifier;
      this.stStruct = stStruct;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlEnv          = new JPanel();
    JPanel pnlTop          = new JPanel();
    JPanel pnlMatrix       = new JPanel();
    JPanel pnlMButtons     = new JPanel();
    JPanel pnlSelect       = new JPanel();
    JPanel pnlTrack        = new JPanel();
    JPanel pnlList         = new JPanel();
    JPanel pnlButtons      = new JPanel();
    JPanel pnlListButtons  = new JPanel();

    JPanel pnlECenter      = new JPanel();

    JPanel pnlEnvText      = new JPanel();
    JPanel pnlRemarks      = new JPanel();
    JPanel pnlModel        = new JPanel();
    JPanel pnlType         = new JPanel();
    JPanel pnlGroup        = new JPanel();
    JPanel pnlSubgroup     = new JPanel();

    JScrollPane scrollList  = new JScrollPane();
    JScrollPane scrollXList = new JScrollPane();

    JLabel lblEnv          = new JLabel();
    JLabel lblModel        = new JLabel();
    JLabel lblType         = new JLabel();
    JLabel lblGroup        = new JLabel();
    JLabel lblSubgroup     = new JLabel();

    ButtonGroup group      = new ButtonGroup();

    titledBorderEntry = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Data Entry Panel: ");
    titledBorderEntry.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnv = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Environment:");
    titledBorderEnv.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderRemarks = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Comments:");
    titledBorderRemarks.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderModel = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Model:");
    titledBorderModel.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderType = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Type of Environment:");
    titledBorderType.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderGroup = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Environment Group:");
    titledBorderGroup.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSubgroup = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Environment Subgroup:");
    titledBorderSubgroup.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setLayout(new BorderLayout());

    // Environment Panel

    pnlEnv.setLayout(new BorderLayout());

    // -- Environment Type Panel Buttons

    pnlTop.setLayout(new GridLayout());
//    pnlTop.setBorder(BorderFactory.createEtchedBorder());

    rbConti.setFont(new java.awt.Font("Dialog", 1, 11));
    rbConti.setHorizontalAlignment(SwingConstants.CENTER);
    rbConti.setSelected(true);
    rbConti.setText("Continental");
    rbConti.addActionListener(this);

    rbTrans.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTrans.setHorizontalAlignment(SwingConstants.CENTER);
    rbTrans.setText("Transitional");
    rbTrans.addActionListener(this);

    rbMarine.setFont(new java.awt.Font("Dialog", 1, 11));
    rbMarine.setHorizontalAlignment(SwingConstants.CENTER);
    rbMarine.setText("Marine");
    rbMarine.addActionListener(this);

    // -- Environment Type Panels

    pnlData.setLayout(new BorderLayout());
    pnlData.setBorder(titledBorderEntry);

    // .. Environment Matrix Table

    pXTable     = new envMatrixTable();
    scrollXList = pXTable.getScrollPane();

    // .. Environment Matrix Buttons

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setText("Select");
    btnSelect.addActionListener(this);

    // .. Environment Level & Color Buttons

    btnLevel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnLevel.setText("Level & Colors");
    btnLevel.addActionListener(this);

    // -- Selected Data Panel

    pnlSelect.setLayout(new BorderLayout());

    // .. Environement Panel

//    pnlEnvText.setBorder(titledBorderEnv);
    pnlEnvText.setLayout(new BorderLayout());

    lblEnv.setText("Environment:           ");
    lblEnv.setFont(new java.awt.Font("Monospaced", 1, 11));

    txtEnv.setText("");
    txtEnv.addFocusListener(new envDataEntryPanelFocusAdapter(this));

    // .. Remarks/Comments/Notes Panel

    pnlRemarks.setBorder(titledBorderRemarks);
    pnlRemarks.setLayout(new BorderLayout());
    pnlRemarks.setPreferredSize(new Dimension(10, 100));

    txtRemarks.setText("");
    txtRemarks.setColumns(3);
    txtRemarks.setLineWrap(true);
    txtRemarks.setWrapStyleWord(true);

    // .. Center E Panel

    pnlECenter.setLayout(new GridLayout(5,1));

    // .... Model Panel

//    pnlModel.setBorder(titledBorderModel);
    pnlModel.setLayout(new BorderLayout());

    lblModel.setText("Sedimentary Model:     ");
    lblModel.setFont(new java.awt.Font("Monospaced", 1, 11));

    txtModel.setText("");

    // .... Type Panel

//    pnlType.setBorder(titledBorderType);
    pnlType.setLayout(new BorderLayout());

    lblType.setText("Type of Environement:  ");
    lblType.setFont(new java.awt.Font("Monospaced", 1, 11));

    txtType.setText("");

    // .... Group Panel

//    pnlGroup.setBorder(titledBorderGroup);
    pnlGroup.setLayout(new BorderLayout());

    lblGroup.setText("Environement Group:    ");
    lblGroup.setFont(new java.awt.Font("Monospaced", 1, 11));

    txtGroup.setText("");

    // .... Subgroup Panel

//    pnlSubgroup.setBorder(titledBorderSubgroup);
    pnlSubgroup.setLayout(new BorderLayout());

    lblSubgroup.setText("Environement Subgroup: ");
    lblSubgroup.setFont(new java.awt.Font("Monospaced", 1, 11));

    txtSubgroup.setText("");

    // .. Plot Track Color Buttons Panel

    pnlTrack.setLayout(new GridLayout());

    btnColors =  new JButton[14];

    for (int i=0; i<14; i++)
    {
      btnColors[i] = new JButton();
      btnColors[i].setFont(new java.awt.Font("Dialog", 1, 10));
      if (i < 10)
        btnColors[i].setText(""+i);
      else if (i == 10)
      {
        btnColors[i].setText("A");
	  }
      else if (i == 11)
      {
        btnColors[i].setText("B");
        btnColors[i].setForeground( Color.gray );
	  }
      else if (i == 12)
      {
        btnColors[i].setText("C");
        btnColors[i].setForeground( Color.gray );
	  }
      else if (i == 13)
      {
        btnColors[i].setText("D");
        btnColors[i].setForeground( Color.gray );
	  }
      btnColors[i].addActionListener(this);

      pnlTrack.add(btnColors[i], null);
    }

    setColors();

    // Global Buttons Panel

    // -- List Panel

    pnlList.setPreferredSize(new Dimension(10, 250));
    pnlList.setLayout(new BorderLayout());

    // .. Add & Clear Data Buttons Panel

    pnlButtons.setLayout(new GridLayout());

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add Data to List");
    btnAdd.addActionListener(this);

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    // .. Selected Depositional Environment Data Panel

    pTable     = new envTable();
    scrollList = pTable.getScrollPane();

    // .. List Buttons Panel

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setText("Modify");
    btnModify.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    // Attach Widgets to Panel

    this.add(pnlTop,                 BorderLayout.NORTH);
    pnlTop.add(rbConti,              null);
    pnlTop.add(rbTrans,              null);
    pnlTop.add(rbMarine,             null);

    group.add(rbConti);
    group.add(rbTrans);
    group.add(rbMarine);

    this.add(pnlEnv,                 BorderLayout.CENTER);
    pnlEnv.add(pnlData,              BorderLayout.CENTER);
    pnlData.add(scrollXList,         BorderLayout.CENTER);
    pnlData.add(pnlMButtons,         BorderLayout.SOUTH);
    pnlMButtons.add(btnSelect,       null);
    pnlMButtons.add(btnLevel,        null);

    pnlEnv.add(pnlSelect,            BorderLayout.SOUTH);

    pnlSelect.add(pnlECenter,        BorderLayout.NORTH);
    pnlECenter.add(pnlEnvText,       null);
    pnlEnvText.add(lblEnv,           BorderLayout.WEST);
    pnlEnvText.add(txtEnv,           BorderLayout.CENTER);

    pnlECenter.add(pnlModel,         null);
    pnlModel.add(lblModel,           BorderLayout.WEST);
    pnlModel.add(txtModel,           BorderLayout.CENTER);

    pnlECenter.add(pnlType,          null);
    pnlType.add(lblType,             BorderLayout.WEST);
    pnlType.add(txtType,             BorderLayout.CENTER);

    pnlECenter.add(pnlGroup,         null);
    pnlGroup.add(lblGroup,           BorderLayout.WEST);
    pnlGroup.add(txtGroup,           BorderLayout.CENTER);

    pnlECenter.add(pnlSubgroup,      null);
    pnlSubgroup.add(lblSubgroup,     BorderLayout.WEST);
    pnlSubgroup.add(txtSubgroup,     BorderLayout.CENTER);

    pnlSelect.add(pnlRemarks,        BorderLayout.CENTER);
    pnlRemarks.add(txtRemarks,       BorderLayout.CENTER);

    pnlSelect.add(pnlTrack,          BorderLayout.SOUTH);

    this.add(pnlList,                BorderLayout.SOUTH);
    pnlList.add(pnlButtons,          BorderLayout.NORTH);
    pnlButtons.add(btnAdd,           null);
    pnlButtons.add(btnClear,         null);

    pnlList.add(scrollList,          BorderLayout.CENTER);

    pnlList.add(pnlListButtons,      BorderLayout.SOUTH);
    pnlListButtons.add(btnModify,    null);
    pnlListButtons.add(btnRemove,    null);
    pnlListButtons.add(btnRemoveAll, null);

    setBorders();
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve the Sequence to be added to the list
   * @return st = Sequence Data Structure
   */

  public envStruct getData()
  {
    envStruct st = new envStruct();

    if (sKEY.equals("0"))
      sKEY = new String(cmn.cmnString.UniqueName());

    st.sKEY = new String( sKEY );

    st.depthStart = dStart;            // Starting Depth.
    if (dEnd <= dStart) dEnd = dStart + 0.5;
    st.depthEnd   = dEnd;              // Ending Depth.
    st.dBed       = (double) ( (int) ((dEnd - dStart) * 100.0)) / 100.0;

    st.icontl     = iLevel[0];   // Continental
    st.icstl      = iLevel[1];   // Coastal
    st.ifrsh      = iLevel[2];   // Fresh Water
    st.ibrcksh    = iLevel[3];   // Brackish Water
    st.ihyper     = iLevel[4];   // Hyper saline Water
    st.ishore     = iLevel[5];   // Shoreface
    st.ioff_tran  = iLevel[6];   // Off shore trasition
    st.ioffshr    = iLevel[7];   // Off Shore
    st.ishlf_m    = iLevel[8];   // Middle Neritic
    st.ishlf_o    = iLevel[9];   // Outer Neritic
    st.ibthyl_u   = iLevel[10];  // Upper Bathyal
    st.ibthyl_m   = iLevel[11];  // Middle Bathyal
    st.ibthyl_l   = iLevel[12];  // Lower Bathyal
    st.iabyssl    = iLevel[13];  // Abyssal

    st.sDepEnv    = new String(sEnv);       // Depositional Environment Phrase
    st.sDescrip   = new String(sDesc);      // Description

    st.iType      = -1;                     // Depositional Environment Type ID
    for (int i=0; i<3; i++)
    {
      if (sModel.equals(TYPE[i]))
        st.iType = i;
    }
    st.sType      = new String( sModel );   // Depositional Environment Type
    st.sGroup     = new String( sType );    // 1st Group: Type
    st.sGroup2    = new String( sGroup );   // 2nd Group:
    st.sGroup3    = new String( subgroup ); // 3rd Group:

    return (st);
  }

  /** Method getList()
   * <p> This method will retrieve the Sequence list data structure
   * @return st = Sequence Data List Structure
   */

  public envListStruct getList() { return (stList); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    this.dEnd   = depthEnd;

    setButtons();
  }

  /** Method setStartText()
   * <p> This method will set the Starting Depth textfield
   * @param txtStart = The Starting Depth Textfield
   */

  public void setStartText(JTextField txtStart) { this.txtStart = txtStart; }

  /** Method setEndText()
   * <p> This method will set the Ending Depth textfield
   * @param txtEnd = The Ending Depth Textfield
   */

  public void setEndText(JTextField txtEnd) { this.txtEnd = txtEnd; }

  /** Method setData()
   * <p> This method will set sequence List Data Structure
   * @param st = the sequence List Data Structure
   */

  public void setData(envListStruct st)
  {
    this.stList = st;
    pTable.setData( st );
  }

  /** Method setAction()
   * <p> This method will set the Action variables
   * @param iAction = The Action to be set
   */

  public void setAction(int iAction)
  {
    this.iAction = iAction;

    switch (iAction)
    {
      case _ADD:
        btnAdd.setText("Add Data to List");
        break;

      case _MODIFY:
        btnAdd.setText("Modify Data in List");
        break;
    }

    setButtons();
  }

  /** Method setButtons()
   * <p> This method will enable or disable buttons depending on data
   */

  private void setButtons()
  {
    btnAdd.setEnabled(true);

    btnModify.setEnabled(false);
    btnRemove.setEnabled(false);
    btnRemoveAll.setEnabled(false);
/*
    if (dStart != dEnd)
    {
      btnAdd.setEnabled(true);
    }
*/
    if (pTable.getTotalRows() > 0)
    {
      btnModify.setEnabled(true);
      btnRemove.setEnabled(true);
      btnRemoveAll.setEnabled(true);
    }
  }

  /** Method setBorders()
   * <p> This method will set the borders of the List and Data Entry Panel
   */

  private void setBorders()
  {
	pXTable.setData(iPanel);

    switch (iPanel)
    {
      case _CONTI:
        titledBorderEntry.setTitle("Continental Environment Types");
        break;

      case _TRANS:
        titledBorderEntry.setTitle("Transitional Environment Types");
        break;

      case _MARINE:
        titledBorderEntry.setTitle("Marine Environment Types");
        break;
    }

    pnlData.updateUI();
  }

  /** Method setColors()
   * <p> This methos will set the enviornment level button colors
   */

  private void setColors()
  {
    for (int i=0; i<14; i++)
    {
      btnColors[i].setBackground(new Color( 255, 255, 255 ));

	  if (iLevel[i] > 0)
	  {
        switch (i)
        {
	  	  case 0: // Continental
            btnColors[i].setBackground(new Color( 160, 82, 45 ));
            break;
		  case 1: // Costal
            btnColors[i].setBackground(new Color( 210, 180, 140 ));
            break;
		  case 2: // Fresh Water
            btnColors[i].setBackground(new Color( 124, 252, 0 ));
            break;
		  case 3: // Brackish Water
            btnColors[i].setBackground(new Color( 102, 205, 170 ));
            break;
		  case 4: // Hyper saline Water
            btnColors[i].setBackground(new Color( 102, 205, 170 ));
            break;
		  case 5: // Shoreface           (  0- 2 m )
            btnColors[i].setBackground(new Color( 127, 255, 212 ));
            break;
		  case 6: // Offshore transition (  2-10 m )
            btnColors[i].setBackground(new Color( 0, 255, 255 ));
            break;
		  case 7: // Offshore            ( 10-30 m )
            btnColors[i].setBackground(new Color( 135, 206, 235 ));
            break;
		  case 8: // Middle Neritic
            btnColors[i].setBackground(new Color( 0, 191, 255 ));
            break;
		  case 9: // Outher Neritic
            btnColors[i].setBackground(new Color( 0, 178, 238 ));
            break;
		  case 10: // Upper Bathyal
            btnColors[i].setBackground(new Color( 128, 128, 255 ));
            break;
		  case 11: // Middle Bathyal
            btnColors[i].setBackground(new Color( 65, 128, 255 ));
            break;
		  case 12: // Lower Bathyal
            btnColors[i].setBackground(new Color( 0, 0, 255 ));
            break;
		  case 13: // Abyssal
            btnColors[i].setBackground(new Color( 0, 0, 180 ));
            break;
        }
      }
    }
  }

  /** Method setXSelect()
   *  <p> This method will set the selected table value in the Text edit fields
   */

  private void setXSelect()
  {
	int    iXtable = -1;
    String sLevel  = "";
    char   c[]     = null;

	if (pXTable != null)
	{
	  iXtable = pXTable.getRow();
	  setWidgets( iXtable );
	}
  }

  /** Method setWidgets()
   * <p> This method will set the enviornment data to the Select Panel
   * @param iXtable is the row data to load into the text edit fields
   */

  private void setWidgets(int iXtable)
  {
    String sLevel  = "";
    char   c[]     = null;

	if (iXtable > -1)
	{
      sModel   = new String( horizon.env.envMatrixStruct.ENV[iXtable][0] );
      sType    = new String( horizon.env.envMatrixStruct.ENV[iXtable][1] );
      sGroup   = new String( horizon.env.envMatrixStruct.ENV[iXtable][2] );
      subgroup = new String( horizon.env.envMatrixStruct.ENV[iXtable][3] );

	  sEnv     = new String( horizon.env.envMatrixStruct.ENV[iXtable][4] );
      sLevel   = new String( horizon.env.envMatrixStruct.ENV[iXtable][5] );

      c = sLevel.toCharArray();

      for (int i=0; i<14; i++)
      {
		iLevel[i] = 0;
		if (c[i] == 'X')
		  iLevel[i] = 5;
	  }

	  if (sEnv.length() == 0)
	  {
	    if (sModel.length() > 0)
          sEnv = new String( sModel );

	    if (sType.length() > 0)
          sEnv = new String( sType );

	    if (sGroup.length() > 0)
          sEnv = new String( sGroup );

	    if (subgroup.length() > 0)
          sEnv = new String( subgroup );
	  }

      txtEnv.setText( sEnv );           // Environment
      txtModel.setText( sModel );       // Sedimentary Model
      txtType.setText( sType );         // Type of Enviornment
      txtGroup.setText( sGroup );       // Environment Group
      txtSubgroup.setText( subgroup );  // Environment Subgroup

	  setColors();
	}
  }

  /** Method setWidgets()
   * <p> This method will set the enviornment data to the Select Panel
   * @param st depositional environment data structure
   */

  private void setWidgets(envStruct st)
  {
	if (st != null)
	{
	  sKEY       = new String( st.sKEY );

      iLevel[0]  = st.icontl;    // Continental
      iLevel[1]  = st.icstl;     // Coastal
      iLevel[2]  = st.ifrsh;     // Fresh Water
      iLevel[3]  = st.ibrcksh;   // Brackish Water
      iLevel[4]  = st.ihyper;    // Hyper saline Water
      iLevel[5]  = st.ishore;    // Shoreface
      iLevel[6]  = st.ioff_tran; // Off shore trasition
      iLevel[7]  = st.ioffshr;   // Off Shore
      iLevel[8]  = st.ishlf_m;   // Middle Neritic
      iLevel[9]  = st.ishlf_o;   // Outer Neritic
      iLevel[10]  = st.ibthyl_u;  // Upper Bathyal
      iLevel[11] = st.ibthyl_m;  // Middle Bathyal
      iLevel[12] = st.ibthyl_l;  // Lower Bathyal
      iLevel[13] = st.iabyssl;   // Abyssal

      // Depositional Environment selected for the depth range

      sModel   = new String( st.sType );
      sType    = new String( st.sGroup );
      sGroup   = new String( st.sGroup2 );
      subgroup = new String( st.sGroup3 );

      sDesc    = new String( st.sDescrip );
	  sEnv     = new String( st.sDepEnv );

      txtEnv.setText( sEnv );           // Environment
      txtModel.setText( sModel );       // Sedimentary Model
      txtType.setText( sType );         // Type of Enviornment
      txtGroup.setText( sGroup );       // Environment Group
      txtSubgroup.setText( subgroup );  // Environment Subgroup
      txtRemarks.setText( sDesc );      // Description

	  setColors();
	}
  }

  /** Method clear()
   * <p> This method will clear the Rock Color color to none
   */

  public void clear()
  {
	sKEY     = new String( "0" );
    sModel   = new String( "" );
    sType    = new String( "" );
    sGroup   = new String( "" );
    subgroup = new String( "" );

    sDesc    = new String( "" );
    sEnv     = new String( "" );

    for (int i=0; i<14; i++)
    {
	  iLevel[i] = 1;
    }

    txtEnv.setText( sEnv );           // Environment
    txtModel.setText( sModel );       // Sedimentary Model
    txtType.setText( sType );         // Type of Enviornment
    txtGroup.setText( sGroup );       // Environment Group
    txtSubgroup.setText( subgroup );  // Environment Subgroup
    txtRemarks.setText( sDesc );      // Description

    setColors();

    setAction( _ADD );

    if (stModify != null)
      stModify.delete();
    stModify = null;
  }

  /* ===================================================================== *
   * -------------------------- DATA ACTIONS ----------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add the selection and depth to the depositional
   *     environment List and plot it on the profile plot in the
   *     depositional environment track.
   */

  private void add()
  {
    envStruct st = getData();

    if (iAction == _MODIFY)
      stList = horizon.env.envUtility.modify( st.sKEY, st, stList );
    else
      stList = horizon.env.envUtility.add( st, stList);

    stList = horizon.env.envUtility.bubbleSort(stList);

    setData(stList);

    notifier.notifyObservers(new String("Depositional Environment Changed"));

    setAction( _ADD );
    clear();
  }

  /** Method modify()
   * <p> This method will modify the data
   */

  private void modify()
  {
    String sTemp  = "";

    setAction( _MODIFY );

    stModify    = horizon.env.envUtility.copy( pTable.getRowData() );
    this.dStart = stModify.depthStart;
    this.dEnd   = stModify.depthEnd;

    if (txtStart != null)
      txtStart.setText("" + this.dStart);

    if (txtEnd  != null)
      txtEnd.setText("" + this.dEnd);

    setWidgets( stModify );
  }

  /** Method removeItem()
   * <p> This method will remove a record from the list
   */

  private void removeItem()
  {
    envStruct st = horizon.env.envUtility.copy(pTable.getRowData());
    stList       = horizon.env.envUtility.remove( st.sKEY, stList );

    setData(stList);

    notifier.notifyObservers(new String("Depositional Environment Changed"));
  }

  /** Method removeAllRows()
   * <p> This method will remove all records from the list
   */

  private void removeAllRows()
  {
    if (stList != null)
      stList.delete();
    stList = null;

    setData(stList);

    notifier.notifyObservers(new String("Depositional Environment Changed"));
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method displayLevel()
   *  <p> this method will display the legend for the envirnment level & colors
   */

  private void displayLevel()
  {
    if (!stStruct.stCMN.bStandalone)
    {
      pBrowser = new utilBrowser( stStruct.stCMN.appletContext,
                                  cmn.cmnStruct.PROFILE + "environment.html" );
    }
    else
    {
      util.BrowserControl.displayURL( cmn.cmnStruct.PROFILE + "environment.html" );
    }
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == rbConti)  { iPanel = _CONTI; }
    if (event.getSource() == rbTrans)  { iPanel = _TRANS; }
    if (event.getSource() == rbMarine) { iPanel = _MARINE; }

    for (int i=0; i<14; i++)
    {
	  if (event.getSource() == btnColors[i])
	  {
		if (iLevel[i] > 0)
		  iLevel[i] = 0;
		else if (iLevel[i] == 0)
		  iLevel[i] = 5;

		setColors();
	  }
	}

    // Global Panel Buttons

    if (event.getSource() == btnSelect) { setXSelect(); }
    if (event.getSource() == btnLevel)  { displayLevel(); }

    if (event.getSource() == btnAdd)    { add(); }
    if (event.getSource() == btnClear)  { clear(); }

    // Data List Buttons

    if (event.getSource() == btnModify)    { modify(); }        // Modify Row
    if (event.getSource() == btnRemove)    { removeItem(); }    // Remove Row
    if (event.getSource() == btnRemoveAll) { removeAllRows(); } // Remove All Rows

    setButtons();
    setBorders();
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int iFound = 0;

    if (e.getSource() == txtEnv)
    {
	  for (int i=0; i<horizon.env.envMatrixStruct.ENV.length; i++)
	  {
		for (int j=0; j<6; j++)
		{
		  if (iFound == 0)
		  {
		    if ((txtEnv.getText()).toLowerCase().equals(
				  horizon.env.envMatrixStruct.ENV[i][j].toLowerCase()))
  		    {
			  setWidgets(i);
			  iFound = 1;
			}
	      }
		}
      }
    }
  }
}

/** CLASS envDataEntryPanelFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class envDataEntryPanelFocusAdapter extends java.awt.event.FocusAdapter
{
  envDataEntryPanel adaptee;

  envDataEntryPanelFocusAdapter(envDataEntryPanel adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 11/15/2011
 *  @author  John Victorine
 */
