/*
 * @envUtility.java Version 1.1 11/02/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.env;

import horizon.env.envListStruct;
import horizon.env.envStruct;

import iqstrat.iqstratHeadersStruct;

/** Class envUtility
 *  <p> This Class will provide basic utilities for the Depositional Environment
 *      data structures.
 *
 *  @version 1.1 11/02/2011
 *  @author  John R. Victorine
 */

public class envUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Depositional Environment List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Depositional Environment List Data Structure
   * @return st       = The Depositional Environment List Data Structure.
   */

  public static envListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                            envListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one sDepEnv stratigraphic unit list
   *     structure to another
   * @param  stOld = Old Depositional Environment list data structure
   * @return stNew = New Depositional Environment list data structure
   */

  public static envListStruct copyList( envListStruct stOld )
  {
    envListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new envListStruct();

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID);   // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
        stNew.sName  = new String(stOld.sName);  // Name
        stNew.status = new String(stOld.status); // Status

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude;  // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL;   // Ground Level

        // LAS 3 Dependent Variable

        stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
        stNew.source     = new String( stOld.source );    // Source of Data
        stNew.sReference = new String( stOld.sReference );  // sReference

        // Data List

        stNew.iCount = stOld.iCount;
        stNew.stItem = new envStruct[stOld.iCount];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static envListStruct transfer(envListStruct stOld)
  {
    envListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method add()
   * <p> This method will add the Depositional Environment to an existing list
   * @param  stNew = The Depositional Environment Data Structure
   * @param  st    = The Old Depositional Environment List Data Structure
   * @return st    = The new Depositional Environment List Data Structure.
   */

  public static envListStruct add( envStruct stNew, envListStruct st )
  {
    int    i=0;
    int    iRecords   = 0;
    int    iCount     = 0;
    int    iSource    = -1;
    envListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Reference

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new envListStruct();
    stTemp.stItem = new envStruct[iRecords];

    if (st != null)
    {
	  iSource  = st.iSource;

      sKGS       = new String( st.sKGS );        // KGS Saved Data Indicator
      source     = new String( st.source );      // Source of Data
      sReference = new String( st.sReference );  // Reference

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new envStruct();
    stTemp.stItem[iCount] = copy(stNew);
    iCount++;

    stTemp.iCount = iCount;

    st        = new envListStruct();
    st.stItem = new envStruct[iCount];
    st.iCount = iCount;

    st.iSource  = iSource;

    st.sKGS       = new String( sKGS );        // KGS Saved Data Indicator
    st.source     = new String( source );      // Source of Data
    st.sReference = new String( sReference );  // Reference

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Depositional Environment in an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  stNew  = Depositional Environment data structure
   * @param  st     = Depositional Environment data list structure
   * @return st     = New Depositional Environment data list structure
   */

  public static envListStruct modify( String sKEY,
                                      envStruct stNew,
                                      envListStruct st )
  {
    int i=0;
    int    iSource    = -1;
    envListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Reference

    if (st != null)
    {
	  iSource  = st.iSource;

      sKGS       = new String( st.sKGS );        // KGS Saved Data Indicator
      source     = new String( st.source );      // Source of Data
      sReference = new String( st.sReference );  // Reference

      stTemp        = new envListStruct();
      stTemp.stItem = new envStruct[st.iCount];
      stTemp.iCount = st.iCount;

      for (i=0; i<st.iCount; i++)
      {
        if (sKEY.equals(st.stItem[i].sKEY))
          stTemp.stItem[i] = copy(stNew);
        else
          stTemp.stItem[i] = copy(st.stItem[i]);
      }

      st.delete();

      st        = new envListStruct();
      st.stItem = new envStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.iSource  = iSource;

      st.sKGS       = new String( sKGS );        // KGS Saved Data Indicator
      st.source     = new String( source );      // Source of Data
      st.sReference = new String( sReference );  // Reference

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Depositional Environment from an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  st     = Depositional Environment data list structure
   * @return st     = New Depositional Environment data list structure
   */

  public static envListStruct remove( String sKEY, envListStruct st )
  {
    int i=0;
    int    iRecords   = 0;
    int    iCount     = 0;
    int    iFound     = -1;
    int    iSource    = -1;
    envListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Reference

    if (st != null)
    {
      if (st.iCount == 1)
      {
        st.delete();
        st = null;
      }
      else
      {
        iRecords = st.iCount-1;

        stTemp        = new envListStruct();
        stTemp.stItem = new envStruct[iRecords];

        iSource  = st.iSource;

        sKGS       = new String( st.sKGS );        // KGS Saved Data Indicator
        source     = new String( st.source );      // Source of Data
        sReference = new String( st.sReference );  // Reference

        for (i=0; i<st.iCount; i++)
        {
          if ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0")))
          {
            iFound = i;
          }
        }

        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();

        stTemp.iCount = iCount;

        st        = new envListStruct();
        st.stItem = new envStruct[iCount];
        st.iCount = iCount;

        st.iSource  = iSource;

        st.sKGS       = new String( sKGS );        // KGS Saved Data Indicator
        st.source     = new String( source );      // Source of Data
        st.sReference = new String( sReference );  // Reference

        for (i=0; i<stTemp.iCount; i++)
        {
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        stTemp.delete();
      }
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one region structure to another
   * @param  stOld = Old Depositional Environment data structure
   * @return stNew = New Depositional Environment data structure
   */

  public static envStruct copy( envStruct stOld )
  {
	int i=0;
    envStruct stNew = new envStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY);    // User defined KEY of Record

//      stNew.sLabel     = new String(stOld.sLabel);  // Label of Depth Range
      stNew.depthStart = stOld.depthStart;          // Starting Depth.
      stNew.depthEnd   = stOld.depthEnd;            // Ending Depth.
      stNew.dBed       = stOld.dBed;                // Bed Thickness.

//      stNew.iFrom      = stOld.iFrom;               // Beginning Level (NOT USED)
//      stNew.iTo        = stOld.iTo;                 // Ending Level    (NOT USED)

      stNew.icstl      = stOld.icstl;     // Coastal
      stNew.ifrsh      = stOld.ifrsh;     // Fresh Water
      stNew.ibrcksh    = stOld.ibrcksh;   // Brackish Water
      stNew.ihyper     = stOld.ihyper;    // Hyper saline Water
      stNew.ishore     = stOld.ishore;    // Shoreface
      stNew.ioff_tran  = stOld.ioff_tran; // Off shore trasition
      stNew.ioffshr    = stOld.ioffshr;   // Off Shore
      stNew.ishlf_m    = stOld.ishlf_m;   // Middle Neritic
      stNew.ishlf_o    = stOld.ishlf_o;   // Outer Neritic
      stNew.ibthyl_u   = stOld.ibthyl_u;  // Upper Bathyal
      stNew.ibthyl_m   = stOld.ibthyl_m;  // Middle Bathyal
      stNew.ibthyl_l   = stOld.ibthyl_l;  // Lower Bathyal
      stNew.iabyssl    = stOld.iabyssl;   // Abyssal

//      stNew.iRGB = new int[3];            // (NOT USED)
//      for (int i=0; i<3; i++)             //     .
//      {                                   //     .
//		stNew.iRGB[i] = stOld.iRGB[i];    //     .
//	  }                                   //    ___

      stNew.sDepEnv    = new String(stOld.sDepEnv);   // Depositional Environment Phrase
      stNew.sDescrip   = new String(stOld.sDescrip);  // Description

      stNew.iType      = stOld.iType;                 // Depositional Environment Type ID
      stNew.sType      = new String(stOld.sType);     // Depositional Environment Type
      stNew.sGroup     = new String( stOld.sGroup );  // 1st Group: Type
      stNew.sGroup2    = new String( stOld.sGroup2 ); // 2nd Group:
      stNew.sGroup3    = new String( stOld.sGroup3 ); // 3rd Group:

//      stNew.abbrev     = new String(stOld.abbrev);    // Abbreviation (NOT USED)

      /* ------------------------------------------------------------------------- *
       * This Section is used to hold the Parse Depositional Environment List Data
       * ------------------------------------------------------------------------- */

      stNew.iData      = new int[6];
      for ( i=0; i<6; i++)
        stNew.iData[i] = stOld.iData[i];

      stNew.iLith_T    = stOld.iLith_T;  // Total number of Unique Lithologies found
      stNew.iLith      = stOld.iLith;    // Total number of Lithologies found
      stNew.iColor     = stOld.iColor;   // Total number of Colors found
      stNew.iFossil    = stOld.iFossil;  // Total number of Generic Fossils Types found
      stNew.iGenus_T   = stOld.iGenus_T; // Total number of Unique Genera found
      stNew.iGenus     = stOld.iGenus;   // Total number of Genera found

      if (stNew.iLith_T > 0)
      {
        stNew.sLith_T = new String[stNew.iLith_T];
        for (i=0; i<stNew.iLith_T; i++)
          stNew.sLith_T[i] = new String( stOld.sLith_T[i] );
      }

      if (stNew.iLith > 0)
      {
        stNew.sLith = new String[stNew.iLith];
        for (i=0; i<stNew.iLith; i++)
          stNew.sLith[i] = new String( stOld.sLith[i] );
      }

      if (stNew.iColor > 0)
      {
        stNew.sColor = new String[stNew.iColor];
        for (i=0; i<stNew.iColor; i++)
          stNew.sColor[i] = new String( stOld.sColor[i] );
      }

      if (stNew.iFossil > 0)
      {
        stNew.sFossil = new String[stNew.iFossil];
        for (i=0; i<stNew.iFossil; i++)
          stNew.sFossil[i] = new String( stOld.sFossil[i] );
      }

      if (stNew.iGenus_T > 0)
      {
        stNew.sGenus_T = new String[stNew.iGenus_T];
        for (i=0; i<stNew.iGenus_T; i++)
          stNew.sGenus_T[i] = new String( stOld.sGenus_T[i] );
      }

      if (stNew.iGenus > 0)
      {
        stNew.sGenus = new String[stNew.iGenus];
        for (i=0; i<stNew.iGenus; i++)
          stNew.sGenus[i] = new String( stOld.sGenus[i] );
      }
    }

    return (stNew);
  }

  /** Method copyEnv()
   * <p> This method will copy one environment structure to another
   * @param  stOld = Old Depositional Environment data structure
   * @return stNew = New Depositional Environment data structure
   */

  public static envStruct copyEnv( envStruct stOld )
  {
	int i=0;
    envStruct stNew = new envStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY);    // User defined KEY of Record

//      stNew.sLabel     = new String(stOld.sLabel);  // Label of Depth Range
      stNew.depthStart = stOld.depthStart;          // Starting Depth.
      stNew.depthEnd   = stOld.depthEnd;            // Ending Depth.
      stNew.dBed       = stOld.dBed;                // Bed Thickness.

//      stNew.iFrom      = stOld.iFrom;               // Beginning Level (NOT USED)
//      stNew.iTo        = stOld.iTo;                 // Ending Level    (NOT USED)

      stNew.icstl      = stOld.icstl;     // Coastal
      stNew.ifrsh      = stOld.ifrsh;     // Fresh Water
      stNew.ibrcksh    = stOld.ibrcksh;   // Brackish Water
      stNew.ihyper     = stOld.ihyper;    // Hyper saline Water
      stNew.ishore     = stOld.ishore;    // Shoreface
      stNew.ioff_tran  = stOld.ioff_tran; // Off shore trasition
      stNew.ioffshr    = stOld.ioffshr;   // Off Shore
      stNew.ishlf_m    = stOld.ishlf_m;   // Middle Neritic
      stNew.ishlf_o    = stOld.ishlf_o;   // Outer Neritic
      stNew.ibthyl_u   = stOld.ibthyl_u;  // Upper Bathyal
      stNew.ibthyl_m   = stOld.ibthyl_m;  // Middle Bathyal
      stNew.ibthyl_l   = stOld.ibthyl_l;  // Lower Bathyal
      stNew.iabyssl    = stOld.iabyssl;   // Abyssal

//      stNew.iRGB = new int[3];            // (NOT USED)
//      for (int i=0; i<3; i++)             //     .
//      {                                   //     .
//		stNew.iRGB[i] = stOld.iRGB[i];    //     .
//	  }                                   //    ___

      stNew.sDepEnv    = new String(stOld.sDepEnv);   // Depositional Environment Phrase
      stNew.sDescrip   = new String(stOld.sDescrip);  // Description

      stNew.iType      = stOld.iType;                 // Depositional Environment Type ID
      stNew.sType      = new String(stOld.sType);     // Depositional Environment Type
      stNew.sGroup     = new String( stOld.sGroup );  // 1st Group: Type
      stNew.sGroup2    = new String( stOld.sGroup2 ); // 2nd Group:
      stNew.sGroup3    = new String( stOld.sGroup3 ); // 3rd Group:

//      stNew.abbrev     = new String(stOld.abbrev);    // Abbreviation (NOT USED)
    }

    return (stNew);
  }

  /** Method addDataItem()
   *  <p> This method will add a data item to the lookup list
   *  @param  in  = The total data items in the list
   *  @param  sIn = The Data item to be added to list
   *  @param  id  = The ID number to add to correct variable
   *  @param  st  = The environment data structure
   *  @return st  = The environemnt data structure
   */

  public static envStruct addDataItem( int in, String sIn, int id, envStruct st )
  {
	int    i       = 0;
	int    icnt    = 0;
	String sData[] = null;
	String str     = "";

	if (st != null)
	{
	  sData = new String[in + 1];

	  for (i=0; i<in; i++)
	  {
	    switch (id)
	    {
          case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
            str = new String( st.sLith_T[i] );
            break;
          case horizon.env.envStruct._L:  // Total number of Lithologies found
            str = new String( st.sLith[i] );
            break;
          case horizon.env.envStruct._C:  // Total number of Colors found
            str = new String( st.sColor[i] );
            break;
          case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
            str = new String( st.sFossil[i] );
            break;
          case horizon.env.envStruct._Gt: // Total number of Unique Genera found
            str = new String( st.sGenus_T[i] );
            break;
          case horizon.env.envStruct._G:  // Total number of Genera found
            str = new String( st.sGenus[i] );
            break;
		}

	    sData[icnt] = new String(str);
	    icnt++;
      }

	  sData[icnt] = new String( sIn );
	  icnt++;

	  switch (id)
	  {
        case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
          st.sLith_T = new String[icnt];
          st.iLith_T = icnt;
          for (i=0; i<icnt; i++)
            st.sLith_T[i] = new String( sData[i] );
          break;
        case horizon.env.envStruct._L:  // Total number of Lithologies found
          st.sLith = new String[icnt];
          st.iLith = icnt;
          for (i=0; i<icnt; i++)
            st.sLith[i] = new String( sData[i] );
          break;
        case horizon.env.envStruct._C:  // Total number of Colors found
          st.sColor = new String[icnt];
          st.iColor = icnt;
          for (i=0; i<icnt; i++)
            st.sColor[i] = new String( sData[i] );
          break;
        case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
          st.sFossil = new String[icnt];
          st.iFossil = icnt;
          for (i=0; i<icnt; i++)
            st.sFossil[i] = new String( sData[i] );
          break;
        case horizon.env.envStruct._Gt: // Total number of Unique Genera found
          st.sGenus_T = new String[icnt];
          st.iGenus_T = icnt;
          for (i=0; i<icnt; i++)
            st.sGenus_T[i] = new String( sData[i] );
          break;
        case horizon.env.envStruct._G:  // Total number of Genera found
          st.sGenus = new String[icnt];
          st.iGenus = icnt;
          for (i=0; i<icnt; i++)
            st.sGenus[i] = new String( sData[i] );
          break;
      }
	}

	return (st);
  }

  /** Method doesMatch()
   *  <p> This method will add a data item to the lookup list
   *  @param  id     = The ID number to add to correct variable
   *  @param  sIn    = The Text to be checked
   *  @param  st     = The environment data structure
   *  @return bValue = The environemnt data structure
   */

  public static boolean doesMatch( int id, String sIn, envStruct st )
  {
	int i=0;
	boolean bValue = false;

	if (st != null)
	{
	  if (st.iData[id] > 0)
	  {
	    switch (id)
	    {
          case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
//System.out.println(id+" "+sIn+" "+st.iData[id]);
            for (i=0; i<st.iLith_T; i++)
            {
//System.out.println("<"+sIn.toLowerCase()+"><"+st.sLith_T[i].toLowerCase()+">");
	  		  if (sIn.toLowerCase().equals(st.sLith_T[i].toLowerCase()))
	 		    bValue = true;
//System.out.println(bValue);
		    }
            break;
          case horizon.env.envStruct._L:  // Total number of Lithologies found
            for (i=0; i<st.iLith; i++)
            {
			  if (sIn.toLowerCase().equals(st.sLith[i].toLowerCase()))
			    bValue = true;
		    }
            break;
          case horizon.env.envStruct._C:  // Total number of Colors found
            for (i=0; i<st.iColor; i++)
            {
		  	  if (sIn.toLowerCase().equals(st.sColor[i].toLowerCase()))
			    bValue = true;
		    }
            break;
          case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
            for (i=0; i<st.iFossil; i++)
            {
			  if (sIn.toLowerCase().equals(st.sFossil[i].toLowerCase()))
			    bValue = true;
		    }
            break;
          case horizon.env.envStruct._Gt: // Total number of Unique Genera found
            for (i=0; i<st.iGenus_T; i++)
            {
			  if (sIn.toLowerCase().equals(st.sGenus_T[i].toLowerCase()))
			    bValue = true;
		    }
            break;
          case horizon.env.envStruct._G:  // Total number of Genera found
            for (i=0; i<st.iGenus; i++)
            {
			  if (sIn.toLowerCase().equals(st.sGenus[i].toLowerCase()))
			    bValue = true;
		    }
            break;
        }
	  }
	}

	return (bValue);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = Depositional Environment data list structure
   * @return st = sorted Depositional Environment data list structure
   */

  public static envListStruct bubbleSort( envListStruct st )
  {
    boolean   swappedOnPrevRun = true;
    envStruct stTemp           = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print( envListStruct st )
  {
	int i,j, k;
	String sLevel = "";
	String str    = ".";
	int    len    = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
		if (!st.stItem[i].sKEY.equals("0"))
		{
          System.out.print( st.stItem[i].sKEY       +" "+
                            st.stItem[i].depthStart +" "+
                            st.stItem[i].depthEnd   +" ("+
                            st.stItem[i].dBed       +") " );
		}

        sLevel = new String("");

		str = new String(".");
        if (st.stItem[i].icontl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].icstl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ifrsh > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ibrcksh > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ihyper > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ishore > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ioff_tran > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ioffshr > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ishlf_m > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ishlf_o > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ibthyl_u > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ibthyl_m > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].ibthyl_l > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.stItem[i].iabyssl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);


        System.out.print( st.stItem[i].sDepEnv    + " " +  // Environment name
                          st.stItem[i].sGroup3    + " " +  // Environment subgroup
                          st.stItem[i].sGroup2    + " " +  // Environment group
                          st.stItem[i].sGroup     + " " +  // Type of Environment
                          st.stItem[i].sType      + " " +  // Model
                          sLevel);           // Level
        System.out.println("");

        for (k=0; k<6; k++)
        {
          if (st.stItem[i].iData[k] > -1)
          {
			len = 0;
	  		switch (k)
			{
              case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
                len = st.stItem[i].iLith_T;
                break;
              case horizon.env.envStruct._L:  // Total number of Lithologies found
                len = st.stItem[i].iLith;
                break;
              case horizon.env.envStruct._C:  // Total number of Colors found
                len = st.stItem[i].iColor;
                break;
              case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
                len = st.stItem[i].iFossil;
                break;
              case horizon.env.envStruct._Gt: // Total number of Unique Genera found
                len = st.stItem[i].iGenus_T;
                break;
              case horizon.env.envStruct._G:  // Total number of Genera found
                len = st.stItem[i].iGenus;
                break;
            }

		    for (j=0; j<len; j++)
		    {
  	  		  switch (k)
			  {
                case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
                  str = new String( st.stItem[i].sLith_T[j] );
                  break;
                case horizon.env.envStruct._L:  // Total number of Lithologies found
                  str = new String( st.stItem[i].sLith[j] );
                  break;
                case horizon.env.envStruct._C:  // Total number of Colors found
                  str = new String( st.stItem[i].sColor[j] );
                  break;
                case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
                  str = new String( st.stItem[i].sFossil[j] );
                  break;
                case horizon.env.envStruct._Gt: // Total number of Unique Genera found
                  str = new String( st.stItem[i].sGenus_T[j] );
                  break;
                case horizon.env.envStruct._G:  // Total number of Genera found
                  str = new String( st.stItem[i].sGenus[j] );
                  break;
              }

			  if (j == 0)
			  {
			    switch (k)
			    {
                  case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
   		            System.out.print( "-- Lt: " + str );
                    break;
                  case horizon.env.envStruct._L:  // Total number of Lithologies found
   		            System.out.print( "-- L: " + str );
                    break;
                  case horizon.env.envStruct._C:  // Total number of Colors found
   		            System.out.print( "-- C: " + str );
                    break;
                  case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
   		            System.out.print( "-- F: " + str );
                    break;
                  case horizon.env.envStruct._Gt: // Total number of Unique Genera found
   		            System.out.print( "-- Gt: " + str );
                    break;
                  case horizon.env.envStruct._G:  // Total number of Genera found
   		            System.out.print( "-- G: " + str );
                    break;
                }
			  }
			  else
			  {
	            System.out.print( ", " + str );
			  }
  		    }

   	        System.out.println("");
	  	  }
	    }

        System.out.println(" ------------------------------------------------- ");
      }
    }
  }

  public static void print( envStruct st )
  {
	int i,j, k;
	String sLevel = "";
	String str    = ".";
	int    len    = 0;

    if (st != null)
    {
		if (!st.sKEY.equals("0"))
		{
          System.out.print( st.sKEY       +" "+
                            st.depthStart +" "+
                            st.depthEnd   +" ("+
                            st.dBed       +") " );
		}

        sLevel = new String("");

		str = new String(".");
        if (st.icontl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.icstl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ifrsh > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ibrcksh > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ihyper > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ishore > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ioff_tran > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ioffshr > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ishlf_m > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ishlf_o > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ibthyl_u > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ibthyl_m > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.ibthyl_l > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);

		str = new String(".");
        if (st.iabyssl > 0)
          str = new String("X");
        sLevel = new String(sLevel + str);


        System.out.print( st.sDepEnv    + " " +  // Environment name
                          st.sGroup3    + " " +  // Environment subgroup
                          st.sGroup2    + " " +  // Environment group
                          st.sGroup     + " " +  // Type of Environment
                          st.sType      + " " +  // Model
                          sLevel);           // Level
        System.out.println("");

        for (k=0; k<6; k++)
        {
          if (st.iData[k] > -1)
          {
			len = 0;
	  		switch (k)
			{
              case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
                len = st.iLith_T;
                break;
              case horizon.env.envStruct._L:  // Total number of Lithologies found
                len = st.iLith;
                break;
              case horizon.env.envStruct._C:  // Total number of Colors found
                len = st.iColor;
                break;
              case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
                len = st.iFossil;
                break;
              case horizon.env.envStruct._Gt: // Total number of Unique Genera found
                len = st.iGenus_T;
                break;
              case horizon.env.envStruct._G:  // Total number of Genera found
                len = st.iGenus;
                break;
            }

		    for (j=0; j<len; j++)
		    {
  	  		  switch (k)
			  {
                case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
                  str = new String( st.sLith_T[j] );
                  break;
                case horizon.env.envStruct._L:  // Total number of Lithologies found
                  str = new String( st.sLith[j] );
                  break;
                case horizon.env.envStruct._C:  // Total number of Colors found
                  str = new String( st.sColor[j] );
                  break;
                case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
                  str = new String( st.sFossil[j] );
                  break;
                case horizon.env.envStruct._Gt: // Total number of Unique Genera found
                  str = new String( st.sGenus_T[j] );
                  break;
                case horizon.env.envStruct._G:  // Total number of Genera found
                  str = new String( st.sGenus[j] );
                  break;
              }

			  if (j == 0)
			  {
			    switch (k)
			    {
                  case horizon.env.envStruct._Lt: // Total number of Unique Lithologies found
   		            System.out.print( "-- Lt: " + str );
                    break;
                  case horizon.env.envStruct._L:  // Total number of Lithologies found
   		            System.out.print( "-- L: " + str );
                    break;
                  case horizon.env.envStruct._C:  // Total number of Colors found
   		            System.out.print( "-- C: " + str );
                    break;
                  case horizon.env.envStruct._F:  // Total number of Generic Fossils Types found
   		            System.out.print( "-- F: " + str );
                    break;
                  case horizon.env.envStruct._Gt: // Total number of Unique Genera found
   		            System.out.print( "-- Gt: " + str );
                    break;
                  case horizon.env.envStruct._G:  // Total number of Genera found
   		            System.out.print( "-- G: " + str );
                    break;
                }
			  }
			  else
			  {
	            System.out.print( ", " + str );
			  }
  		    }

   	        System.out.println("");
	  	  }
	    }

        System.out.println(" ------------------------------------------------- ");
      }
  }
}

/*
 *  @version 1.1 11/02/2011
 *  @author  John Victorine
 */
