/*
 * @bioStratPlotTrack.java Version 1.1 10/21/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.bio.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.net.*;
import javax.swing.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;

import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;

/** CLASS bioStratPlotTrack
 *  <p> This Class will create the Fossil Track.
 *
 *  @version 1.1 10/21/2011
 *  @author  John R. Victorine
 */

public class bioStratPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private bioStratListStruct    stFossil  = null; // Fossil List Data Structure
  private maskSymbolsListStruct stSymbols = null; // Fossil Symbols Data
  private maskPlotSymbol        plot      = null; // Plot Mask Symbol

  public int    iRows        = 0;     // Total number of depth ranges for species
  public int    iAbundance[] = null;  // Array of Abundance indicators
  public double dSo[]        = null;  // Array of Starting Depths
  public double dEo[]        = null;  // Array if Ending Depths

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int _PLOT_SP = horizon.bio.bioStratUtility._PLOT_SP;

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR bioStratPlotTrack( int iDataType,
   *                               double depthStart,
   *                               double depthEnd,
   *                               int iScale,
   *                               maskSymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param stSymbols  = Fossil Plot Symbols List Data Structure
   */

  public bioStratPlotTrack( int iDataType,
                          double depthStart,
                          double depthEnd,
                          int iScale,
                          maskSymbolsListStruct stSymbols )
  {
    this.iDataType = iDataType;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    iRows      = 0;     // Total number of depth ranges for species
    iAbundance = null;  // Array of Abundance indicators
    dSo        = null;  // Array of Starting Depths
    dEo        = null;  // Array if Ending Depths

    stFossil  = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getFossilData()
   * <p> This method will get an existing Fossil List Data Structure
   * @return stFossil = Fossil List Data Structure
   */

  public bioStratListStruct getFossilData() { return (stFossil); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setFossilData( bioStratListStruct stFossil )
   * <p> This method will set an existing Fossil List Data Structure
   * @param stFossil = Fossil List Data Structure
   */

  public void setFossilData( bioStratListStruct stFossil )
  {
    this.stFossil = stFossil;
    setAverageData();
  }

  /** Method setAverageData()
   *  <p> This method will set the total number of species by depth
   */

  public void setAverageData()
  {
	int i,j,k;
	int iMax = 0;

    iRows      = 0;     // Total number of depth ranges for species
    iAbundance = null;  // Array of Abundance indicators
    dSo        = null;  // Array of Starting Depths
    dEo        = null;  // Array if Ending Depths

	if (stFossil != null)
	{
	  for (i=0; i<stFossil.iCount; i++)
	  {
		for (j=0; j<stFossil.stItem[i].iRows; j++)
		{
		  if (!isDepth(stFossil.stItem[i].depthStart[j], stFossil.stItem[i].depthEnd[j]))
		  {
			addDepth(stFossil.stItem[i].depthStart[j], stFossil.stItem[i].depthEnd[j]);
		  }

		  addSpecies(stFossil.stItem[i].depthStart[j], stFossil.stItem[i].depthEnd[j]);
		}
	  }
	}

    for (k=0; k<iRows; k++)
    {
	  if (iAbundance[k] > iMax)
	  {
		iMax = iAbundance[k];
	  }
    }

    for (k=0; k<iRows; k++)
    {
	  iAbundance[k] = ( iAbundance[k] * 30 ) / iMax;
//System.out.println(k+" "+dSo[k]+" "+dEo[k]+" "+iAbundance[k]);
     }
  }

  /** Method isDepth()
   *  <p> This method will determine if the depth range exists in the list
   *  @param  dStart = starting depth
   *  @param  dEnd   = ending depth
   *  @return bValue = true the depth exists in list, false it does not
   */

  public boolean isDepth(double dStart, double dEnd)
  {
	boolean bValue = false;

	for (int i=0; i<iRows; i++)
	{
	  if ((dSo[i] == dStart) && (dEo[i] == dEnd))
	  {
		bValue = true;
	  }
	}

	return (bValue);
  }

  /** Method addSpecies()
   *  <p> This method will add a Species to the depth range
   *  @param  dStart = starting depth
   *  @param  dEnd   = ending depth
   */

  public void addSpecies(double dStart, double dEnd)
  {
	for (int i=0; i<iRows; i++)
	{
	  if ((dSo[i] == dStart) && (dEo[i] == dEnd))
	  {
		iAbundance[i]++;
	  }
	}
  }

  /** Method addDepth()
   *  <p> This method will add a depth range to the array if it is not present
   *  @param dStart = starting depth
   *  @param dEnd   = ending depth
   */

  public void addDepth(double dStartA, double dEndA)
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iAbund[] = null;  // Array of Abundance indicators
    double dStart[] = null;  // Array of Starting Depths
    double dEnd[]   = null;  // Array if Ending Depths

    iRecords = iRows+1;
    iAbund   = new int[iRecords];
    dStart   = new double[iRecords];
    dEnd     = new double[iRecords];

    for (i=0; i<iRows; i++)
    {
      if (iCount < iRecords)
      {
        iAbund[iCount] = iAbundance[i];
        dStart[iCount] = dSo[i];
	    dEnd[iCount]   = dEo[i];
        iCount++;
      }
    }

	iAbund[iCount] = 0;
	dStart[iCount] = dStartA;
	dEnd[iCount]   = dEndA;
    iCount++;

    iRows       = iCount;
    iAbundance  = new int[iCount];
    dSo         = new double[iCount];
    dEo         = new double[iCount];

    for (i=0; i<iRows; i++)
    {
      iAbundance[i] = iAbund[i];
      dSo[i]        = dStart[i];
	  dEo[i]        = dEnd[i];
    }
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int         iStart    = 0;
    int         i         = 0;
    int         j         = 0;
    int         jXinc     = 0;
    int         iY1       = 0;
    int         iY2       = 0;
    int         iEndTrack = iStartTrack+iWidthTrack;
    Font        fsb       = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm      = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    iStart = iWidthTrack/10;

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.black);

    plot.drawSymbol(
          g, stSymbols.stItem[0].iRows, 0,
          stSymbols.stItem[0].symbol,
          iStartTrack+5, LABELSTART+4 );

    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    Graphics2D g2 = (Graphics2D) g;
//    g2.rotate( (Math.PI / 2.0));
//    g2.drawString("Species", LABELSTART+20, -1*(iStartTrack+5*(iStart-1)+3));
//    g2.rotate( -1.0 * (Math.PI / 2.0));

    g2.translate(iStartTrack+5, LABELSTART+20);
    g2.rotate( Math.PI / 2.0);
    g2.drawString("Species", 0, 0);
    g2.rotate( -1.0*(Math.PI / 2.0));
    g2.translate(-1.0*(iStartTrack+5), -1.0*(LABELSTART+20));
  }

  /** Method drawData( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the species abundance that were selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawData( Graphics g, int iStartTrack, int iWidthTrack )
  {
	int    iFound    = -1;
    int    iDepth    = 0;
    int    iDepth1   = 0;
    int    iDepth2   = 0;
    int    iX        = 0;
    int    iY        = 0;
    double dRatio    = 0.0;
    double depth1    = 0.0;
    double depth2    = 0.0;
    Graphics2D  g2   = (Graphics2D) g;
    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    if (stFossil != null)
    {
      for (int i=0; i<stFossil.iCount; i++)
      {
        iFound = mask.maskSymbolsUtility.getSymbolID(
                    stFossil.stItem[i].sFossilID,
                    mask.maskSymbolsListStruct._SYMBOLS,
                    stSymbols );

        if (iFound > -1)
        {
          plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                           stSymbols.stItem[iFound].symbol,
                           iStartTrack + (i+1) * _PLOT_SP+10, PLOT_TITLES-12 );

          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack + (i+1)*_PLOT_SP+10, PLOT_TITLES,
                     iStartTrack + (i+1)*_PLOT_SP+10, iHeight);
		}

		if (stFossil.stItem[i].iRows > 0)
		{
		  for (int j=0; j<stFossil.stItem[i].iRows; j++)
		  {
            depth1 = stFossil.stItem[i].depthStart[j];
            depth2 = stFossil.stItem[i].depthEnd[j];

            dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;

            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;

            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = Math.abs(iDepth2 - iDepth1);
            if (iDepth == 0) { iDepth = 5; }

            if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                (iDepth1 < iHeight)      && (iDepth2 <= iHeight))
            {
              g.setColor(Color.black);
              g.fillRect( iStartTrack + (i+1)*_PLOT_SP+10 - stFossil.stItem[i].iAbundance[j],
                          iDepth1, 2 * stFossil.stItem[i].iAbundance[j] + 1, iDepth );

              iX = iStartTrack + (i+1)*_PLOT_SP+10 + 8;
              iY = PLOT_TITLES + 5; //  + iHeight / 2;

              g2.translate(iX, iY);
              g2.rotate( Math.PI / 2.0);
              g2.drawString(stFossil.stItem[i].sName, 0, 0);
              g2.rotate( -1.0*(Math.PI / 2.0));
              g2.translate(-1.0*iX, -1.0*iY);
			}
	      }
		}
	  }
    }
  }

  /** Method drawAverage( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the species abundance that were selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawAverage( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iDepth    = 0;
    int    iDepth1   = 0;
    int    iDepth2   = 0;
    double dRatio    = 0.0;
    double depth1    = 0.0;
    double depth2    = 0.0;
    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g.setColor(Color.black);
    g.drawString("Relative", iStartTrack+3, PLOT_TITLES-12);
    g.drawString("Totals",   iStartTrack+5, PLOT_TITLES);
    g.setColor(Color.lightGray);
    g.drawLine(iStartTrack + 15, PLOT_TITLES, iStartTrack + 15, iHeight);

    for (int j=0; j<iRows; j++)
	{
      depth1 = dSo[j];
      depth2 = dEo[j];

      dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
      iDepth1 = PLOT_TITLES + (int) dRatio;

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iDepth1 = iHeight - (int) dRatio;

      dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iDepth2 = iHeight - (int) dRatio;

      iDepth = Math.abs(iDepth2 - iDepth1);
      if (iDepth == 0) { iDepth = 5; }

      if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
          (iDepth1 < iHeight)      && (iDepth2 <= iHeight))
      {
        g.setColor(Color.black);
        g.fillRect( iStartTrack + 15 - iAbundance[j]/2, iDepth1,
                    iAbundance[j], iDepth );
      }
	}
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Porosity Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_BIO; //_SRC_ROCK;

    if (iSelected == iqstrat.iqstratTracksStruct._BIO_STRATIGRAPHY)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iStartTrack, iWidthTrack );
      drawAverage( g, iStartTrack, iWidthTrack );
      drawData( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 10/21/2011
 *  @author  John Victorine
 */
