/*
 * @ReadBioStratUnitsXMLFile.java Version 1.0 11/05/2009
 *
 * Copyright (c) 2009 John R. Victorine
 * All Rights Reserved.
 */

package horizon.bio.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;
import horizon.strat.stratListStruct;
import parse.parseFossilListStruct;

/** Class ReadBioStratUnitsXMLFile()
 * <p> This Class will parse the Regions XML Data Stream
 *     directly to the Regions List Data Structure.
 *
 *  @version 1.1 11/05/2009
 *  @author  John R. Victorine
 */

public class ReadBioStratUnitsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  public static final String ROOTNAME     = "fossils";     // Root Name of XML
  public static final String RECORDS      = "records";     // Record Count

  public static final String FOSSIL       = "fossil";      // ELEMENT age

  public static final String KEY          = "key";         // Primary Key
  public static final String ID           = "id";          // Unique ID
  public static final String SPECIES      = "species";     // Species
  public static final String GENUS        = "genus";       // Genus
  public static final String TRIBE        = "tribe";       // Tribe
  public static final String FAMILY       = "family";      // Family
  public static final String SUB_FAMILY   = "subfamily";   // Subfamily
  public static final String SUPER_FAMILY = "superfamily"; // Super Family
  public static final String INFRA_ORDER  = "infraorder";  // Infraorder
  public static final String SUB_ORDER    = "suborder";    // Suborder
  public static final String ORDER        = "order";       // Order
  public static final String SUB_CLASS    = "subclass";    // Subclass
  public static final String CLASS        = "class";       // Class
  public static final String SUPER_CLASS  = "superclass";  // Superclass
  public static final String SUB_PHYLUM   = "subphylum";   // Subphylum
  public static final String PHYLUM       = "phylum";      // Phylum
  public static final String SUPER_PHYLUM = "superphylum"; // Superphylum
  public static final String SUB_KINGDOM  = "subkingdom";  // Subkingdom
  public static final String KINGDOM      = "kingdom";     // Kingdom
  public static final String AGE_TOP      = "from";        // Strat Unit ID From
  public static final String AGE_BASE     = "to";          // Strat Unit ID To
  public static final String TYPE         = "type";        // type of data
  public static final String AUTHOR       = "author";      // Author

  private int                iType  = FILE;
  private int                iRows  =  0;
  private int                iCount = -1;
  private int                iError = 0;
  private String             sError = "";
  private bioStratListStruct stList = new bioStratListStruct();

  private parseFossilListStruct  stFossils = null;
  private stratListStruct        stICS     = null;

  /** Method ReadBioStratUnitsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadBioStratUnitsXMLFile() { iType = FILE; }

  /** Method ReadBioStratUnitsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadBioStratUnitsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename  = The File to be opened.
   *  @param  stFossils = List of Common Fossil Types
   *  @param  stICS     = International Commission on Stratigraphy Chart
   *  @return stList    = The XML List Data Structure (Array List).
   */

  public bioStratListStruct Process( String                filename,
                                     parseFossilListStruct stFossils,
                                     stratListStruct       stICS )
  {
    this.stFossils = stFossils;
    this.stICS     = stICS;

    stList.iCount = 0;
    Read(filename);
    stList.iCount = iCount+1;

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(FOSSIL))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new bioStratStruct();
      }
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i<attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new bioStratStruct[iRows];
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            if (eName.equals(FOSSIL))
            {
              stList.stItem[iCount] =
                  ParseData(aName, sOut.trim(), stList.stItem[iCount]);
            }
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** Method ParseData()
   * <p> This method will parse the Data Stream
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Stratigraphic Units String
   * @param  st         = The Stratigraphic Units Data Structure
   * @return st         = The Stratigraphic Units Data Structure
   */

  private bioStratStruct ParseData( String sIdentifier,
                                    String sData,
                                    bioStratStruct st )
  {
    int i=0;

    if (sIdentifier.equals(KEY))          { st.sKID        = new String(sData); }
    if (sIdentifier.equals(ID))           { st.sKEY        = new String(sData); }
    if (sIdentifier.equals(SPECIES))      { st.species     = new String(sData); }
    if (sIdentifier.equals(GENUS))        { st.sGenus      = new String(sData); }
    if (sIdentifier.equals(TRIBE))        { st.sTribe      = new String(sData); }
    if (sIdentifier.equals(SUB_FAMILY))   { st.subfamily   = new String(sData); }
    if (sIdentifier.equals(FAMILY))       { st.sFamily     = new String(sData); }
    if (sIdentifier.equals(SUPER_FAMILY)) { st.superfamily = new String(sData); }
    if (sIdentifier.equals(INFRA_ORDER))  { st.sInfraorder = new String(sData); }
    if (sIdentifier.equals(SUB_ORDER))    { st.suborder    = new String(sData); }
    if (sIdentifier.equals(ORDER))        { st.sOrder      = new String(sData); }
    if (sIdentifier.equals(SUB_CLASS))    { st.subclass    = new String(sData); }
    if (sIdentifier.equals(CLASS))        { st.sClass      = new String(sData); }
    if (sIdentifier.equals(SUPER_CLASS))  { st.superclass  = new String(sData); }
    if (sIdentifier.equals(SUB_PHYLUM))   { st.subphylum   = new String(sData); }
    if (sIdentifier.equals(PHYLUM))       { st.sPhylum     = new String(sData); }
    if (sIdentifier.equals(SUPER_PHYLUM)) { st.superphylum = new String(sData); }
    if (sIdentifier.equals(SUB_KINGDOM))  { st.subkingdom  = new String(sData); }
    if (sIdentifier.equals(KINGDOM))      { st.sKingdom    = new String(sData); }

    if (sIdentifier.equals(AUTHOR))       { st.sAuthor     = new String(sData); }

    if (sIdentifier.equals(AGE_TOP))
    {
      if (!st.sKEY.equals("0"))
      {
        st.sKey_e = new String(sData);

        if (stICS != null)
        {
          for (i=0; i<stICS.iCount; i++)
          {
            if (st.sKey_e.equals(stICS.stItem[i].sKEY))
            {
			  st.dFrom       = stICS.stItem[i].dAgeStart;
              st.system_e    = new String( stICS.stItem[i].system );
              st.series_e    = new String( stICS.stItem[i].series );
              st.subSystem_e = new String( stICS.stItem[i].subSystem );
              st.subSeries_e = new String( stICS.stItem[i].subSeries );
            }
          }
	    }
      }
      else
      {
		if (cmn.cmnString.isNumeric(sData))
		{
		  st.dFrom = cmn.cmnString.stringToDouble(sData);

          if (stICS != null)
          {
            for (i=0; i<stICS.iCount; i++)
            {
              if ((st.dFrom >= stICS.stItem[i].dAgeStart) &&
                  (st.dFrom < stICS.stItem[i].dAgeEnd))
              {
                st.sKey_e      = new String( stICS.stItem[i].sKEY );
                st.system_e    = new String( stICS.stItem[i].system );
                st.series_e    = new String( stICS.stItem[i].series );
                st.subSystem_e = new String( stICS.stItem[i].subSystem );
                st.subSeries_e = new String( stICS.stItem[i].subSeries );
//System.out.println("FROM "+ st.dFrom +" "+ stICS.stItem[i].dAgeStart+" "+ stICS.stItem[i].dAgeEnd );
			  }
			}
		  }
		}
	  }
    }

    if (sIdentifier.equals(AGE_BASE))
    {
      if (!st.sKEY.equals("0"))
      {
        st.sKey_l = new String(sData);

        if (stICS != null)
        {
          for (i=0; i<stICS.iCount; i++)
          {
            if (st.sKey_l.equals(stICS.stItem[i].sKEY))
            {
	  	      st.dTo         = stICS.stItem[i].dAgeEnd;
              st.system_l    = new String( stICS.stItem[i].system );
              st.series_l    = new String( stICS.stItem[i].series );
              st.subSystem_l = new String( stICS.stItem[i].subSystem );
              st.subSeries_l = new String( stICS.stItem[i].subSeries );
            }
          }
	    }
      }
      else
      {
		if (cmn.cmnString.isNumeric(sData))
		{
		  st.dTo = cmn.cmnString.stringToDouble(sData);

          if (stICS != null)
          {
            for (i=0; i<stICS.iCount; i++)
            {
              if ((st.dTo > stICS.stItem[i].dAgeStart) &&
                  (st.dTo <= stICS.stItem[i].dAgeEnd))
              {
                st.sKey_l      = new String( stICS.stItem[i].sKEY );
                st.system_l    = new String( stICS.stItem[i].system );
                st.series_l    = new String( stICS.stItem[i].series );
                st.subSystem_l = new String( stICS.stItem[i].subSystem );
                st.subSeries_l = new String( stICS.stItem[i].subSeries );
//System.out.println("TO "+ st.dTo +" "+ stICS.stItem[i].dAgeStart+" "+ stICS.stItem[i].dAgeEnd );
			  }
			}
		  }
		}
	  }
    }

    if (sIdentifier.equals(TYPE))
    {
      st.sFossilID = new String(sData);

      if (stFossils != null)
      {
        for (i=0; i<stFossils.iCount; i++)
        {
          if (st.sFossilID.equals(stFossils.stFossil[i].sID))
          {
            st.sFossil = new String( stFossils.stFossil[i].sPhrase );
          }
        }
      }
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}