/*
 * @bioDataEntryFrame.java Version 1.1 11/15/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.bio.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observer;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;
import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;
import horizon.bio.gui.bioTable;
import horizon.bio.gui.bioDepthTable;
import horizon.bio.gui.bioFossilsFrame;
import horizon.strat.gui.stratPlotICSFrame;

/** Class bioDataEntryFrame
 *  <p> This Class will allow the user to enter the bio stratigraphy
 *      data into the profile program
 *
 *  @version 1.1 11/15/2011
 *  @author  John R. Victorine
 */

public class bioDataEntryFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable    pNotifier = null;  // Calling Frame Observable
  private iqstratStruct stStruct  = null;  // Global IQSTRAT Data Structure

  // Global Variables

  private Observable    notifier = null;  // This Frame Observable

  // Action Variables

  public static final int _ADD    = 0;
  public static final int _MODIFY = 1;

  private int iAction  = _ADD; // Frame Action
  private int iAction2 = _ADD; // Depth Information Action

  private int    iSpecies = -1;   // Species Location in List
  private String sKEY     = "0";  // KEY to Data
  private String sKEYa    = "0";  // KEY to Depth Range Data

  // Depth range variables

  private double dStart     = 0.0; // Starting Depth
  private double dEnd       = 0.0; // Ending Depth
  private int    iAbundance = 0;  // Fossil Abundance

  // Age range variables

  public static final int _FROM = 0;
  public static final int _TO   = 1;

  private int ics = -1;

  // -- Earliest geological time

  public double dFrom        = 0.0; // From Age in Ma
  public String sKey_e       = "";  // Stratigraphic Unit Unique KEY
  public String system_e     = "";  // System Stratigraphic Unit Names
  public String series_e     = "";  // Series Stratigraphic Unit Names
  public String subSystem_e  = "";  // Substage Stratigraphic Unit Names
  public String subSeries_e  = "";  // Subseries Stratigraphic Unit Names

  // -- Latest geological time

  public double dTo          = 0.0; // To Age in Ma
  public String sKey_l       = "";  // Stratigraphic Unit Unique KEY
  public String system_l     = "";  // System Stratigraphic Unit Names
  public String series_l     = "";  // Series Stratigraphic Unit Names
  public String subSystem_l  = "";  // Substage Stratigraphic Unit Names
  public String subSeries_l  = "";  // Subseries Stratigraphic Unit Names

  // Bio Stratigraphy List Table

  private bioTable          pTable  = null; // Species List
  private bioDepthTable     pDepth  = null; // Individual Species Depth Ranges

  private bioFossilsFrame   pFossil = null; // General Fossils Frame
  private stratPlotICSFrame pICS    = null; // ICS Stratigraphy Lookup Frame

  // Generic Rock Fossil Identifier

  public String sFossilID    = "10.2.1";       // Fossil identifier
  public String sFossil      = "macrofossils"; // Generic Fossil Name,
                                               //  i.e. Crinoids, Brachiopods, etc.

  // Bio Stratigraphy Structures

  private bioStratListStruct stList   = null;
  private bioStratStruct     stModify = null;

  // Global Frame Widges

  // -- Fossils Widgets

  private JFrame  frame  = null;

  // -- Classification Widges

  private JTextField txtKingdom     = new JTextField(); // Kingdom
  private JTextField txtSubkingdom  = new JTextField(); // Subkingdom

  private JTextField txtSuperphylum = new JTextField(); // Superphylum
  private JTextField txtPhylum      = new JTextField(); // Phylum
  private JTextField txtSubphylum   = new JTextField(); // Subphylum

  private JTextField txtSuperclass  = new JTextField(); // Superclass
  private JTextField txtClass       = new JTextField(); // Class
  private JTextField txtSubclass    = new JTextField(); // Subclass

  private JTextField txtSuperorder  = new JTextField(); // Superorder
  private JTextField txtOrder       = new JTextField(); // Order
  private JTextField txtSuborder    = new JTextField(); // Suborder
  private JTextField txtInfraorder  = new JTextField(); // Infraorder

  private JTextField txtSuperfamily = new JTextField(); // Superfamily
  private JTextField txtFamily      = new JTextField(); // Family
  private JTextField txtSubfamily   = new JTextField(); // Subfamily

  private JTextField txtTribe       = new JTextField(); // Tribe

  private JTextField txtGenus       = new JTextField(); // Genus
  private JTextField txtSpecies     = new JTextField(); // Species

  private JTextField txtGroup       = new JTextField(); // Group
  private JButton    btnGroup       = new JButton();    // Change Group

  // -- Species Abundance

  // No Fossils
  public static final int _NONE     = horizon.bio.bioStratStruct._NONE;
  // (s) Sparse. Applies to fossils very poorly represented in the rock.
  public static final int _SPARSE   = horizon.bio.bioStratStruct._SPARSE;
  // No modifier the fossils are represented in the rock.
  public static final int _PRESENT  = horizon.bio.bioStratStruct._PRESENT;
  // (c) Common. Refers to fossils that are numerous but not abundant
  //         enough to be immediately conspicuous in the sample.
  public static final int _COMMON   = horizon.bio.bioStratStruct._COMMON;
  // (a) Abundant. Refers to fossils that are numerous enough to be readily
  //               seen either on an etched surface or in a washed sample.
  public static final int _ABUNDANT = horizon.bio.bioStratStruct._ABUNDANT;
  // (p) Profuse. Applies to fossils that are a major constituent of the rock.
  public static final int _PROFUSE  = horizon.bio.bioStratStruct._PROFUSE;

  private JRadioButton rbNone       = new JRadioButton();  // No Fossils
  private JRadioButton rbSparse     = new JRadioButton();  // Sparse Fossils
  private JRadioButton rbPresent    = new JRadioButton();  // Fossils Present
  private JRadioButton rbCommon     = new JRadioButton();  // Fossils Common
  private JRadioButton rbAbundant   = new JRadioButton();  // Abundant Fossils
  private JRadioButton rbProfuse    = new JRadioButton();  // Profuse Fossils

  // -- Depth Range Textfields

  private JTextField txtStart   = new JTextField();  // Starting Depth
  private JTextField txtEnd     = new JTextField();  // Ending Depth

  // -- Depth Range Table Buttons

  private JButton btnDClear     = new JButton();     // Clear Depth Edit Fields
  private JButton btnDAdd       = new JButton();     // Add Depth Information

  private JButton btnModify     = new JButton();     // Modify Depth Range
  private JButton btnDRemove    = new JButton();     // Remove Depth Range
  private JButton btnDRemoveAll = new JButton();     // Remove all Depth Ranges

  // -- Age Limits Widgets

  private JTextField txtFrom    = new JTextField();  // From Age
  private JTextField txtTo      = new JTextField();  // To Age

  private JButton btnFrom       = new JButton();     // From Age Button
  private JButton btnTo         = new JButton();     // To Age Button

  // -- Global Panel Buttons

  private JButton btnClear     = new JButton();     // Clear Species Edit Fields
  private JButton btnAdd       = new JButton();     // Add/Modify Species Information

  private JButton btnSelect     = new JButton();
  private JButton btnRemove     = new JButton();
  private JButton btnRemoveAll  = new JButton();

  private JButton btnCancel     = new JButton();

  /** Constructor bioDataEntryFrame()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public bioDataEntryFrame(Observable pNotifier, iqstratStruct stStruct)
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;

      jbInit();
      addWindowListener( new bioDataEntryFrame_WindowListener() );
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	JPanel pnlCenter       = new JPanel();

	JPanel pnlNEW          = new JPanel();

	JLabel lblNEW          = new JLabel();

	JPanel pnlAo           = new JPanel();

	JPanel pnlKingdom      = new JPanel();
	JPanel pnlSubkingdom   = new JPanel();
	JPanel pnlA            = new JPanel();

	JLabel lblKingdom      = new JLabel();
	JLabel lblSubkingdom   = new JLabel();

	JPanel pnlSuperphylum  = new JPanel();
	JPanel pnlPhylum       = new JPanel();
	JPanel pnlSubphylum    = new JPanel();
	JPanel pnlB            = new JPanel();

	JLabel lblSuperphylum  = new JLabel();
	JLabel lblPhylum       = new JLabel();
	JLabel lblSubphylum    = new JLabel();

	JPanel pnlSuperclass   = new JPanel();
	JPanel pnlClass        = new JPanel();
	JPanel pnlSubclass     = new JPanel();
	JPanel pnlC            = new JPanel();

	JLabel lblSuperclass   = new JLabel();
	JLabel lblClass        = new JLabel();
	JLabel lblSubclass     = new JLabel();

	JPanel pnlSuperorder   = new JPanel();
	JPanel pnlOrder        = new JPanel();
	JPanel pnlSuborder     = new JPanel();
	JPanel pnlInfraorder   = new JPanel();
	JPanel pnlD            = new JPanel();

	JLabel lblSuperorder   = new JLabel();
	JLabel lblOrder        = new JLabel();
	JLabel lblSuborder     = new JLabel();
	JLabel lblInfraorder   = new JLabel();

	JPanel pnlSuperfamily  = new JPanel();
	JPanel pnlFamily       = new JPanel();
	JPanel pnlSubfamily    = new JPanel();
	JPanel pnlE            = new JPanel();

	JLabel lblSuperfamily  = new JLabel();
	JLabel lblFamily       = new JLabel();
	JLabel lblSubfamily    = new JLabel();

	JPanel pnlTribe        = new JPanel();
	JPanel pnlF            = new JPanel();

	JLabel lblTribe        = new JLabel();

	JPanel pnlGenus        = new JPanel();

	JLabel lblGenus        = new JLabel();

	JPanel pnlSpecies      = new JPanel();

	JLabel lblSpecies      = new JLabel();
	JLabel lblFormat       = new JLabel();

	JPanel pnlOther        = new JPanel();

	JPanel pnlGroup        = new JPanel();
	JLabel lblGroup        = new JLabel();

	JPanel pnlDepths       = new JPanel();
	JPanel pnlAbundance    = new JPanel();
	JPanel pnlDepthCenter  = new JPanel();
	JPanel pnlDepthTable   = new JPanel();
    JPanel pnlDepthButtons = new JPanel();

	JPanel pnlDepthRange   = new JPanel();
    JPanel pnlStart        = new JPanel();
    JPanel pnlEnd          = new JPanel();
    JPanel pnlAdd          = new JPanel();

	JLabel lblA            = new JLabel();

	JPanel pnlOtherSouth   = new JPanel();

    JPanel pnlAgeLimits    = new JPanel();
    JPanel pnlFrom         = new JPanel();
    JPanel pnlTo           = new JPanel();

	JLabel lblRequired     = new JLabel();

    JPanel pnlList         = new JPanel();

    JPanel pnlListButtons  = new JPanel();

    JScrollPane scrollList  = new JScrollPane();
    JScrollPane scrollDepth = new JScrollPane();
    ButtonGroup groupAbund  = new ButtonGroup();

    TitledBorder titledBorderClass = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Classification");
    titledBorderClass.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSp = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Fossil Species List");
    titledBorderSp.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAbundance = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Fossil Abundance:");
    titledBorderAbundance.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderAbundance.setTitleColor(Color.blue);

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Starting Depth:");
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderStart.setTitleColor(Color.blue);

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Ending Depth:");
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEnd.setTitleColor(Color.blue);

    TitledBorder titledBorderFrom = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Age From (Ma):");
    titledBorderFrom.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderTo = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Age To (Ma):");
    titledBorderTo.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Create an Observable

    notifier = new bioDataEntryFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Create Frame

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Enter Bio-Stratigraphy Data:");

    // Classification Panel

    pnlCenter.setLayout(new BorderLayout());

    // -- NEW Panels

    pnlNEW.setLayout(new BorderLayout());

    lblNEW.setFont(new java.awt.Font("Dialog", 1, 14));
    lblNEW.setText("*** New Species ***");

    // -- Classification Panels

    pnlAo.setLayout(new GridLayout(24,1));
    pnlAo.setBorder(titledBorderClass);

    // .. Kingdom Panels
    // .... Kingdom Panel

    pnlKingdom.setLayout(new BorderLayout());

    lblKingdom.setFont(new java.awt.Font("monospaced", 1, 12));
    lblKingdom.setText("Kingdom:          ");

    txtKingdom.setText("");
    txtKingdom.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Subkingdom Panel

    pnlSubkingdom.setLayout(new BorderLayout());

    lblSubkingdom.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSubkingdom.setText("  Subkingdom:     ");

    txtSubkingdom.setText("");
    txtSubkingdom.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Phylum Panels
    // .... Superphylum Panel

    pnlSuperphylum.setLayout(new BorderLayout());

    lblSuperphylum.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSuperphylum.setText("Superphylum:      ");

    txtSuperphylum.setText("");
    txtSuperphylum.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Phylum Panel

    pnlPhylum.setLayout(new BorderLayout());

    lblPhylum.setFont(new java.awt.Font("monospaced", 1, 12));
    lblPhylum.setText("  Phylum:         ");

    txtPhylum.setText("");
    txtPhylum.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Subphylum Panel

    pnlSubphylum.setLayout(new BorderLayout());

    lblSubphylum.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSubphylum.setText("    Subphylum:    ");

    txtSubphylum.setText("");
    txtSubphylum.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Class Panels
    // .... Superclass Panel

    pnlSuperclass.setLayout(new BorderLayout());

    lblSuperclass.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSuperclass.setText("Superclass:       ");

    txtSuperclass.setText("");
    txtSuperclass.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Class Panel

    pnlClass.setLayout(new BorderLayout());

    lblClass.setFont(new java.awt.Font("monospaced", 1, 12));
    lblClass.setText("  Class:          ");

    txtClass.setText("");
    txtClass.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Subclass Panel

    pnlSubclass.setLayout(new BorderLayout());

    lblSubclass.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSubclass.setText("    Subclass:     ");

    txtSubclass.setText("");
    txtSubclass.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Order Panels
    // .... Superorder Panel

    pnlSuperorder.setLayout(new BorderLayout());

    lblSuperorder.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSuperorder.setText("Superorder:       ");

    txtSuperorder.setText("");
    txtSuperorder.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Order Panel

    pnlOrder.setLayout(new BorderLayout());

    lblOrder.setFont(new java.awt.Font("monospaced", 1, 12));
    lblOrder.setText("  Order:          ");

    txtOrder.setText("");
    txtOrder.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Suborder Panel

    pnlSuborder.setLayout(new BorderLayout());

    lblSuborder.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSuborder.setText("    Suborder:     ");

    txtSuborder.setText("");
    txtSuborder.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Infraorder Panel

    pnlInfraorder.setLayout(new BorderLayout());

    lblInfraorder.setFont(new java.awt.Font("monospaced", 1, 12));
    lblInfraorder.setText("      Infraorder: ");

    txtInfraorder.setText("");
    txtInfraorder.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Family Panels
    // .... Superfamily Panel

    pnlSuperfamily.setLayout(new BorderLayout());

    lblSuperfamily.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSuperfamily.setText("Superfamily:      ");

    txtSuperfamily.setText("");
    txtSuperfamily.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Family Panel

    pnlFamily.setLayout(new BorderLayout());

    lblFamily.setFont(new java.awt.Font("monospaced", 1, 12));
    lblFamily.setText("  Family:         ");

    txtFamily.setText("");
    txtFamily.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .... Subfamily Panel

    pnlSubfamily.setLayout(new BorderLayout());

    lblSubfamily.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSubfamily.setText("    Subfamily:    ");

    txtSubfamily.setText("");
    txtSubfamily.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Tribe Panel

    pnlTribe.setLayout(new BorderLayout());

    lblTribe.setFont(new java.awt.Font("monospaced", 1, 12));
    lblTribe.setText("      Tribe:      ");

    txtTribe.setText("");
    txtTribe.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Genus Panel

    pnlGenus.setLayout(new BorderLayout());

    lblGenus.setFont(new java.awt.Font("monospaced", 1, 12));
    lblGenus.setForeground(Color.blue);
    lblGenus.setText("        Genus:    ");

    txtGenus.setText("");
    txtGenus.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Species Panel

    pnlSpecies.setLayout(new BorderLayout());

    lblSpecies.setFont(new java.awt.Font("monospaced", 1, 12));
    lblSpecies.setForeground(Color.blue);
    lblSpecies.setText("          Species:");

    txtSpecies.setText("");
    txtSpecies.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. Required Fields Label

    lblFormat.setFont(new java.awt.Font("monospaced", 3, 12));
//    lblFormat.setForeground(Color.blue);
    lblFormat.setText("                    " +
                      "Species Format: \"Genus species\" or \"Genus (sp)\"");

    // -- Other Data Panel

    pnlOther.setLayout(new BorderLayout());

    // .. General Fossil Group Panel

    pnlGroup.setLayout(new BorderLayout());

    lblGroup.setFont(new java.awt.Font("monospaced", 1, 12));
    lblGroup.setForeground(Color.blue);
    lblGroup.setText("General Fossil Group:");

    txtGroup.setText(sFossil);
    txtGroup.setFont(new java.awt.Font("Dialog", 1, 12));
    txtGroup.setEditable(false);

    btnGroup.setFont(new java.awt.Font("Dialog", 1, 11));
    btnGroup.setText("Group?");
    btnGroup.addActionListener(this);

    // .. Depth Range Panels

    pnlDepths.setLayout(new BorderLayout());
    pnlDepths.setPreferredSize(new Dimension(10, 200));

    // .... Abundance Panel

    pnlAbundance.setLayout(new GridLayout(6,1));
    pnlAbundance.setBorder(titledBorderAbundance);

    rbNone.setFont(new java.awt.Font("monospaced", 1, 11));
    rbNone.setSelected(true);
    rbNone.setText("None     (0)");
    rbNone.addActionListener(this);

    rbSparse.setFont(new java.awt.Font("monospaced", 1, 11));
    rbSparse.setText("Sparse   (1)");
    rbSparse.addActionListener(this);

    rbPresent.setFont(new java.awt.Font("monospaced", 1, 11));
    rbPresent.setText("Present  (2)");
    rbPresent.addActionListener(this);

    rbCommon.setFont(new java.awt.Font("monospaced", 1, 11));
    rbCommon.setText("Common   (3)");
    rbCommon.addActionListener(this);

    rbAbundant.setFont(new java.awt.Font("monospaced", 1, 11));
    rbAbundant.setText("Abundant (4)");
    rbAbundant.addActionListener(this);

    rbProfuse.setFont(new java.awt.Font("monospaced", 1, 11));
    rbProfuse.setText("Profuse  (5)");
    rbProfuse.addActionListener(this);

    // .... Depths Center Panel

    pnlDepthCenter.setLayout(new BorderLayout());

    // .. -- .. Depth Ranges Panels

    pnlDepthRange.setLayout(new GridLayout());

    // ... -- ... Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("" + dStart);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // ... -- ... Ending Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("" + dEnd);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    // .. -- .. Species Depth Ranges

    pnlDepthTable.setPreferredSize(new Dimension(10, 100));
    pnlDepthTable.setLayout(new BorderLayout());

    pDepth      = new bioDepthTable();
    scrollDepth = pDepth.getScrollPane();

    // ... -- ... Add Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnDAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDAdd.setText("Add Depth Data");
    btnDAdd.addActionListener(this);

    btnDClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDClear.setText("Clear Depth Data");
    btnDClear.addActionListener(this);

    // .. -- .. Species Depth Ranges Table Buttons

    pnlDepthButtons.setLayout(new GridLayout());

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setText("Modify");
    btnModify.addActionListener(this);

    btnDRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDRemove.setText("Remove");
    btnDRemove.addActionListener(this);

    btnDRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDRemoveAll.setText("Remove All");
    btnDRemoveAll.addActionListener(this);

    // -- Other South Panel

    pnlOtherSouth.setLayout(new BorderLayout());

    // .. Age Limits Panel

    pnlAgeLimits.setLayout(new GridLayout());

    // .... Starting Depth Panel

    pnlFrom.setBorder(titledBorderFrom);
    pnlFrom.setLayout(new BorderLayout());

    txtFrom.setText("" + dFrom);
    txtFrom.setHorizontalAlignment(SwingConstants.TRAILING);
    txtFrom.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    btnFrom.setFont(new java.awt.Font("Dialog", 1, 11));
    btnFrom.setBackground(new Color(220, 220, 220));
    btnFrom.setText("Age From");
    btnFrom.addActionListener(this);

    // .... Ending Depth Panel

    pnlTo.setBorder(titledBorderTo);
    pnlTo.setLayout(new BorderLayout());

    txtTo.setText("" + dTo);
    txtTo.setHorizontalAlignment(SwingConstants.TRAILING);
    txtTo.addFocusListener(new bioDataEntryFrameFocusAdapter(this));

    btnTo.setFont(new java.awt.Font("Dialog", 1, 11));
    btnTo.setBackground(new Color(220, 220, 220));
    btnTo.setText("Age To");
    btnTo.addActionListener(this);

    // .. Required Fields Label

    lblRequired.setFont(new java.awt.Font("monospaced", 3, 16));
    lblRequired.setForeground(Color.red);
    lblRequired.setText("NOTE: Blue Label is a Required Field");

    // Global Buttons Panel

    // -- Depth List Buttons Panel

    // Selected Bio Stratigraphy Data Panel

    pnlList.setPreferredSize(new Dimension(10, 200));
    pnlList.setLayout(new BorderLayout());
    pnlList.setBorder(titledBorderSp);

    pTable     = new bioTable();
    scrollList = pTable.getScrollPane();

    // List Buttons Panel

    pnlListButtons.setLayout(new GridLayout(7,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setText("Select Species");
    btnSelect.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setText("Close Dialog");
    btnCancel.addActionListener(this);

    // Attach Widgets to Panel

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(pnlAo,                 BorderLayout.NORTH);

    pnlAo.add(pnlKingdom,                null);
    pnlKingdom.add(lblKingdom,           BorderLayout.WEST);
    pnlKingdom.add(txtKingdom,           BorderLayout.CENTER);

    pnlAo.add(pnlSubkingdom,             null);
    pnlSubkingdom.add(lblSubkingdom,     BorderLayout.WEST);
    pnlSubkingdom.add(txtSubkingdom,     BorderLayout.CENTER);

    pnlAo.add(pnlA,                      null);

    pnlAo.add(pnlSuperphylum,            null);
    pnlSuperphylum.add(lblSuperphylum,   BorderLayout.WEST);
    pnlSuperphylum.add(txtSuperphylum,   BorderLayout.CENTER);

    pnlAo.add(pnlPhylum,                 null);
    pnlPhylum.add(lblPhylum,             BorderLayout.WEST);
    pnlPhylum.add(txtPhylum,             BorderLayout.CENTER);

    pnlAo.add(pnlSubphylum,              null);
    pnlSubphylum.add(lblSubphylum,       BorderLayout.WEST);
    pnlSubphylum.add(txtSubphylum,       BorderLayout.CENTER);

    pnlAo.add(pnlB,                      null);

    pnlAo.add(pnlSuperclass,             null);
    pnlSuperclass.add(lblSuperclass,     BorderLayout.WEST);
    pnlSuperclass.add(txtSuperclass,     BorderLayout.CENTER);

    pnlAo.add(pnlClass,                  null);
    pnlClass.add(lblClass,               BorderLayout.WEST);
    pnlClass.add(txtClass,               BorderLayout.CENTER);

    pnlAo.add(pnlSubclass,               null);
    pnlSubclass.add(lblSubclass,         BorderLayout.WEST);
    pnlSubclass.add(txtSubclass,         BorderLayout.CENTER);

    pnlAo.add(pnlC,                      null);

    pnlAo.add(pnlSuperorder,             null);
    pnlSuperorder.add(lblSuperorder,     BorderLayout.WEST);
    pnlSuperorder.add(txtSuperorder,     BorderLayout.CENTER);

    pnlAo.add(pnlOrder,                  null);
    pnlOrder.add(lblOrder,               BorderLayout.WEST);
    pnlOrder.add(txtOrder,               BorderLayout.CENTER);

    pnlAo.add(pnlSuborder,               null);
    pnlSuborder.add(lblSuborder,         BorderLayout.WEST);
    pnlSuborder.add(txtSuborder,         BorderLayout.CENTER);

    pnlAo.add(pnlInfraorder,             null);
    pnlInfraorder.add(lblInfraorder,     BorderLayout.WEST);
    pnlInfraorder.add(txtInfraorder,     BorderLayout.CENTER);

    pnlAo.add(pnlD,                      null);

    pnlAo.add(pnlSuperfamily,            null);
    pnlSuperfamily.add(lblSuperfamily,   BorderLayout.WEST);
    pnlSuperfamily.add(txtSuperfamily,   BorderLayout.CENTER);

    pnlAo.add(pnlFamily,                 null);
    pnlFamily.add(lblFamily,             BorderLayout.WEST);
    pnlFamily.add(txtFamily,             BorderLayout.CENTER);

    pnlAo.add(pnlSubfamily,              null);
    pnlSubfamily.add(lblSubfamily,       BorderLayout.WEST);
    pnlSubfamily.add(txtSubfamily,       BorderLayout.CENTER);

    pnlAo.add(pnlE,                      null);

    pnlAo.add(pnlTribe,                  null);
    pnlTribe.add(lblTribe,               BorderLayout.WEST);
    pnlTribe.add(txtTribe,               BorderLayout.CENTER);

//    pnlAo.add(pnlF,                      null);

//    pnlAo.add(lblRequired,               null);

    pnlAo.add(pnlGenus,                  null);
    pnlGenus.add(lblGenus,               BorderLayout.WEST);
    pnlGenus.add(txtGenus,               BorderLayout.CENTER);

    pnlAo.add(pnlSpecies,                null);
    pnlSpecies.add(lblSpecies,           BorderLayout.WEST);
    pnlSpecies.add(txtSpecies,           BorderLayout.CENTER);

    pnlAo.add(lblFormat,                 null);

    pnlCenter.add(pnlOther,              BorderLayout.CENTER); //BorderLayout.SOUTH);

    pnlOther.add(pnlGroup,               BorderLayout.NORTH);
    pnlGroup.add(lblGroup,               BorderLayout.WEST);
    pnlGroup.add(txtGroup,               BorderLayout.CENTER);
    pnlGroup.add(btnGroup,               BorderLayout.EAST);

    pnlOther.add(pnlDepths,              BorderLayout.CENTER);
    pnlDepths.add(pnlAbundance,          BorderLayout.WEST);
    pnlAbundance.add(rbNone,             null);
    pnlAbundance.add(rbSparse,           null);
    pnlAbundance.add(rbPresent,          null);
    pnlAbundance.add(rbCommon,           null);
    pnlAbundance.add(rbAbundant,         null);
    pnlAbundance.add(rbProfuse,          null);

    groupAbund.add( rbNone );
    groupAbund.add( rbSparse );
    groupAbund.add( rbPresent );
    groupAbund.add( rbCommon );
    groupAbund.add( rbAbundant );
    groupAbund.add( rbProfuse );

    pnlDepths.add(pnlDepthCenter,        BorderLayout.CENTER);
    pnlDepthCenter.add(pnlDepthTable,    BorderLayout.CENTER);
    pnlDepthTable.add(scrollDepth,       BorderLayout.CENTER);

    pnlDepthCenter.add(pnlDepthRange,    BorderLayout.NORTH);
    pnlDepthRange.add(pnlStart,          null);
    pnlStart.add(txtStart,               BorderLayout.CENTER);

    pnlDepthRange.add(pnlEnd,            null);
    pnlEnd.add(txtEnd,                   BorderLayout.CENTER);

    pnlDepthRange.add(pnlAdd,            null);
    pnlAdd.add(btnDAdd,                  null);
    pnlAdd.add(btnDClear,                null);

    pnlDepthCenter.add(pnlDepthButtons,  BorderLayout.SOUTH);
    pnlDepthButtons.add(btnModify,       null);
    pnlDepthButtons.add(btnDRemove,      null);
    pnlDepthButtons.add(btnDRemoveAll,   null);

    pnlOther.add(pnlOtherSouth,          BorderLayout.SOUTH);
    pnlOtherSouth.add(pnlAgeLimits,      BorderLayout.NORTH);
    pnlAgeLimits.add(pnlFrom,            null);
    pnlFrom.add(txtFrom,                 BorderLayout.CENTER);
    pnlFrom.add(btnFrom,                 BorderLayout.SOUTH);

    pnlAgeLimits.add(pnlTo,              null);
    pnlTo.add(txtTo,                     BorderLayout.CENTER);
    pnlTo.add(btnTo,                     BorderLayout.SOUTH);

    pnlOtherSouth.add(lblRequired,       BorderLayout.SOUTH);

    // -- Attach the Species List at the bottom of Panel

    this.getContentPane().add(pnlList,   BorderLayout.SOUTH);
    pnlList.add(scrollList,              BorderLayout.CENTER);

    // .. Attach the Species List Buttons below Table

    pnlList.add(pnlListButtons,          BorderLayout.EAST);
    pnlListButtons.add(btnClear,         null);
    pnlListButtons.add(btnAdd,           null);
    pnlListButtons.add(lblA,             null);
    pnlListButtons.add(btnSelect,        null);
    pnlListButtons.add(btnRemove,        null);
    pnlListButtons.add(btnRemoveAll,     null);
    pnlListButtons.add(btnCancel,        null);


    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(600, d.height-50));
    this.setLocation((d.width - this.getSize().width) / 2, 10);
//    this.setLocation((d.width - this.getSize().width) / 2,
//                     (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve the Bio Stratigraphy to be added to the list
   * @return st = Bio Stratigraphy Data Structure
   */

  public void getData()
  {
    switch (iAction)
    {
      case _ADD:
  	    if (stModify != null)
	      stModify.delete();
	    stModify = null;

        sKEY     = new String( cmn.cmnString.UniqueName() );
        stModify = new bioStratStruct();

        stModify.iRows         = 1;
        stModify.sKEYa         = new String[1];
        stModify.iAbundance    = new int[1];
        stModify.depthStart    = new double[1];
        stModify.depthEnd      = new double[1];

        stModify.sKEYa[0]      = new String( sKEY );
        stModify.iAbundance[0] = iAbundance;
        stModify.depthStart[0] = dStart;
        stModify.depthEnd[0]   = dEnd;
       break;

	  case _MODIFY:
        switch (iAction2)
        {
          case _ADD:
	        stModify = horizon.bio.bioStratUtility.addDepth(
			  	         stModify,
                         cmn.cmnString.UniqueName(),
                         iAbundance,
                         dStart,
                         dEnd );
           break;

    	  case _MODIFY:
	        stModify = horizon.bio.bioStratUtility.modifyDepth(
			  	         stModify,
                         sKEYa,
                         iAbundance,
                         dStart,
                         dEnd );
    	    break;
		}
	    break;
  	}

	if (stModify != null)
	{
      stModify.sKEY        = new String( sKEY );

      stModify.sKingdom    = new String( txtKingdom.getText() );
      stModify.subkingdom  = new String( txtSubkingdom.getText() );

      stModify.superphylum = new String( txtSuperphylum.getText() );
      stModify.sPhylum     = new String( txtPhylum.getText() );
      stModify.subphylum   = new String( txtSubphylum.getText() );

      stModify.superclass  = new String( txtSuperclass.getText() );
      stModify.sClass      = new String( txtClass.getText() );
      stModify.subclass    = new String( txtSubclass.getText() );

      stModify.superorder  = new String( txtSuperorder.getText() );
      stModify.sOrder      = new String( txtOrder.getText() );
      stModify.suborder    = new String( txtSuborder.getText() );
      stModify.sInfraorder = new String( txtInfraorder.getText() );

      stModify.superfamily = new String( txtSuperfamily.getText() );
      stModify.sFamily     = new String( txtFamily.getText() );
      stModify.subfamily   = new String( txtSubfamily.getText() );

      stModify.sTribe      = new String( txtTribe.getText() );

      stModify.sGenus      = new String( txtGenus.getText() );
      stModify.sName       = new String( txtSpecies.getText() );

      stModify.sFossilID   = new String( sFossilID );
      stModify.sFossil     = new String( sFossil );

      stModify.dFrom       = dFrom;
      stModify.sKey_e      = new String( sKey_e );
      stModify.system_e    = new String( system_e );
      stModify.series_e    = new String( series_e );
      stModify.subSystem_e = new String( subSystem_e );
      stModify.subSeries_e = new String( subSeries_e );

      stModify.dTo         = dTo;
      stModify.sKey_l      = new String( sKey_l );
      stModify.system_l    = new String( system_l );
      stModify.series_l    = new String( series_l );
      stModify.subSystem_l = new String( subSystem_l );
      stModify.subSeries_l = new String( subSeries_l );
	}
  }

  /** Method getList()
   * <p> This method will retrieve the Bio Stratigraphy list data structure
   * @return st = Bio Stratigraphy Data List Structure
   */

  public bioStratListStruct getList() { return (stList); }

  /** Method getFossilGroup()
   *  <p> This method will set the General Fossil Group
   */

  private void getFossilGroup()
  {
	if (pFossil != null)
	{
      sFossilID = new String(pFossil.getID());    // Fossil identifier
      sFossil   = new String(pFossil.getName());  // Generic Fossil Name,
      txtGroup.setText(sFossil);                // Fossil Group

      closeFossils();
	}
  }

  /** Method getICSData()
   * <p> This method will set the ICS data.
   */

  private  void getICSData()
  {
    String sID = "";

    if (pICS != null)
    {
      sID = new String( pICS.getData() );

      if (stStruct.stICS != null)
      {
        for (int i=0; i<stStruct.stICS.iCount; i++)
        {
          if (sID.equals(stStruct.stICS.stItem[i].sKEY))
          {
			switch (ics)
			{
              case _FROM:
                setAge( _FROM, sID );
                break;
              case _TO:
                setAge( _TO, sID );
                break;
		    }
          }
        }
      }

      pICS.close();
      pICS = null;
      ics  = -1;
    }
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    this.dEnd   = depthEnd;

    setButtons();
  }

  /** Method setData()
   * <p> This method will set bio Stratigraphy List Data Structure
   * @param st = the bio Stratigraphy List Data Structure
   */

  public void setData(bioStratListStruct st)
  {
    this.stList = st;
    pTable.setData( st );
  }

  /** Method setData()
   * <p> This method will set bio Stratigraphy Data Structure to the panel
   * @param st = the bio Stratigraphy Data Structure
   */

  public void setData( bioStratStruct st )
  {
	int i=0;
	int    iSecond  = 0;
	String system   = "";
	String series   = "";
	String sICSSys  = "";
	String sICSSer  = "";
	int    iColor[] = {220, 220, 220 };

    if (pDepth != null) { pDepth.setData( st ); }

	if (st != null)
	{
      sKEY = new String( st.sKEY );           // Primary KEY of Species

      txtKingdom.setText( st.sKingdom );      // Kingdom
      txtSubkingdom.setText(st.subkingdom);   // Subkingdom

      txtSuperphylum.setText(st.superphylum); // Superphylum
      txtPhylum.setText(st.sPhylum);          // Phylum
      txtSubphylum.setText(st.subphylum);     // Subphylum

      txtSuperclass.setText(st.superclass);   // Superclass
      txtClass.setText(st.sClass);            // Class
      txtSubclass.setText(st.subclass);       // Subclass

      txtSuperorder.setText(st.superorder);   // Superorder
      txtOrder.setText(st.sOrder);            // Order
      txtSuborder.setText(st.suborder);       // Suborder
      txtInfraorder.setText(st.sInfraorder);  // Infraorder

      txtSuperfamily.setText(st.superfamily); // Superfamily
      txtFamily.setText(st.sFamily);          // Family
      txtSubfamily.setText(st.subfamily);     // Subfamily

      txtTribe.setText(st.sTribe);            // Tribe

      txtGenus.setText(st.sGenus);            // Genus
      txtSpecies.setText(st.sName);           // Species

      sFossilID = new String(st.sFossilID);   // Fossil identifier
      sFossil   = new String(st.sFossil);     // Generic Fossil Name,
      txtGroup.setText(sFossil);              // Fossil Group

      setAge( _FROM, st.sKey_e );
      setAge( _TO,   st.sKey_l );
	}
  }

  /** Method setAge()
   *  <p> This method will set the Age button
   *  @param iAge = The Button to be modified
   *  @param sKEY = The ICS Stratigraphic Unit KEY
   */

  private void setAge( int iAge, String sKEY )
  {
	String sLabel    = "";
	String system    = "";
	String series    = "";
	String subSystem = "";
	String subSeries = "";
	double dAgeStart = 0.0;
	double dAgeEnd   = 0.0;
	int    iColor[]  = {220, 220, 220 };

    if (sKEY.length() > 1)
    {
      if (stStruct.stICS != null)
      {
	    for (int i=0; i<stStruct.stICS.iCount; i++)
		{
		  if (sKEY.equals(stStruct.stICS.stItem[i].sKEY))
		  {
		    iColor[0] = stStruct.stICS.stItem[i].iRed;
		    iColor[1] = stStruct.stICS.stItem[i].iGreen;
		    iColor[2] = stStruct.stICS.stItem[i].iBlue;

            system    = new String(stStruct.stICS.stItem[i].system);
            series    = new String(stStruct.stICS.stItem[i].series);
            subSystem = new String(stStruct.stICS.stItem[i].subSystem);
            subSeries = new String(stStruct.stICS.stItem[i].subSeries);
            dAgeStart = stStruct.stICS.stItem[i].dAgeStart;
            dAgeEnd   = stStruct.stICS.stItem[i].dAgeEnd;

      	    if ((subSystem.length() > 0) && (subSeries.length() > 0))
		    {
		      sLabel = new String(subSystem + " ("+ subSeries + ")");
		    }
            else if (subSystem.length() > 0)
            {
		      sLabel = new String(subSystem);
		    }
		    else if ((system.length() > 0) && (series.length() > 0))
		    {
		      sLabel = new String(system + " ("+ series + ")");
		    }
		    else if (system.length() > 0)
		    {
		      sLabel = new String(system);
		    }
	      }
	    }
	  }
	}

    switch (iAge)
	{
      case _FROM:
        dFrom        = dAgeStart;              // From Age in Ma
        sKey_e       = new String(sKEY);       // Stratigraphic Unit Unique KEY
        system_e     = new String(system);     // System Stratigraphic Unit Names
        series_e     = new String(series);     // Series Stratigraphic Unit Names
        subSystem_e  = new String(subSystem);  // Substage Stratigraphic Unit Names
        subSeries_e  = new String(subSeries);  // Subseries Stratigraphic Unit Names

        if (dTo <= dFrom)
        {
          dTo          = dAgeEnd;                // To Age in Ma
          sKey_l       = new String(sKEY);       // Stratigraphic Unit Unique KEY
          system_l     = new String(system);     // System Stratigraphic Unit Names
          series_l     = new String(series);     // Series Stratigraphic Unit Names
          subSystem_l  = new String(subSystem);  // Substage Stratigraphic Unit Names
          subSeries_l  = new String(subSeries);  // Subseries Stratigraphic Unit Names

          btnTo.setBackground( new Color(iColor[0], iColor[1], iColor[2]) );
          if (sLabel.length() > 0) btnTo.setText( sLabel );
		}

        btnFrom.setBackground( new Color(iColor[0], iColor[1], iColor[2]) );
        if (sLabel.length() > 0) btnFrom.setText( sLabel );
        break;

      case _TO:
        dTo          = dAgeEnd;                // To Age in Ma
        sKey_l       = new String(sKEY);       // Stratigraphic Unit Unique KEY
        system_l     = new String(system);     // System Stratigraphic Unit Names
        series_l     = new String(series);     // Series Stratigraphic Unit Names
        subSystem_l  = new String(subSystem);  // Substage Stratigraphic Unit Names
        subSeries_l  = new String(subSeries);  // Subseries Stratigraphic Unit Names

        btnTo.setBackground( new Color(iColor[0], iColor[1], iColor[2]) );
        if (sLabel.length() > 0) btnTo.setText( sLabel );
        break;
    }

    txtFrom.setText("" + dFrom); // From Age
    txtTo.setText("" + dTo);     // To Age
  }

  /** Method setAction()
   * <p> This method will set the Action variables
   * @param iAction = The Frame Action to be set
   * @param iAction = The Depth Information Action to be set
   */

  public void setAction(int iAction, int iAction2)
  {
    this.iAction  = iAction;
    this.iAction2 = iAction2;

    switch (iAction)
    {
      case _ADD:
        btnAdd.setText("Add");
        break;

      case _MODIFY:
        btnAdd.setText("Modify Info");
        break;
    }

    switch (iAction2)
    {
      case _ADD:
        btnDAdd.setText("Add Depth Data");
        break;

      case _MODIFY:
        btnDAdd.setText("Modify Data");
        break;
    }

    setButtons();
  }

  /** Method setButtons()
   * <p> This method will enable or disable buttons depending on data
   */

  private void setButtons()
  {
    btnDAdd.setEnabled(false);
    btnModify.setEnabled(false);
    btnDRemove.setEnabled(false);
    btnDRemoveAll.setEnabled(false);

    btnAdd.setEnabled(false);
    btnSelect.setEnabled(false);
    btnRemove.setEnabled(false);
    btnRemoveAll.setEnabled(false);

    if ((dStart < dEnd) &&
        (txtGenus.getText().length() > 0) &&
        (txtSpecies.getText().length() > 0))
    {
      btnDAdd.setEnabled(true);
    }

    if ((txtGenus.getText().length() > 0) &&
        (txtSpecies.getText().length() > 0))
    {
	  if ((iAction == _ADD) && (dStart < dEnd))
	  {
        btnAdd.setEnabled(true);
      }
      else if (iAction == _MODIFY)
      {
        btnAdd.setEnabled(true);
	  }
    }

    if (pDepth.getTotalRows() > 0)
    {
      btnModify.setEnabled(true);
      btnDRemove.setEnabled(true);
      btnDRemoveAll.setEnabled(true);
    }

    if (pTable.getTotalRows() > 0)
    {
      btnSelect.setEnabled(true);
      btnRemove.setEnabled(true);
      btnRemoveAll.setEnabled(true);
    }

    if (iAction2 == _MODIFY)
	{
      btnModify.setEnabled(false);
      btnDRemove.setEnabled(false);
      btnDRemoveAll.setEnabled(false);
	}

    if (iAction == _MODIFY)
	{
      btnSelect.setEnabled(false);
      btnRemove.setEnabled(false);
      btnRemoveAll.setEnabled(false);
	}
  }

  /* ===================================================================== *
   * -------------------- DISPLAY LOOKUP ACTIONS ------------------------- *
   * ===================================================================== */

  /** Method displayFossils()
   *  <p> This methos will display the General Fossil Group List Frame
   */

  private void displayFossils()
  {
	closeFossils();
	pFossil = new bioFossilsFrame( notifier, stStruct.stFossils );
  }

  /** Method displayICS()
   * <p> This method will show the International Stratigraphic Chart
   * @param ics = the Age Button selected
   */

  private void displayICS(int ics)
  {
	this.ics = ics;

    closeICS();
    pICS = new stratPlotICSFrame( notifier, 0, 0, 0, 0, stStruct.stICS );
  }


  /* ===================================================================== *
   * --------------------- SPECIES DEPTH ACTIONS ------------------------- *
   * ===================================================================== */

  /** Method modifyDepth()
   * <p> This method will modify the Depth data
   */

  private void modifyDepth()
  {
	int iRow = pDepth.getRow();

    setAction( _MODIFY, _MODIFY );

    sKEYa = new String( stModify.sKEYa[iRow] ); // Primary KEY of Depth Information

    switch (stModify.iAbundance[iRow])
    {
      case _NONE:
        rbNone.setSelected(true);     // No Fossils
        iAbundance = 0;
        break;
      case _SPARSE:
        rbSparse.setSelected(true);   // Sparse Fossils
        iAbundance = 1;
        break;
      case _PRESENT:
        rbPresent.setSelected(true);  // Fossils Present
        iAbundance = 2;
        break;
      case _COMMON:
        rbCommon.setSelected(true);   // Fossils Common
        iAbundance = 3;
        break;
      case _ABUNDANT:
        rbAbundant.setSelected(true); // Abundant Fossils
        iAbundance = 4;
        break;
      case _PROFUSE:
        rbProfuse.setSelected(true);  // Profuse Fossils
        iAbundance = 5;
        break;
	}

    dStart = stModify.depthStart[iRow];
    dEnd   = stModify.depthEnd[iRow];

    txtStart.setText( "" + stModify.depthStart[iRow] );   // Starting Depth
    txtEnd.setText( "" + stModify.depthEnd[iRow] );       // Ending Depth
  }

  /** Method removeDepth()
   * <p> This method will remove a specific depth range in the species list
   */

  private void removeDepth()
  {
	int iRow = pDepth.getRow();  // Get the Row Depth Range Primary KEY

    if (pDepth.getTotalRows() > 1)
    {
      // Remove Depth Range from Species

      stModify = horizon.bio.bioStratUtility.deleteDepth(
	  	           stModify, stModify.sKEYa[iRow] );

      // Reflect Change in Depth Range Table

      setData( stModify );

      // Modify Species Data in Species List

      stList   = horizon.bio.bioStratUtility.modify( sKEY, stModify, stList );

      // Show Change on Plot

      pNotifier.notifyObservers(new String("Bio Stratigraphy Changed"));
    }
    else
    {
	  removeDepths();
	}
  }

  /** Method removeDepths()
   * <p> This method will remove a specific depth range in the species list
   */

  private void removeDepths()
  {
	stModify.iRows = 0;
	stModify.iAbundance = null;
	stModify.depthStart = null;
	stModify.depthEnd   = null;

    // Reflect Change in Species List Table

    stList = horizon.bio.bioStratUtility.remove( sKEY, stList );

    pTable.setData( stList );

    clear();

    // Show Change on Plot

    pNotifier.notifyObservers(new String("Bio Stratigraphy Changed"));
  }

  /* ===================================================================== *
   * ---------------------- SPECIES LIST ACTIONS ------------------------- *
   * ===================================================================== */

  /** Method checkSpecies()
   * <p> This method will test if the Species is already present and
   *     if it is in the list the user would be requested to edit it.
   * @return iContinue = Continue with processing
   */

  public int checkSpecies()
  {
	int    iTest     = -1;
	int    iContinue = 1;
	int    iSpecies  = -1;
	String species   = txtSpecies.getText();

    iSpecies = horizon.bio.bioStratUtility.isSpecies( species, stList );

    if (iAction == _ADD)
    {
      if (iSpecies > -1)
      {
        iTest = JOptionPane.showConfirmDialog(
		          frame,
		          "This Species is already in your list, modify existing data?",
		          "Species Already Exists",
	  	          JOptionPane.YES_NO_OPTION );

        if (iTest == JOptionPane.YES_OPTION)
        {
          setAction( _MODIFY, _ADD );
          stModify  = horizon.bio.bioStratUtility.copy( stList.stItem[iSpecies] );
          setData( stModify );
        }
        else
        {
		  iContinue = 0;
		}
      }
    }

    return (iContinue);
  }

  /** Method add()
   * <p> This method will add the selection and depth to the bio
   *     stratigraphy List and plot it on the profile plot in the
   *     bio stratigraphy track.
   */

  private void add()
  {
    int iContinue = checkSpecies();

    if (iContinue == 1)
    {
      getData();

      switch (iAction)
      {
	    case _ADD:
  	      stList = horizon.bio.bioStratUtility.add( stModify, stList );
	      break;

	    case _MODIFY:
  	      stList = horizon.bio.bioStratUtility.modify( sKEY, stModify, stList );
	      break;
  	  }

      setData( stModify );

      stList = horizon.bio.bioStratUtility.bubbleSort(stList);
      pTable.setData( stList );

      clearDepth();

      pNotifier.notifyObservers(new String("Bio Stratigraphy Changed"));
    }
    else
    {
        JOptionPane.showMessageDialog((Component) null,
                                       "Action Aborted",
                                       "WARNING",
                                       JOptionPane.WARNING_MESSAGE);
	}
  }

  /** Method modify()
   * <p> This method will modify the data
   */

  private void modify()
  {
    setAction( _MODIFY, _ADD );

    stModify = horizon.bio.bioStratUtility.copy(pTable.getRowData());

    setData( stModify );
  }

  /** Method removeItem()
   * <p> This method will remove a record from the list
   */

  private void removeItem()
  {
    bioStratStruct st = horizon.bio.bioStratUtility.copy( pTable.getRowData() );

    stList = horizon.bio.bioStratUtility.remove( st.sKEY, stList );

    pTable.setData( stList );

    pNotifier.notifyObservers(new String("Bio Stratigraphy Changed"));
  }

  /** Method removeAllRows()
   * <p> This method will remove all records from the list
   */

  private void removeAllRows()
  {
    if (stList != null)
      stList.delete();
    stList = null;

    pTable.setData( stList );

    pNotifier.notifyObservers(new String("Bio Stratigraphy Changed"));
  }

  /* ===================================================================== *
   * ---------------------- CLEAR FIELDS ACTIONS ------------------------- *
   * ===================================================================== */

  /** Method clear()
   * <p> This method will clear the data fields
   */

  public void clear()
  {
    if (stModify != null)
      stModify.delete();
    stModify = null;

    if (pDepth != null) { pDepth.setData( stModify ); }

	// Clear Text fields

    sKEY  = new String( "0" );   // Primary KEY of Species
    sKEYa = new String( "0" );   // Primary KEY of Depth Information

    txtKingdom.setText("");      // Kingdom
    txtSubkingdom.setText("");   // Subkingdom

    txtSuperphylum.setText("");  // Superphylum
    txtPhylum.setText("");       // Phylum
    txtSubphylum .setText("");   // Subphylum

    txtSuperclass.setText("");   // Superclass
    txtClass.setText("");        // Class
    txtSubclass.setText("");     // Subclass

    txtSuperorder.setText("");   // Superorder
    txtOrder.setText("");        // Order
    txtSuborder.setText("");     // Suborder
    txtInfraorder.setText("");   // Infraorder

    txtSuperfamily.setText("");  // Superfamily
    txtFamily.setText("");       // Family
    txtSubfamily.setText("");    // Subfamily

    txtTribe.setText("");        // Tribe

    txtGenus.setText("");        // Genus
    txtSpecies.setText("");      // Species

    dStart = 0.0;
    dEnd   = 0.0;

    txtStart.setText("0.0");     // Starting Depth
    txtEnd.setText("0.0");       // Ending Depth

    sFossilID = new String("10.2.1");       // Fossil identifier
    sFossil   = new String("macrofossils"); // Generic Fossil Name,
    txtGroup.setText(sFossil);              // Fossil Group

    // -- Earliest geological time

    dFrom        = 0.0; // From Age in Ma
    sKey_e       = "";  // Stratigraphic Unit Unique KEY
    system_e     = "";  // System Stratigraphic Unit Names
    series_e     = "";  // Series Stratigraphic Unit Names
    subSystem_e  = "";  // Substage Stratigraphic Unit Names
    subSeries_e  = "";  // Subseries Stratigraphic Unit Names

    // -- Latest geological time

    dTo          = 0.0; // To Age in Ma
    sKey_l       = "";  // Stratigraphic Unit Unique KEY
    system_l     = "";  // System Stratigraphic Unit Names
    series_l     = "";  // Series Stratigraphic Unit Names
    subSystem_l  = "";  // Substage Stratigraphic Unit Names
    subSeries_l  = "";  // Subseries Stratigraphic Unit Names

    txtFrom.setText("0.0");      // From Age
    txtTo.setText("0.0");        // To Age

    btnFrom.setBackground(new Color(220, 220, 220));
    btnFrom.setText("Age From");

    btnTo.setBackground(new Color(220, 220, 220));
    btnTo.setText("Age To");

    rbNone.setSelected(true);    // No Fossils

    setAction( _ADD, _ADD );
  }

  /** Method clearDepth()
   * <p> This method will clear the Depth data fields
   */

  public void clearDepth()
  {
    sKEYa = new String( "0" );   // Primary KEY of Depth Information

    dStart     = 0.0;
    dEnd       = 0.0;
    iAbundance = 0;

    txtStart.setText("0.0");     // Starting Depth
    txtEnd.setText("0.0");       // Ending Depth
    rbNone.setSelected(true);    // No Fossils

    setAction( _MODIFY, _ADD );
  }

  /* ===================================================================== *
   * ------------------------- CLOSE ACTIONS ----------------------------- *
   * ===================================================================== */

  /** METHOD bioDataEntryFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class bioDataEntryFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /** Method closeFossils()
   * <p> This method will close the General Fossils Dialog)
   */

  public void closeFossils()
  {
    if (pFossil != null)
      pFossil.close();
    pFossil = null;
  }


  /** Method closeICS()
   * <p> This method will close the ICS Dialog)
   */

  public void closeICS()
  {
    if (pICS != null)
      pICS.close();
    pICS = null;
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    // Input Variables

    pNotifier = null;  // Calling Frame Observable
    stStruct  = null;  // Global IQSTRAT Data Structure

    // Global Variables

    notifier  = null;  // This Frame Observable

    // Action Variables

    sKEY      = null;  // KEY to Data
    sKEYa     = null;  // KEY to Depth Range Data

    // Bio Stratigraphy List Table

    if (pTable != null)
      pTable.close();
    pTable = null;  // Species List

    if (pDepth != null)
      pDepth.close();
    pDepth = null;  // Individual Species Depth Ranges

    // Generic Rock Fossil Identifier

    sFossilID    = null; // Fossil identifier
    sFossil      = null; // Generic Fossil Name,

    // Bio Stratigraphy Structures

    stList   = null;

    if (stModify != null)
      stModify.delete();
    stModify = null;

    closeFossils();
    closeICS();

    // Global Frame Widges

    txtKingdom     = null; // Kingdom
    txtSubkingdom  = null; // Subkingdom

    txtSuperphylum = null; // Superphylum
    txtPhylum      = null; // Phylum
    txtSubphylum   = null; // Subphylum

    txtSuperclass  = null; // Superclass
    txtClass       = null; // Class
    txtSubclass    = null; // Subclass

    txtSuperorder  = null; // Superorder
    txtOrder       = null; // Order
    txtSuborder    = null; // Suborder
    txtInfraorder  = null; // Infraorder

    txtSuperfamily = null; // Superfamily
    txtFamily      = null; // Family
    txtSubfamily   = null; // Subfamily

    txtTribe       = null; // Tribe

    txtGenus       = null; // Genus
    txtSpecies     = null; // Species

    txtGroup       = null; // Group
    btnGroup       = null; // Change Group

    // -- Species Abundance

    rbNone        = null;  // No Fossils
    rbSparse      = null;  // Sparse Fossils
    rbPresent     = null;  // Fossils Present
    rbCommon      = null;  // Fossils Common
    rbAbundant    = null;  // Abundant Fossils
    rbProfuse     = null;  // Profuse Fossils

    // -- Depth Range Textfields

    txtStart      = null;  // Starting Depth
    txtEnd        = null;  // Ending Depth

    // -- Depth Range Table Buttons

    btnDClear     = null;  // Clear Depth Edit Fields
    btnDAdd       = null;  // Add/Modify Depth Information

    btnModify     = null;  // Modify Depth Range
    btnDRemove    = null;  // Remove Depth Range
    btnDRemoveAll = null;  // Remove all Depth Ranges

  // -- Age Limits Widgets

    txtFrom       = null;  // From Age
    txtTo         = null;  // To Age

    btnFrom       = null;  // From Age Button
    btnTo         = null;  // To Age Button

  // -- Global Panel Buttons

    btnClear      = null;  // Clear Species Edit Fields
    btnAdd        = null;  // Add/Modify Species Information

    btnSelect     = null;  // Modify Species
    btnRemove     = null;  // Remove Species
    btnRemoveAll  = null;  // Remove All Species

    btnCancel     = null;  // Exit Dialog

    dispose();
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == rbNone)     { iAbundance = 0; }    // No Fossils
    if (event.getSource() == rbSparse)   { iAbundance = 1; }    // Sparse Fossils
    if (event.getSource() == rbPresent)  { iAbundance = 2; }    // Fossils Present
    if (event.getSource() == rbCommon)   { iAbundance = 3; }    // Fossils Common
    if (event.getSource() == rbAbundant) { iAbundance = 4; }    // Abundant Fossils
    if (event.getSource() == rbProfuse)  { iAbundance = 5; }    // Profuse Fossils

	if (event.getSource() == btnGroup) { displayFossils(); }    // General Fossil Group
    if (event.getSource() == btnFrom)  { displayICS(_FROM); }   // From Age Button
    if (event.getSource() == btnTo)    { displayICS(_TO); }     // To Age Button

    // Depth Panel Buttons

    if (event.getSource() == btnDAdd)       { add(); }          // Add Depth
    if (event.getSource() == btnModify)     { modifyDepth(); }  // Modify Depth Info
    if (event.getSource() == btnDClear)     { clearDepth(); }   // Clear Depth Info
    if (event.getSource() == btnDRemove)    { removeDepth(); }  // Remove Depth Range
    if (event.getSource() == btnDRemoveAll) { removeDepths(); } // Remove all Depths

    // Data List Buttons

    if (event.getSource() == btnAdd)       { add(); }           // Add Species
    if (event.getSource() == btnClear)     { clear(); }         // Clear Species Info

    if (event.getSource() == btnSelect)    { modify(); }        // Modify Species Data
    if (event.getSource() == btnRemove)    { removeItem(); }    // Remove Species Data
    if (event.getSource() == btnRemoveAll) { removeAllRows(); } // Remove All Species

    setButtons();

    if (event.getSource() == btnCancel) { close(); }            // Exit Dialog

  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtGenus)   // Genus
    {
	  if (txtSpecies.getText().length() == 0)
	  {
		txtSpecies.setText( txtGenus.getText() + " (sp)" );
	  }

	  if (rbNone.isSelected())
	  {
		rbPresent.setSelected(true);
	  }
	}

    if (e.getSource() == txtSpecies) // Species
    {
      checkSpecies();

	  if (rbNone.isSelected())
	  {
		rbPresent.setSelected(true);
	  }
	}

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
//        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

//        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());
      }
    }

    setButtons();
  }

  /* ===================================================================== *
   * ---------------------- OBSERVABLES ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg      = new String((String) arg);
    if (sArg.equals("ICS Stratigraphic Unit Selected")) { getICSData(); }
    if (sArg.equals("Fossil Selected")) { getFossilGroup(); }
  }

}

/** CLASS bioDataEntryFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class bioDataEntryFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  bioDataEntryFrame adaptee;

  bioDataEntryFrameFocusAdapter(bioDataEntryFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/** CLASS bioDataEntryFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class bioDataEntryFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 11/15/2011
 *  @author  John Victorine
 */
