/*
 * @bioStratUtility.java Version 1.1 11/05/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.bio;

import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;

/** Class bioStratUtility
 *  <p> This Class will provide basic utilities for the Bio-Stratigraphic Unit
 *      data structures.
 *
 *  @version 1.1 11/05/2011
 *  @author  John R. Victorine
 */

public class bioStratUtility
{
  public static final int _PLOT_SP = 35;  // Width of each Species Width on Plot

  /** Method copyList()
   * <p> This method will copy one bio-Stratigraphic unit list structure to another
   * @param  stOld = Old bio-Stratigraphic unit list data structure
   * @return stNew = New bio-Stratigraphic unit list data structure
   */

  public static bioStratListStruct copyList( bioStratListStruct stOld )
  {
    bioStratListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew = new bioStratListStruct();

        // LAS 3 Dependent Variable

        stNew.iSource    = stOld.iSource;
        stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
        stNew.source     = new String( stOld.source );    // Source of Data
        stNew.sReference = new String( stOld.sReference );  // Created Date

        // Data List

        stNew.iCount = stOld.iCount;
        stNew.stItem = new bioStratStruct[stOld.iCount];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy( stOld.stItem[i] );
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static bioStratListStruct transfer(bioStratListStruct stOld)
  {
    bioStratListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy one bio-Stratigraphic unit structure to another
   * @param  stOld = Old bio-Stratigraphic unit data structure
   * @return stNew = New bio-Stratigraphic unit data structure
   */

  public static bioStratStruct copy( bioStratStruct stOld )
  {
    bioStratStruct stNew = null;

    if (stOld != null)
    {
      stNew             = new bioStratStruct();

      // Unique Fossil Entry

      stNew.sKID        = new String( stOld.sKID );
      stNew.sKEY        = new String( stOld.sKEY );

      stNew.sName       = new String( stOld.sName );

      stNew.iRows       = stOld.iRows;
      if (stOld.iRows > 0)
      {
		stNew.sKEYa       = new String[stOld.iRows];
        stNew.iAbundance  = new int[stOld.iRows];
        stNew.depthStart  = new double[stOld.iRows];
        stNew.depthEnd    = new double[stOld.iRows];

        for (int i=0; i<stOld.iRows; i++)
        {
		  stNew.sKEYa[i]      = new String( stOld.sKEYa[i] );
          stNew.iAbundance[i] = stOld.iAbundance[i];
          stNew.depthStart[i] = stOld.depthStart[i];
          stNew.depthEnd[i]   = stOld.depthEnd[i];
	    }
      }

      // Fossil Scientific Name

      stNew.species     = new String( stOld.species );
      stNew.sGenus      = new String( stOld.sGenus );
      stNew.sTribe      = new String( stOld.sTribe );
      stNew.subfamily   = new String( stOld.subfamily );
      stNew.sFamily     = new String( stOld.sFamily );
      stNew.superfamily = new String( stOld.superfamily );
      stNew.sInfraorder = new String( stOld.sInfraorder );
      stNew.suborder    = new String( stOld.suborder );
      stNew.sOrder      = new String( stOld.sOrder );
      stNew.superorder  = new String( stOld.superorder );
      stNew.subclass    = new String( stOld.subclass );
      stNew.sClass      = new String( stOld.sClass );
      stNew.superclass  = new String( stOld.superclass );
      stNew.subphylum   = new String( stOld.subphylum );
      stNew.sPhylum     = new String( stOld.sPhylum );
      stNew.superphylum = new String( stOld.superphylum );
      stNew.subkingdom  = new String( stOld.subkingdom );
      stNew.sKingdom    = new String( stOld.sKingdom );
      stNew.sAuthor     = new String( stOld.sAuthor );

      // Generic Rock Fossil Identifier

      stNew.sFossilID   = new String( stOld.sFossilID );
      stNew.sFossil     = new String( stOld.sFossil );

      // Stratigraphic Unit Range

      // -- Earliest geological time

      stNew.dFrom       = stOld.dFrom; // From Age in Ma
      stNew.sKey_e      = new String( stOld.sKey_e );
      stNew.system_e    = new String( stOld.system_e );
      stNew.series_e    = new String( stOld.series_e );
      stNew.subSystem_e = new String( stOld.subSystem_e );
      stNew.subSeries_e = new String( stOld.subSeries_e );

      // -- Latest geological time

      stNew.dTo         = stOld.dTo; // From Age in Ma
      stNew.sKey_l      = new String( stOld.sKey_l );
      stNew.system_l    = new String( stOld.system_l );
      stNew.series_l    = new String( stOld.series_l );
      stNew.subSystem_l = new String( stOld.subSystem_l );
      stNew.subSeries_l = new String( stOld.subSeries_l );
    }

    return (stNew);
  }

  /** Method isSpecies()
   * <p> This method will test if the Species is already present and
   *     return the position in the list.
   * @param  species = the species name (Genera species) or (Genera (sp))
   * @param  st      = The bio-Stratigraphic unit List Data Structure
   * @return iValue  = true species already in list or false it is not
   */

  public static int isSpecies( String species, bioStratListStruct st )
  {
	int     iValue = -1;
	String  sText  = new String( species.toLowerCase() );

	if (st != null)
	{
	  for (int i=0; i<st.iCount; i++)
	  {
		if ((sText.equals(st.stItem[i].sName.toLowerCase())) ||
		    (sText.equals(st.stItem[i].species.toLowerCase())) )
		{
		  iValue = i;
	    }
	  }
	}

	return (iValue);
  }

  /** Method exists()
   * <p> This method will test if the Fossil Genera Name exists in the data
   *     structure.
   * @param  sGenera = Genera name being tested
   * @param  st      = The bio-Stratigraphic unit List Data Structure
   * @return bValue  = true, genera exists in list or false it is not
   */

  public static boolean exists( String sGenera, bioStratListStruct st )
  {
	boolean bValue = false;
	String  sText  = new String( sGenera.toLowerCase() );

	if (st != null)
	{
	  for (int i=0; i<st.iCount; i++)
	  {
		if (sText.equals(st.stItem[i].sGenus.toLowerCase()))
		{
		  bValue = true;
	    }
	  }
	}

	return (bValue);
  }

  /** Method getPosition()
   * <p> This method will identify the position of the genera term in list.
   * @param  sGenera = Genera name being tested
   * @param  st      = The bio-Stratigraphic unit List Data Structure
   * @return iValue  = the position of the genera term in the list
   */

  public static int getPosition( String sGenera, bioStratListStruct st )
  {
	int iValue = -1;
	String  sText  = new String( sGenera.toLowerCase() );

	if (st != null)
	{
	  for (int i=0; i<st.iCount; i++)
	  {
		if (sText.equals(st.stItem[i].sGenus.toLowerCase()))
		{
		  iValue = i;
	    }
	  }
	}

	return (iValue);
  }

  /** Method add()
   * <p> This method will add the bio-Stratigraphic unit to an existing list
   * @param  stBio = The Region Data Structure
   * @param  st    = The Old bio-Stratigraphic unit List Data Structure
   * @return st    = The new bio-Stratigraphic unit List Data Structure.
   */

  public static bioStratListStruct add( bioStratStruct stBio,
                                        bioStratListStruct st )
  {
    int    i=0;
    int    iRecords   = 0;
    int    iCount     = 0;
    bioStratListStruct stTemp = null;
    int    iSource    = -1;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Created Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new bioStratListStruct();
    stTemp.stItem = new bioStratStruct[iRecords];

    if (st != null)
    {
	  iSource    = st.iSource;
      sKGS       = new String( st.sKGS );      // KGS Saved Data Indicator
      source     = new String( st.source );    // Source of Data
      sReference = new String( st.sReference );  // Created Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new bioStratStruct();
    stTemp.stItem[iCount] = copy( stBio );
    iCount++;

    stTemp.iCount = iCount;

    st            = new bioStratListStruct();
    st.stItem     = new bioStratStruct[iCount];
    st.iCount     = iCount;

    st.iSource    = iSource;
    st.sKGS       = new String( sKGS );      // KGS Saved Data Indicator
    st.source     = new String( source );    // Source of Data
    st.sReference = new String( sReference );  // Created Date

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method addDepth()
   * <p> This method will add a depth range to a specific species
   * @param  st         = The Fossil Data Structure
   * @param  sKEYa      = Unique Key for depth range
   * @param  iAbundance = Abundance inicator
   * @param  depthStart = Starting depth
   * @param  depthEnd   = Ending depth
   * @return st         = The Fossil Data Structure.
   */

  public static bioStratStruct addDepth( bioStratStruct st,
                                         String         sKEYa,
                                         int            iAbundance,
                                         double         depthStart,
                                         double         depthEnd )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    String sKEY[]   = null;  // Array of Unique Key for each Depth Sample
    int    iAbund[] = null;  // Array of Abundance indicators
    double dStart[] = null;  // Array of Starting Depths
    double dEnd[]   = null;  // Array if Ending Depths

    if (st != null)
    {
      iRecords = st.iRows+1;
      sKEY     = new String[iRecords];
      iAbund   = new int[iRecords];
      dStart   = new double[iRecords];
      dEnd     = new double[iRecords];

      for (i=0; i<st.iRows; i++)
      {
        if (iCount < iRecords)
        {
		  sKEY[iCount]   = new String( st.sKEYa[i] );
		  iAbund[iCount] = st.iAbundance[i];
		  dStart[iCount] = st.depthStart[i];
		  dEnd[iCount]   = st.depthEnd[i];
          iCount++;
        }
      }

      sKEY[iCount]   = new String( sKEYa );
	  iAbund[iCount] = iAbundance;
	  dStart[iCount] = depthStart;
	  dEnd[iCount]   = depthEnd;
      iCount++;

      st.iRows       = 0;
      st.sKEYa       = null;
      st.iAbundance  = null;
      st.depthStart  = null;
      st.depthEnd    = null;

      st.iRows       = iCount;
      st.sKEYa       = new String[iCount];
      st.iAbundance  = new int[iCount];
      st.depthStart  = new double[iCount];
      st.depthEnd    = new double[iCount];

      for (i=0; i<st.iRows; i++)
      {
        st.sKEYa[i]      = new String( sKEY[i] );
		st.iAbundance[i] = iAbund[i];
		st.depthStart[i] = dStart[i];
		st.depthEnd[i]   = dEnd[i];
      }
    }

    return (st);
  }

  /** Method modifyDepth()
   *  <p> this method will modify a depth range to a specific species
   * @param  st         = The Fossil Data Structure
   * @param  sKEYa      = Unique Key for depth range
   * @param  iAbund     = Abundance of species
   * @param  dStart     = Starting Depth
   * @param  dEnd       = Ending Depth
   * @return st         = The Fossil Data Structure.
   */

  public static bioStratStruct modifyDepth( bioStratStruct st,
                                            String         sKEYa,
                                            int            iAbund,
                                            double         dStart,
                                            double         dEnd )
  {
	if (st != null)
	{
      for (int i=0; i<st.iRows; i++)
      {
        if (sKEYa.equals(st.sKEYa[i]))
        {
          st.iAbundance[i] = iAbund;
          st.depthStart[i] = dStart;
  	      st.depthEnd[i]   = dEnd;
        }
      }
    }

    return (st);
  }

  /** Method deleteDepth()
   * <p> This method will add a depth range to a specific species
   * @param  st         = The Fossil Data Structure
   * @param  sKEYa      = Unique Key for depth range
   * @return st         = The Fossil Data Structure.
   */

  public static bioStratStruct deleteDepth( bioStratStruct st,
                                            String         sKEYa )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    String sKEY[]   = null;  // Array of Unique Key for each Depth Sample
    int    iAbund[] = null;  // Array of Abundance indicators
    double dStart[] = null;  // Array of Starting Depths
    double dEnd[]   = null;  // Array if Ending Depths

    if (st != null)
    {
	  if (st.iRows < 2)
	  {
        st.iRows       = 0;
        st.sKEYa       = null;
        st.iAbundance  = null;
        st.depthStart  = null;
        st.depthEnd    = null;
	  }
	  else
	  {
        iRecords = st.iRows-1;
        sKEY     = new String[iRecords];
        iAbund   = new int[iRecords];
        dStart   = new double[iRecords];
        dEnd     = new double[iRecords];

        for (i=0; i<st.iRows; i++)
        {
          if ((iCount < iRecords) && (!sKEYa.equals(st.sKEYa[i])))
          {
		    sKEY[iCount]   = new String( st.sKEYa[i] );
		    iAbund[iCount] = st.iAbundance[i];
		    dStart[iCount] = st.depthStart[i];
		    dEnd[iCount]   = st.depthEnd[i];
            iCount++;
          }
        }

        st.iRows       = 0;
        st.sKEYa       = null;
        st.iAbundance  = null;
        st.depthStart  = null;
        st.depthEnd    = null;

        st.iRows       = iCount;
        st.sKEYa       = new String[iCount];
        st.iAbundance  = new int[iCount];
        st.depthStart  = new double[iCount];
        st.depthEnd    = new double[iCount];

        for (i=0; i<st.iRows; i++)
        {
          st.sKEYa[i]      = new String( sKEY[i] );
		  st.iAbundance[i] = iAbund[i];
	 	  st.depthStart[i] = dStart[i];
	  	  st.depthEnd[i]   = dEnd[i];
        }
      }
    }

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Bio Stratigrapy in an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  stNew  = Bio Stratigrapy data structure
   * @param  st     = Bio Stratigrapy data list structure
   * @return st     = New Bio Stratigrapy data list structure
   */

  public static bioStratListStruct modify( String             sKEY,
                                           bioStratStruct     stNew,
                                           bioStratListStruct st )
  {
    int i=0;
    int    iSource    = -1;
    bioStratListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Created Date

    if (st != null)
    {
	  iSource    = st.iSource;

      sKGS       = new String( st.sKGS );      // KGS Saved Data Indicator
      source     = new String( st.source );    // Source of Data
      sReference = new String( st.sReference );  // Created Date

      stTemp        = new bioStratListStruct();
      stTemp.stItem = new bioStratStruct[st.iCount];
      stTemp.iCount = st.iCount;

      for (i=0; i<st.iCount; i++)
      {
        if (sKEY.equals(st.stItem[i].sKEY))
          stTemp.stItem[i] = copy(stNew);
        else
          stTemp.stItem[i] = copy(st.stItem[i]);
      }

      st.delete();

      st            = new bioStratListStruct();
      st.stItem     = new bioStratStruct[stTemp.iCount];
      st.iCount     = stTemp.iCount;

      st.iSource    = iSource;

      st.sKGS       = new String( sKGS );      // KGS Saved Data Indicator
      st.source     = new String( source );    // Source of Data
      st.sReference = new String( sReference );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Bio Stratigrapy from an existing list
   * @param  sKEY   = The User Created Primary Key of the data to be removed
   * @param  st     = Bio Stratigrapy data list structure
   * @return st     = New Bio Stratigrapy data list structure
   */

  public static bioStratListStruct remove( String sKEY, bioStratListStruct st )
  {
    int i=0;
    int    iRecords   = 0;
    int    iCount     = 0;
    int    iFound     = -1;
    int    iSource    = -1;
    bioStratListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    String source     = "";    // Source of Data
    String sReference = "";    // Created Date

    if (st != null)
    {
      if (st.iCount == 1)
      {
        st.delete();
        st = null;
      }
      else
      {
        iRecords = st.iCount-1;

        stTemp        = new bioStratListStruct();
        stTemp.stItem = new bioStratStruct[iRecords];

        iSource    = st.iSource;

        sKGS       = new String( st.sKGS );      // KGS Saved Data Indicator
        source     = new String( st.source );    // Source of Data
        sReference = new String( st.sReference );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          if ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0")))
          {
            iFound = i;
          }
        }

        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();

        stTemp.iCount = iCount;

        st            = new bioStratListStruct();
        st.stItem     = new bioStratStruct[iCount];
        st.iCount     = iCount;

        st.iSource    = iSource;

        st.sKGS       = new String( sKGS );      // KGS Saved Data Indicator
        st.source     = new String( source );    // Source of Data
        st.sReference = new String( sReference );  // Created Date

        for (i=0; i<stTemp.iCount; i++)
        {
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        stTemp.delete();
      }
    }

    return (st);
  }

  /** Method convert( String sGenus, bioStratListStruct st )
   * <p> This method will convert the genus name to a common name
   * @param  sGenus = genus name of fossil
   * @param  st     = bio-stratigraphy data list structure
   * @return sName  = Common Name for the fossil
   */

  public static String convert( String sGenus, bioStratListStruct st )
  {
    String sName = new String(sGenus.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if (sName.equals(st.stItem[i].sGenus.toLowerCase()))
        {
          sName = new String( st.stItem[i].sFossil );
        }
      }
    }

    return (sName);
  }

  /** Method computeWidth( bioStratListStruct st )
   * <p> This method will compute the plot track width from the total number of
   *     objects in the observation data
   * @param  st     = the bio stratigraphy data list structure
   * @return iWidth = the plot track width
   */

  public static int computeWidth( bioStratListStruct st )
  {
    int iWidth = 10;

    if (st != null)
    {
	  iWidth = (st.iCount+1) * _PLOT_SP;
	  if (iWidth == 0) { iWidth = 10; }
	}

    return (iWidth);
  }

  /** Method getAbundance()
   *  <p> This method will determine if the term represents a quantity term
   *  @param  sTerm = the term being tested
   *  @return iNo   = the number of fossils
   */

  public static int getAbundance( String sTerm )
  {
	int iNo = horizon.bio.bioStratStruct._NONE;

	if (sTerm.equals("sparse"))
	  iNo = horizon.bio.bioStratStruct._SPARSE;
    if (sTerm.equals("present"))
	  iNo = horizon.bio.bioStratStruct._PRESENT;
    if (sTerm.equals("common"))
	  iNo = horizon.bio.bioStratStruct._COMMON;
    if (sTerm.equals("abundant"))
	  iNo = horizon.bio.bioStratStruct._ABUNDANT;
    if (sTerm.equals("profuse"))
	  iNo = horizon.bio.bioStratStruct._PROFUSE;

	return (iNo);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static bioStratListStruct bubbleSort(bioStratListStruct st)
  {
    boolean swappedOnPrevRun = true;
    bioStratStruct stTemp    = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].sFossilID.compareTo(st.stItem[i+1].sFossilID) > 0)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print( bioStratListStruct st )
   * <p> This method will print the contents of the Bio stratigraphic units
   * @param  st = bio-Stratigraphic unit list data structure
   */

  public static void print( bioStratListStruct st )
  {
    String sAge_e = "";
    String sAge_l = "";

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println( st.stItem[i].sGenus + " " +
                            st.stItem[i].sClass + " " +
                            st.stItem[i].sPhylum );

        if (st.stItem[i].sKey_e.equals("0"))
          sAge_e = new String( "Recent" );
        else if (st.stItem[i].subSystem_e.length() > 0)
          sAge_e = new String( st.stItem[i].subSeries_e + " " +
                               st.stItem[i].subSystem_e );
        else
          sAge_e = new String( st.stItem[i].series_e + " " +
                               st.stItem[i].system_e );

        if (st.stItem[i].subSystem_l.length() > 0)
          sAge_l = new String( st.stItem[i].subSeries_l + " " +
                               st.stItem[i].subSystem_l );
        else
          sAge_l = new String( st.stItem[i].series_l + " " +
                               st.stItem[i].system_l );

        System.out.println("   from " + sAge_l + " to " + sAge_e);
        System.out.println("   " + st.stItem[i].sFossil + "\n");
      }
    }
  }

  /** Method printSpecies( bioStratListStruct st )
   * <p> This method will print the contents of the species list
   * @param  st = bio-Stratigraphic unit list data structure
   */

  public static void printSpecies( bioStratListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
		System.out.println( st.stItem[i].sKEY + " " +
		                    st.stItem[i].sName + " " +
		                    st.stItem[i].sFossilID + " " +
		                    st.stItem[i].sFossil );

		for (int j=0; j<st.stItem[i].iRows; j++)
		{
		  System.out.println( " -- " + st.stItem[i].sKEYa[j] + " " +
		                      st.stItem[i].sAbundance[st.stItem[i].iAbundance[j]] + " " +
		                      st.stItem[i].depthStart[j] + " " +
		                      st.stItem[i].depthEnd[j] );
		}
	  }
    }
  }
}

/*
 *  @version 1.1 11/05/2011
 *  @author  John Victorine
 */
