/*
 * @bioStandardTools.java Version 1.1 11/29/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.bio;

/** Class bioStandardTools
 *  <p> This Class holds the standard tool information to map the
 *      Bio-Stratigraphic LAS File curves to standard names.
 */

public class bioStandardTools
{
  // Species Depth Range and abundance

  public static final int _SP    = 0;  // Species",
  public static final int _ABUN  = 1;  // Abundance, (sparse, common, etc.)",
  public static final int _STRT  = 2;  // Starting Depth",
  public static final int _END   = 3;  // Ending Depth",

    // Species Classification

  public static final int _GENUS = 4;  // Genus",
  public static final int _TRIBE = 5;  // Tribe",

  public static final int _SBFAM = 6;  // Subfamily",
  public static final int _FAM   = 7;  // Family",
  public static final int _SPFAM = 8;  // Superfamily",

  public static final int _INORD = 9;  // Infraorder",
  public static final int _SBORD = 10; // Suborder",
  public static final int _ORDER = 11; // Order",
  public static final int _SPORD = 12; // Superorder",

  public static final int _SBCLS = 13; // Subclass",
  public static final int _CLASS = 14; // Class",
  public static final int _SPCLS = 15; // Superclass",

  public static final int _SBPHY = 16; // Subphylum",
  public static final int _PHYLM = 17; // Phylum",
  public static final int _SPPHY = 18; // Superphylum",

  public static final int _SBKNG = 19; // Subkingdom",
  public static final int _KNGDM = 20; // Kingdom",

  public static final int _AUTHR = 21; // Author",

  public static final int _GENID = 22; // Generic Fossil ID",
  public static final int _GENRC = 23; // Generic Fossil Name",

  public static final int _FROM  = 24; // Age From",
  public static final int _KEYE  = 25; // Stratigraphic Unit KEY",
  public static final int _SYSE  = 26; // System stratigraphic unit name",
  public static final int _SERE  = 27; // Series stratigraphic unit name",
  public static final int _SSYSE = 28; // Subsystem stratigraphic unit name",
  public static final int _SSERE = 29; // Subseries stratigraphic unit name",

  public static final int _TO    = 30; // Age To",
  public static final int _KEYL  = 31; // Stratigraphic Unit KEY",
  public static final int _SYSL  = 32; // System stratigraphic unit name",
  public static final int _SERL  = 33; // Series stratigraphic unit name",
  public static final int _SSYSL = 34; // Subsystem stratigraphic unit name",
  public static final int _SSERL = 35; // Subseries stratigraphic unit name",

  public static final int TOTAL  = 36; // Total Possible Columns

  // Curve Column Numbers

  public static final int _C_ID     = 0;
  public static final int _C_MNEM   = 1;
  public static final int _C_NAME   = 2;
  public static final int _C_UNIT   = 3;

  public static final String CURVES[][] =
  {
    // ID    MNEMONIC  CURVE NAME                          UNIT
    // Species Depth Range and abundance
    { "BIO1010", "SP",    "Species",                           "" },
    { "BIO1011", "ABUN",  "Abundance, (sparse, common, etc.)", "" },
    { "BIO1012", "STRT",  "Starting Depth",                    "F" },
    { "BIO1013", "END",   "Ending Depth",                      "F" },

    // Species Classification
    { "BIO2010", "GENUS", "Genus",                             "" },
    { "BIO2011", "TRIBE", "Tribe",                             "" },

    { "BIO2021", "SBFAM", "Subfamily",                         "" },
    { "BIO2022", "FAM",   "Family",                            "" },
    { "BIO2023", "SPFAM", "Superfamily",                       "" },

    { "BIO2031", "INORD", "Infraorder",                        "" },
    { "BIO2032", "SBORD", "Suborder",                          "" },
    { "BIO2033", "ORDER", "Order",                             "" },
    { "BIO2034", "SPORD", "Superorder",                        "" },

    { "BIO2041", "SBCLS", "Subclass",                          "" },
    { "BIO2042", "CLASS", "Class",                             "" },
    { "BIO2043", "SPCLS", "Superclass",                        "" },

    { "BIO2051", "SBPHY", "Subphylum",                         "" },
    { "BIO2052", "PHYLM", "Phylum",                            "" },
    { "BIO2053", "SPPHY", "Superphylum",                       "" },

    { "BIO2151", "SBKNG", "Subkingdom",                        "" },
    { "BIO2152", "KNGDM", "Kingdom",                           "" },

    { "BIO2061", "AUTHR", "Author",                            "" },

    { "BIO2071", "GENID", "Generic Fossil ID",                 "" },
    { "BIO2072", "GENRC", "Generic Fossil Name",               "" },

    { "BIO2081", "FROM",  "Age From",                          "Ma" },
    { "BIO2082", "KEYE",  "Stratigraphic Unit KEY",            "" },
    { "BIO2083", "SYSE",  "System stratigraphic unit name",    "" },
    { "BIO2084", "SERE",  "Series stratigraphic unit name",    "" },
    { "BIO2085", "SSYSE", "Subsystem stratigraphic unit name", "" },
    { "BIO2086", "SSERE", "Subseries stratigraphic unit name", "" },

    { "BIO2091", "TO",    "Age To",                            "Ma" },
    { "BIO2092", "KEYL",  "Stratigraphic Unit KEY",            "" },
    { "BIO2093", "SYSL",  "System stratigraphic unit name",    "" },
    { "BIO2094", "SERL",  "Series stratigraphic unit name",    "" },
    { "BIO2095", "SSYSL", "Subsystem stratigraphic unit name", "" },
    { "BIO2096", "SSERL", "Subseries stratigraphic unit name", "" },
  };

  /** Method isCurve()
   * <p> This method will determine if the curve is part of the standard
   *     bio stratigraphic units curves from the LAS File.
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return bValue  = true it is a valid curve, false it is not
   */

  public static boolean isCurve( int iColumn, String str )
  {
    boolean bValue = false;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method getRow()
   * <p> This method will return the row number of the
   *     Bio Stratigraphic Unit Curve
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return iRow = row number
   */

  public static int getRow( int iColumn, String str )
  {
    int iRow = -1;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        iRow = i;
      }
    }

    return (iRow);
  }
}

/*
 *  @version 1.1 11/29/2011
 *  @author  John Victorine
 */
