/*
 * @guiSearchDirectoryFrame.java Version 1.1 12/10/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package gui;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Observable;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class guiSearchDirectoryFrame
 *  <p> This Class will search the user's directory to save the files.
 *
 *  @version 1.1 12/10/2009
 *  @author  John R. Victorine
 */

public class guiSearchDirectoryFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable notifier = null; // Observable Object
  private String sDirectory = "";
  private String sFile      = "";
  private String sImage     = ".png";
  private String sType      = ".pdf";

  // Global Frame Widgets

  private JButton btnContinue = new JButton();  // Send Message Path Selected
  private JButton btnCancel   = new JButton();  // Cancel Search
  private JButton btnSearch   = new JButton();  // Search for Directory Path

  private JTextField txtDirectory = new JTextField(); // Directory Path
  private JLabel     lblImageFile = new JLabel();     // Image Filename
  private JTextField txtFilename  = new JTextField(); // Filename

  /** Constructor guiSearchDirectoryFrame( Observable notifier,
   *                                       String     sDirectory,
   *                                       String     sFile,
   *                                       String     sType )
   *  <p> This is the Constructor for this class.
   *  @param notifier   = Observable
   *  @param sDirectory = Starting Directory Path
   *  @param sFile      = Filename
   *  @param sType      = File Type assume .pdf or .html
   */

  public guiSearchDirectoryFrame( Observable notifier,
                                  String     sDirectory,
                                  String     sFile,
                                  String     sType )
  {
    try
    {
      this.notifier   = notifier;
      this.sDirectory = sDirectory;
      this.sFile      = sFile;
      this.sType      = sType;
      sImage          = new String(sType);

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop       = new JPanel();
    JPanel pnlBase      = new JPanel();
    JPanel pnlCenter    = new JPanel();
    JPanel pnlDirectory = new JPanel();
    JPanel pnlFiles     = new JPanel();
    JPanel pnlImage     = new JPanel();
    JPanel pnlFile      = new JPanel();
    JPanel pnlSearch    = new JPanel();
    JPanel pnlPath      = new JPanel();

    JLabel lblPNG       = new JLabel();
    JLabel lblFile      = new JLabel();

    TitledBorder titledBorderSaveTo = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Save Files To:");

    TitledBorder titledBorderImage = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Image Filename:");

    TitledBorder titledBorderFilename = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Filename:");

    titledBorderSaveTo.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderImage.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderFilename.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setTitle("Select a Different Directory Path");
    this.getContentPane().setLayout(new BorderLayout());

    // Base Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnContinue.setFont(new java.awt.Font("Dialog", 1, 11));
    btnContinue.setText("Continue");
    btnContinue.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

//    txtName.addFocusListener(new guiSearchDirectoryFrameFocusAdapter(this));

    // Center Panel

    pnlCenter.setLayout(new GridLayout(2,1));

    // Directory Panel

    pnlDirectory.setLayout(new BorderLayout());
    pnlDirectory.setBorder(titledBorderSaveTo);

    // Directory Path Panel

    pnlPath.setLayout(new BorderLayout());

    txtDirectory.setText( sDirectory );
    txtDirectory.addFocusListener(new guiSearchDirectoryFrameFocusAdapter(this));

    btnSearch.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSearch.setText("Search");
    btnSearch.addActionListener(this);

    // Files Panel

    pnlFiles.setLayout(new GridLayout());

    // Image Filename Panel

    pnlImage.setBorder(titledBorderImage);
    pnlImage.setLayout(new BorderLayout());

    lblImageFile.setHorizontalAlignment(SwingConstants.TRAILING);
    lblImageFile.setText( sFile );

    lblPNG.setText( sImage );

    // Main Filename Panel

    pnlFile.setBorder(titledBorderFilename);
    pnlFile.setLayout(new BorderLayout());

    txtFilename.setText( sFile );
    txtFilename.setHorizontalAlignment(SwingConstants.TRAILING);
    txtFilename.addFocusListener(new guiSearchDirectoryFrameFocusAdapter(this));

    lblFile.setText( sType );

    // Attach Widgets to Frame

    this.getContentPane().add(pnlTop,    BorderLayout.NORTH);

    this.getContentPane().add(pnlBase,   BorderLayout.SOUTH);
    pnlBase.add(btnContinue,             null);
    pnlBase.add(btnCancel,               null);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(pnlDirectory,          null);
    pnlDirectory.add(pnlSearch,          BorderLayout.EAST);
    pnlSearch.add(btnSearch,             null);

    pnlDirectory.add(pnlPath,            BorderLayout.CENTER);
    pnlPath.add(txtDirectory,            BorderLayout.CENTER);

    pnlCenter.add(pnlFiles,              null);
    pnlFiles.add(pnlImage,               null);
    pnlImage.add(lblPNG,                 BorderLayout.EAST);
    pnlImage.add(lblImageFile,           BorderLayout.CENTER);

    pnlFiles.add(pnlFile,                null);
    pnlFile.add(lblFile,                 BorderLayout.EAST);
    pnlFile.add(txtFilename,             BorderLayout.CENTER);

    // Set the Size and Display the Frame

    this.setSize(550, 200);

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width  - this.getSize().width)/2,
                     (d.height - this.getSize().height)/2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getPath()
   * <p> This method will return the Directory Path
   * @return sDirectory = the Directory Path
   */

  public String getPath() { return (sDirectory); }

  /** Method getFile()
   * <p> This method will return the Main Filename to use on the files
   * @return sFile = the Main Filename to use on the files
   */

  public String getFile() { return (sFile); }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method search()
   * <p> This method will search the PC for a directory path
   */

  private void search()
  {
    JFileChooser chooser = new JFileChooser();

    chooser.setCurrentDirectory( new File( sDirectory ) );
    chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY );

    int result = chooser.showDialog(this, "Select");

    if (result == JFileChooser.APPROVE_OPTION)
    {
      sDirectory = new String( chooser.getSelectedFile().getPath() );
      txtDirectory.setText( sDirectory );
    }
  }

  /** Method select()
   * <p> This method will send a message to retrieve the Path & Filename
   */

  private void select()
  {
    sFile      = new String( txtFilename.getText() );
    sDirectory = new String( txtDirectory.getText() );

    if (notifier != null)
      notifier.notifyObservers( new String("Directory Path Selected") );
  }

  /** Method cancel()
   * <p> This method will send a message to retrieve the Path & Filename
   */

  private void cancel()
  {
	close();

//    if (notifier != null)
//      notifier.notifyObservers( new String("Directory Path Cancel") );
  }

  /** Method close()
   * <p> This method will set the global variables to null in order to force
   *     JAVA to free the memory allocated for this panel.
   */

  public void close()
  {
    // Input Variables

    notifier   = null; // Observable Object
    sDirectory = null;
    sFile      = null;
    sImage     = null;
    sType      = null;

    // Global Frame Widgets

    btnContinue = null;  // Send Message Path Selected
    btnCancel   = null;  // Cancel Search
    btnSearch   = null;  // Search for Directory Path

    txtDirectory = null; // Directory Path
    lblImageFile = null; // Image Filename
    txtFilename  = null; // Filename

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnSearch)   { search(); }
    if (event.getSource() == btnContinue) { select(); }
    if (event.getSource() == btnCancel)   { cancel(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    String sMessage = new String("");

    if (e.getSource() == txtFilename)
    {
      sFile = txtFilename.getText();
      lblImageFile.setText( sFile );
    }

    if (e.getSource() == txtDirectory)
    {
      sDirectory = txtDirectory.getText();
    }
  }
}

/** CLASS guiSearchDirectoryFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class guiSearchDirectoryFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  guiSearchDirectoryFrame adaptee;

  guiSearchDirectoryFrameFocusAdapter(guiSearchDirectoryFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}


/*
 *  @version 1.1 12/10/2009
 *  @author  John Victorine
 */
