/*
 * @dstUtility.java Version 1.1 03/16/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package dst;

import dst.dstListStruct;
import dst.dstStruct;

/** Class dstUtility
 *  <p> This Class will provide basic utilities for the dst data structures
 *
 *  @version 1.1 03/16/2012
 *  @author  John R. Victorine
 */

public class dstUtility
{
  public static final int _INITIAL_HYDRO = 0;
  public static final int _FLOW_1        = 1;
  public static final int _SHUT_IN_1     = 2;
  public static final int _END_SHUT_IN_1 = 3;
  public static final int _FLOW_2        = 4;
  public static final int _SHUT_IN_2     = 5;
  public static final int _END_SHUT_IN_2 = 6;
  public static final int _FINAL_HYDRO   = 7;
  public static final int _SUMMARY       = 8;

  public static final int _TIME          = 0;
  public static final int _PRESSURE      = 1;
  public static final int _TEMPERATURE   = 2;
  public static final int _AREA_ROW      = 3;

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static dstListStruct transfer(dstListStruct stOld)
  {
    dstListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copyList( dstListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static dstListStruct copyList( dstListStruct stOld )
  {
    dstListStruct stNew = new dstListStruct();

    if (stOld != null)
    {
      stNew.iCount = stOld.iCount;
      stNew.stItem = new dstStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( dstStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static dstStruct copy( dstStruct stOld )
  {
	int i,j;
    dstStruct stNew = new dstStruct();

    if (stOld != null)
    {
      // -- Units

      stNew.iTime_Unit  = stOld.iTime_Unit; // Time Default Unit (minutes)
      stNew.iPres_Unit  = stOld.iPres_Unit; // Pressure Default Unit (pounds / square inch)
      stNew.iTemp_Unit  = stOld.iTemp_Unit; // Temperature Default Unit (degree Fahrenheit)

      // DST Data Identifiers

      stNew.sKEY        = new String(stOld.sKEY); // Unique KEY
      stNew.sID         = new String(stOld.sID);  // DST Report ID or Number

      // DST General Information

      stNew.depthStart  = stOld.depthStart; // Start Interval Depth (ft)
      stNew.depthEnd    = stOld.depthEnd;   // End Interval Depth   (ft)
      stNew.sFormation  = new String(stOld.sFormation);  // Formation
      stNew.iDeviated   = stOld.iDeviated;  // Hole Deviated
      stNew.dWhipstock  = stOld.dWhipstock; // Whipstock (ft)

      // Pressure Readings

      stNew.serialNo    = new String(stOld.serialNo);    // Serial Number
      stNew.sReading    = new String(stOld.sReading);    // Inside / Outside
      stNew.dPressure   = stOld.dPressure;               // Pressure
      stNew.depth       = stOld.depth;                   // @depth
      stNew.dCapacity   = stOld.dCapacity;               // Pressure Capacity
      stNew.sDateStart  = new String(stOld.sDateStart);  // Starting Date-Time (MM/DD/YYYY-HHMMSS
      stNew.sDateEnd    = new String(stOld.sDateEnd);    // Ending Date-Time (MM/DD/YYYY-HHMMSS

      stNew.sDateCalib  = new String(stOld.sDateCalib);  // Last Calibration Data (MM/DD/YYYY)
      stNew.sTimeOnBtm  = new String(stOld.sTimeOnBtm);  // Time on Bottom
      stNew.sTimeOffBtm = new String(stOld.sTimeOffBtm); // Time off Bottom

      stNew.sTestRemark = new String(stOld.sTestRemark); // Test Remarks

      // DST Company Information

      stNew.sCompany    = new String(stOld.sCompany);    // DST Company
      stNew.sTestType   = new String(stOld.sTestType);   // Test Type
      stNew.sTester     = new String(stOld.sTester);     // Tester
      stNew.sUnitNo     = new String(stOld.sUnitNo);     // Unit Number

      // Hole Information

      stNew.diameter_hole   = stOld.diameter_hole; // Hole diameter
      stNew.sCondition_hole = new String(stOld.sCondition_hole);  // Hole Condition

      // Pressure Summary Table Array

      stNew.dSummary = new double[dst.dstStruct._ROWS][dst.dstStruct._COLUMNS];
      for (i=0; i<dst.dstStruct._ROWS; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.dSummary[i][j] = stOld.dSummary[i][j];
		}
	  }

      // Mud and Cushion Information

      stNew.sMudType        = new String(stOld.sMudType);  // Mud Type
      stNew.dMudWeight      = stOld.dMudWeight; // Mud Weight (lb/gal)
      stNew.dViscosity      = stOld.dViscosity; // Viscosity  (sec/qt)
      stNew.dWaterLoss      = stOld.dWaterLoss; // Water Loss (cubic inches)
      stNew.dOHM_M          = stOld.dOHM_M; // Resistivity (ohm-m)
      stNew.dSalinity       = stOld.dSalinity; // Salinity (ppm)
      stNew.dFilterCake     = stOld.dFilterCake; // Filter Cake (inches)

      stNew.sCushionType    = new String(stOld.sCushionType);  // Cushion Type
      stNew.dCushionLength  = stOld.dCushionLength; // Cushion Length (ft)
      stNew.dCushionVolume  = stOld.dCushionVolume; // Cushion Volume (bbl)

      stNew.sGasCushionType = new String(stOld.sGasCushionType);  // Gas Cushion Type
      stNew.dGasCushionPres = stOld.dGasCushionPres; // Gas Cushion Pressuer (psi)

      stNew.dOil_API        = stOld.dOil_API;       // Oil API (deg API)
      stNew.dWaterSalinity  = stOld.dWaterSalinity; // Water Salinity (ppm)

      // Recovery Table Array

      stNew.iRecovery       = stOld.iRecovery;
      stNew.jRecovery       = stOld.jRecovery;
      stNew.sRecovery       = new String[stOld.iRecovery][stOld.jRecovery];

      for (i=0; i<stOld.iRecovery; i++)
      {
		for (j=0; j<stOld.jRecovery; j++)
		{
		  stNew.sRecovery[i][j] = new String( stOld.sRecovery[i][j] );
		}
	  }

      stNew.dLength         = stOld.dLength;               // Total Length
      stNew.dVolume         = stOld.dVolume;               // Total Volume
      stNew.sComments       = new String(stOld.sComments); // Recovery Comments

      // Type of DST - Fluid or gas

      stNew.iDSTType        = stOld.iDSTType;
      stNew.iDSTTypeFlow    = stOld.iDSTTypeFlow;

      // Gas Rates

      stNew.iGasRates       = stOld.iGasRates;     // Rows
      stNew.jGasRates       = stOld.jGasRates;     // Columns

      stNew.dGasRates = new double[stNew.iGasRates][stNew.jGasRates];

      for (i=0; i<stOld.iGasRates; i++)
      {
		for (j=0; j<stOld.jGasRates; j++)
		{
		  stNew.dGasRates[i][j] = stOld.dGasRates[i][j];
		}
	  }

//      stNew.dTemp           = stOld.dTemp;   // Formation Temperature
//      stNew.density         = stOld.density; // Formation density
//      stNew.dZ              = stOld.dZ;      // Gas Deviation Factor ( Compressibility Factor )

      // Time Pressure Temperature Data

      stNew.iData = stOld.iData;
      stNew.data  = new double[stOld.iData][dst.dstStruct._COLUMNS];  // Time Pressure Temperature Data

      for (i=0; i<stOld.iData; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.data[i][j] = stOld.data[i][j];
		}
	  }

      // Shut In Pressure Data

      stNew.iShut_In_1 = stOld.iShut_In_1;
      stNew.dShut_In_1 = new double[stOld.iShut_In_1][dst.dstStruct._COLUMNS];  // Shut In 1 Data

      for (i=0; i<stOld.iShut_In_1; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.dShut_In_1[i][j] = stOld.dShut_In_1[i][j];
		}
	  }

      stNew.iShut_In_2 = stOld.iShut_In_2;
      stNew.dShut_In_2 = new double[stOld.iShut_In_2][dst.dstStruct._COLUMNS];  // Shut In 2 Data

      for (i=0; i<stOld.iShut_In_2; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.dShut_In_2[i][j] = stOld.dShut_In_2[i][j];
		}
	  }

      // Open Flow Pressure Data

      stNew.iFlow_1 = stOld.iFlow_1;
      stNew.dFlow_1 = new double[stOld.iFlow_1][dst.dstStruct._COLUMNS];  // Flow 1 Data

      for (i=0; i<stOld.iFlow_1; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.dFlow_1[i][j] = stOld.dFlow_1[i][j];
		}
	  }

      stNew.iFlow_2 = stOld.iFlow_2;
      stNew.dFlow_2 = new double[stOld.iFlow_2][dst.dstStruct._COLUMNS];  // Flow 2 Data

      for (i=0; i<stOld.iFlow_2; i++)
      {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  stNew.dFlow_2[i][j] = stOld.dFlow_2[i][j];
		}
	  }

      // Computed Values

      // -- Horner Plot Result Variables

      // .. Horner Plot Parameters

      // .... Titles

      stNew.sTitle1     = new String(stOld.sTitle1); // The First Line of the Title
      stNew.sTitle2     = new String(stOld.sTitle2); // The Second Line of the Title
      stNew.sTitle3     = new String(stOld.sTitle3); // The Third Line of the Title

      // .... X-Axis Values.

      stNew.iXCycles    = stOld.iXCycles;            // Number of Cycles
      stNew.dXMaximum   = stOld.dXMaximum;           // Maximum X-Axis
      stNew.dXMinimum   = stOld.dXMinimum;           // Minimum X-Axis
      stNew.sXAxis      = new String(stOld.sXAxis);  // X-Axis Label.

      // .... Y-Axis Values.

      stNew.dYMaximum   = stOld.dYMaximum;           // Maximum Y-Axis
      stNew.dYMinimum   = stOld.dYMinimum;           // Minimum Y-Axis
      stNew.dYIncrement = stOld.dYIncrement;         // Increment along the Y-Axis
      stNew.sYAxis      = new String(stOld.sYAxis);  // Y-Axis Label.

      // .. Shut In 1 Time Pressure Variables

      stNew.lpt1 = stOld.lpt1; // Number of points excluded below (shut in 1)
      stNew.upt1 = stOld.upt1; // Number of points excluded above (shut in 1)

      stNew.P1   = stOld.P1;   // Intercept Pressure for Shut In 1
      stNew.m1   = stOld.m1;   // Horner Plot Slope of Shut In 1

      // .. Shut In 2 Time Pressure Variables

      stNew.lpt2 = stOld.lpt2; // Number of points excluded below (shut in 2)
      stNew.upt2 = stOld.upt2; // Number of points excluded above (shut in 2)

      stNew.P2   = stOld.P2;   // Intercept Pressure for Shut In 2
      stNew.m2   = stOld.m2;   // Horner Plot Slope of Shut In 2

      // .. Open Flow 1 Time Pressure Variables

      stNew.lpt1f = stOld.lpt1f; // Number of points excluded below (shut in 1)
      stNew.upt1f = stOld.upt1f; // Number of points excluded above (shut in 1)

      stNew.P1f   = stOld.P1f;   // Intercept Pressure for Shut In 1
      stNew.m1f   = stOld.m1f;   // Horner Plot Slope of Shut In 1

      // .. Open Flow 2 Time Pressure Variables

      stNew.lpt2f = stOld.lpt2f; // Number of points excluded below (shut in 2)
      stNew.upt2f = stOld.upt2f; // Number of points excluded above (shut in 2)

      stNew.P2f   = stOld.P2f;   // Intercept Pressure for Shut In 2
      stNew.m2f   = stOld.m2f;   // Horner Plot Slope of Shut In 2

      // -- Input for Computed Variables

      stNew.mu   = stOld.mu;   // Fluid or Gas Viscosity   [cp]
      stNew.beta = stOld.beta; // Formation Volume Factor  [RB]/[STB]
      stNew.h    = stOld.h;    // Vertical thickness of continuous porosity [ft]

      stNew.Tf   = stOld.Tf;   // Flow Time
      stNew.phi  = stOld.phi;  // Porosity
      stNew.c    = stOld.c;    // Compressibilty [vol]/[vol]/[psi]

      stNew.rw   = stOld.rw;   // Well Bore Radius  [ft]
      stNew.re   = stOld.re;   // Effective Radius  [ft]

      // .. Gas Variables

      stNew.Z    = stOld.Z;    // Gas Deviation Factor (compressibility factor)
      stNew.degF = stOld.degF; // Formation Temperature [deg F]
      stNew.G    = stOld.G;    // Specific Gravity of Gas

      // -- Computed Variables

      stNew.q    = stOld.q;    // Flow Rate [Barrels]/[Day] or [Mcf]/[Day]
      stNew.K    = stOld.K;    // Average Effective Permeability

      stNew.KHU  = stOld.KHU;  // Transmissibility
      stNew.KH   = stOld.KH;   // Permeability Thickness

      stNew.DR   = stOld.DR;   // Damage Ratio
      stNew.EDR  = stOld.EDR;  // Estimated Damage Ratio
      stNew.S    = stOld.S;    // Skin Factor
      stNew.PI   = stOld.PI;   // Productivity Index (Prediction for stable flow rates)
      stNew.RI   = stOld.RI;   // Radius of Investigation

      stNew.qs   = stOld.qs;   // Settled Production Rate [Barrels]/[Day]
      stNew.qAOF = stOld.qAOF; // Absolute Open Flow [Mcf]/[Day]

      stNew.Pwfc = stOld.Pwfc; // Flow Pressure corrected for turbulence [psi]
      stNew.dPt  = stOld.dPt;  // Pressure Drop due to Turbulence
      stNew.dPs  = stOld.dPs;  // Pressure Drop due to Skin Factor
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------- IDENTIFY DATA TYPES METHOD ---------------------- *
   * ======================================================================== */

  /** Method parseData()
   *  <p> This method will parse the raw data into Shut In and Open data variables.
   * @param  st = The Data Structure with raw data only
   * @return st = The Data Structure with Shut In and Open Data types identified.
   */

  public static dstStruct parseData( dstStruct stData )
  {
	int i,j,k;
	int    iCount      = 0;
	double dTemp       = 0.0;
	double dTo         = 0.0;
	double dTf         = 0.0;
    double SUMMARY[][] =
    {
    //  Time Pres Temp
	  { 0.0, 0.0, 0.0 }, // Initial Hydrostatic
	  { 0.0, 0.0, 0.0 }, // Open to Flow 1
	  { 0.0, 0.0, 0.0 }, // Shut-in 1
	  { 0.0, 0.0, 0.0 }, // End Shut-in 1
	  { 0.0, 0.0, 0.0 }, // Open to Flow 2
	  { 0.0, 0.0, 0.0 }, // Shut-in 2
	  { 0.0, 0.0, 0.0 }, // End Shut-in 2
	  { 0.0, 0.0, 0.0 }, // Final Hydrostatic
    };

	if (stData != null)
	{
	  for (i=0; i<_SUMMARY; i++)
	  {
	    for (j=0; j<_AREA_ROW; j++)
	    {
	  	  SUMMARY[i][j] = stData.dSummary[i][j];
	    }
	  }

      /* ................................................................... *
       * ............ SHUT IN 1 Time Pressure & Temperature Data ........... *
       * ................................................................... */

      iCount = 0;
	  dTo    = SUMMARY[_SHUT_IN_1][_TIME];
	  dTf    = SUMMARY[_END_SHUT_IN_1][_TIME];

      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
		  iCount++;
	    }
	  }

      stData.iShut_In_1 = iCount;
      stData.dShut_In_1 = new double[stData.iShut_In_1][_AREA_ROW];

      iCount = 0;
      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
  	      if (iCount < stData.iShut_In_1)
		  {
		    for (j=0; j<_AREA_ROW; j++)
		    {
		      stData.dShut_In_1[iCount][j] = stData.data[i][j];
		    }

  	        iCount++;
	  	  }
	    }
	  }

      stData.iShut_In_1 = iCount;

      dTemp = 0;
      if (stData.iShut_In_1 > 0)
        dTemp = stData.dShut_In_1[0][_TIME];

      for (k=0; k<stData.iShut_In_1; k++)
      {
	    stData.dShut_In_1[k][_TIME] = stData.dShut_In_1[k][_TIME] - dTemp;
	  }

      /* ................................................................... *
       * ............ SHUT IN 2 Time Pressure & Temperature Data ........... *
       * ................................................................... */

      iCount = 0;
	  dTo    = SUMMARY[_SHUT_IN_2][_TIME];
	  dTf    = SUMMARY[_END_SHUT_IN_2][_TIME];

      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
		  iCount++;
	    }
	  }

      stData.iShut_In_2 = iCount;
      stData.dShut_In_2 = new double[stData.iShut_In_2][_AREA_ROW];

      iCount = 0;
      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
	      if (iCount < stData.iShut_In_2)
		  {
		    for (j=0; j<_AREA_ROW; j++)
		    {
		      stData.dShut_In_2[iCount][j] = stData.data[i][j];
		    }

	        iCount++;
  	      }
	    }
	  }

      stData.iShut_In_2 = iCount;

      dTemp = 0;
      if (stData.iShut_In_2 > 0)
        dTemp = stData.dShut_In_2[0][_TIME];

      for (k=0; k<stData.iShut_In_2; k++)
      {
	    stData.dShut_In_2[k][_TIME] = stData.dShut_In_2[k][_TIME] - dTemp;
	  }

      /* ................................................................... *
       * ........... Open Flow 2 Time Pressure & Temperature Data .......... *
       * ................................................................... */

      iCount = 0;
	  dTo    = SUMMARY[_FLOW_2][_TIME];
	  dTf    = SUMMARY[_SHUT_IN_2][_TIME];

      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
	   	  iCount++;
	    }
	  }

      stData.iFlow_2 = iCount;
      stData.dFlow_2 = new double[stData.iFlow_2][_AREA_ROW];

      iCount = 0;
      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
	      if (iCount < stData.iFlow_2)
	  	  {
		    for (j=0; j<_AREA_ROW; j++)
		    {
		      stData.dFlow_2[iCount][j] = stData.data[i][j];
		    }

	        iCount++;
  	      }
	    }
	  }

      stData.iFlow_2 = iCount;

      dTemp = 0;
      if (stData.iFlow_2 > 0)
        dTemp = stData.dFlow_2[0][_TIME];

      for (k=0; k<stData.iFlow_2; k++)
      {
	    stData.dFlow_2[k][_TIME] = stData.dFlow_2[k][_TIME] - dTemp;
	  }

      /* ................................................................... *
       * ........... Open Flow 1 Time Pressure & Temperature Data .......... *
       * ................................................................... */

      iCount = 0;
	  dTo    = SUMMARY[_FLOW_1][_TIME];
	  dTf    = SUMMARY[_SHUT_IN_1][_TIME];

      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
	  	  iCount++;
	    }
	  }

      stData.iFlow_1 = iCount;
      stData.dFlow_1 = new double[stData.iFlow_1][_AREA_ROW];

      iCount = 0;
      for (i=0; i<stData.iData; i++)
      {
	    if ((stData.data[i][_TIME] >= dTo) && (stData.data[i][_TIME] < dTf))
	    {
	      if (iCount < stData.iFlow_1)
		  {
		    for (j=0; j<_AREA_ROW; j++)
		    {
		      stData.dFlow_1[iCount][j] = stData.data[i][j];
	  	    }

	        iCount++;
  	      }
	    }
	  }

      stData.iFlow_1 = iCount;

      dTemp = 0;
      if (stData.iFlow_1 > 0)
        dTemp = stData.dFlow_1[0][_TIME];

      for (k=0; k<stData.iFlow_1; k++)
      {
	    stData.dFlow_1[k][_TIME] = stData.dFlow_1[k][_TIME] - dTemp;
	  }
	}

	return (stData);
  }

  /* ======================================================================== *
   * -------------------------- EDIT LIST METHODS --------------------------- *
   * ======================================================================== */

  /** Method add()
   * <p> This method will add the digitized data to an existing list
   * @param  stBlock = The digitized data Block List Data Structure
   * @param  st      = The Old digitized data List Data Structure
   * @return st      = The new digitized data List.
   */

  public static dstListStruct add( dstStruct stBlock, dstListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    dstListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new dstListStruct();
    stTemp.stItem = new dstStruct[iRecords];

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new dstStruct();
    stTemp.stItem[iCount] = copy(stBlock);
    iCount++;

    stTemp.iCount = iCount;

    st        = new dstListStruct();
    st.stItem = new dstStruct[iCount];
    st.iCount = iCount;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Rock dst in an existing list
   * @param  sKEY    = The Primary Key of the data to be removed
   * @param  stBlock = The dst structure that will be replaced
   * @param  st      = The Old digitized data List Data Structure
   * @return st      = The new digitized data List.
   */

  public static dstListStruct modify( String    sKEY,
                                      dstStruct stBlock,
                                      dstListStruct st )
  {
    int i=0;
    dstListStruct stTemp = null;

    if (st != null)
    {
      stTemp        = new dstListStruct();
      stTemp.stItem = new dstStruct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

      st        = new dstListStruct();
      st.stItem = new dstStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Rock dst from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old digitized data List Data Structure
   * @return st     = The new digitized data List.
   */

  public static dstListStruct remove( String sKEY, dstListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    dstListStruct stTemp = null;

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        iRecords = st.iCount-1;

        stTemp        = new dstListStruct();
        stTemp.stItem = new dstStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st        = new dstListStruct();
      st.stItem = new dstStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the contents of the structure
   */

  public static void print(dstStruct st)
  {
	int i,j;

	if (st != null)
	{
	  System.out.println(st.depthStart + " to " + st.depthEnd + " " + st.sFormation);

      if (st.serialNo.length() > 0)
        System.out.println("Serial No: " + st.serialNo);
      if ((st.dPressure != 0.0) && (st.depth != 0.0))
        System.out.println(st.dPressure + " psi @ " + st.depth + " ft");

      if (st.sDateStart.length() > 0)
        System.out.println("Date Start: " + st.sDateStart);
      if (st.sDateEnd.length() > 0)
        System.out.println("Date End: " + st.sDateEnd);

      if (st.sTestRemark.length() > 0)
        System.out.println("Test Comments: " + st.sTestRemark);

      if ((st.sCompany.length() > 0)  ||
          (st.sTestType.length() > 0) ||
          (st.sTester.length() > 0))
      {
        System.out.println("");
		System.out.println(" ** Logging Company Information ** ");
        System.out.println("");

        if (st.sCompany.length() > 0)
          System.out.println("Company: " + st.sCompany);
        if (st.sTestType.length() > 0)
          System.out.println("Test Type: " + st.sTestType);
        if (st.sTester.length() > 0)
          System.out.println("Engineer: " + st.sTester);
        if (st.sUnitNo.length() > 0)
          System.out.println("Unit Number: " + st.sUnitNo);
	  }

      if ((st.diameter_hole > 0.0) || (st.sCondition_hole.length() > 0))
      {
        System.out.println("");
		System.out.println(" ** Hole Information ** ");
        System.out.println("");

        if (st.diameter_hole > 0.0)
          System.out.println("Diameter: " + st.diameter_hole + " in");
        if (st.sCondition_hole.length() > 0)
          System.out.println("Condition: " + st.sCondition_hole);
	  }

      if (st.sTestRemark.length() > 0)
      {
        System.out.println("");
        System.out.println("Test Comments: " + st.sTestRemark);
	  }

      System.out.println("");
      System.out.println(" ** Pressure Summary Table Array ** ");
      System.out.println("");

      System.out.println(" Time  Pressure  Temperature Annotation ");
      System.out.println("(min)   (psig)     (deg F)  ");

	  for (i=0; i<dst.dstStruct._ROWS; i++)
	  {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  System.out.print(st.dSummary[i][j]+" ");
		}

  	    System.out.print(dst.dstStruct.ANNOTATION[i]+" ");
	    System.out.println("");
	  }
	  System.out.println("");

      if ((st.dMudWeight > 0.0) || (st.sMudType.length() > 0) ||
          (st.dViscosity > 0.0) || (st.dWaterLoss > 0.0) ||
          (st.dOHM_M > 0.0) || (st.dSalinity > 0.0) ||
          (st.dFilterCake > 0.0))
      {
        System.out.println("");
		System.out.println(" ** Mud Information ** ");
        System.out.println("");

        if (st.sMudType.length() > 0)
          System.out.println("Mud Type:    " + st.sMudType);
        if (st.dMudWeight > 0.0)
          System.out.println("Mud Weight:  " + st.dMudWeight + " lb/gal");
        if (st.dViscosity > 0.0)
          System.out.println("Viscosity:   " + st.dViscosity + " sec/qt");
        if (st.dWaterLoss > 0.0)
          System.out.println("Water Loss:  " + st.dWaterLoss + " cubic in");
        if (st.dOHM_M > 0.0)
          System.out.println("Resistivity: " + st.dOHM_M + " ohm-m");
        if (st.dSalinity > 0.0)
          System.out.println("Salinity:    " + st.dSalinity + " ppm");
        if (st.dFilterCake > 0.0)
          System.out.println("Filter Cake: " + st.dFilterCake + " in");
	  }

      if ((st.dCushionLength > 0.0)  || (st.sCushionType.length() > 0) ||
          (st.dCushionVolume > 0.0)  || (st.sGasCushionType.length() > 0) ||
          (st.dGasCushionPres > 0.0))
      {
        System.out.println("");
		System.out.println(" ** Cushion Information ** ");
        System.out.println("");

        if (st.sCushionType.length() > 0)
          System.out.println("Cushion Type: " + st.sCushionType);
        if (st.dCushionLength > 0.0)
          System.out.println("Length:       " + st.dCushionLength + " ft");
        if (st.dCushionVolume > 0.0)
          System.out.println("Volume:       " + st.dCushionVolume + " bbl");

        if (st.sGasCushionType.length() > 0)
          System.out.println("Gas Cushion Type: " + st.sGasCushionType);
        if (st.dGasCushionPres > 0.0)
          System.out.println("Pressure:         " + st.dGasCushionPres + " psig");
      }

      if ((st.dOil_API > 0.0)  || (st.dWaterSalinity > 0.0))
      {
        if (st.dOil_API > 0.0)
          System.out.println("Oil Gravity:    " + st.dOil_API + " deg API");
        if (st.dWaterSalinity > 0.0)
          System.out.println("Water Salinity: " + st.dWaterSalinity + " ppm");
	  }

      System.out.println("");
      System.out.println(" ** Recovery Table ** ");
      System.out.println("");

      System.out.println(" Length  Description                      Volume ");
      System.out.println(" (ft)                                      (bbl) ");

	  for (i=0; i<st.iRecovery; i++)
	  {
		for (j=0; j<st.jRecovery; j++)
		{
		  System.out.print(st.sRecovery[i][j]+" ");
		}

	    System.out.println("");
	  }
	  System.out.println("");

      System.out.println("Total Length:" + st.dLength +
                         " (ft)  Total Volume:" + st.dVolume + " (bbl)");

      System.out.println("Recovery Comments: "+ st.sComments);

      System.out.println("");
      System.out.println(" ** Gas Rates Table ** ");
      System.out.println("");

      System.out.println(" Period  Time   Choke   Pressure   Gas Rate ");
      System.out.println("        (min)    (in)    (psig)    (Mcf/day) ");

	  for (i=0; i<st.iGasRates; i++)
	  {
		for (j=0; j<st.jGasRates; j++)
		{
		  System.out.print(st.dGasRates[i][j]+" ");
		}

	    System.out.println("");
	  }
	  System.out.println("");

	  System.out.println("");
	  System.out.println(" ** Quantitative Analysis **");
	  System.out.println("");

      if (st.iDSTType == dst.dstStruct._FLUID_DST)
      {
		System.out.println( " m = 162.6 q B u / k h " );
		System.out.println( " where: " );
		System.out.println( "   slope (m)        = " + Math.abs(st.m2) + " psi/cycle");
		System.out.println( "   Flow Rate (q)    = " + st.q            + " STB/day");
		System.out.println( "   FVF (B)          = " + st.beta         + " RB/STB");
		System.out.println( "   Viscosity (u)    = " + st.mu           + " cp");
		System.out.println( "   Thickness (h)    = " + st.h            + " ft");
		System.out.println( "   Permeability (K) = " + st.K            + " md");
	  }

      if (st.iDSTType == dst.dstStruct._GAS_DST)
      {
		System.out.println( " m = 1637 q T Z u / k h " );
		System.out.println( " where: " );
		System.out.println( "   slope (m)         = " + Math.abs(st.m2) + " psi/cycle");
		System.out.println( "   Flow Rate (q)     = " + st.q            + " STB/day");
		System.out.println( "   Temperature (T)   = " + st.degF         + " deg F");
		System.out.println( "   Gas Deviation (Z) = " + st.Z );
		System.out.println( "   Viscosity (u)     = " + st.mu           + " cp");
		System.out.println( "   Thickness (h)     = " + st.h            + " ft");
		System.out.println( "   Permeability (K)  = " + st.K            + " md");
	  }

	  System.out.println("");

	  System.out.println("Transmissibility (Kh/u)     = " + st.KHU );
	  System.out.println("Permeability Thickness (Kh) = " + st.KH );

	  System.out.println("Damage Ratio                = " + st.DR );
	  System.out.println("Estimated Damage Ratio      = " + st.EDR );
	  System.out.println("Skin Factor                 = " + st.S );
	  System.out.println("Skin Factor Pressure Drop   = " + st.dPs + " (psi)");

      if (st.iDSTType == dst.dstStruct._FLUID_DST)
      {
  	    System.out.println("Productive Index            = " + st.PI );
  	    System.out.println("Radius of Investigation     = " + st.RI + " (ft)" );
  	    System.out.println("Settled Production Rate (qs)= " + st.qs + " (STB/day)" );
	  }

      if (st.iDSTType == dst.dstStruct._GAS_DST)
      {
	    System.out.println("Flow Pressure Turbulence (Pwfc) = " + st.Pwfc + " (psi)" );
	    System.out.println("Turbulence Pressure Drop        = " + st.dPt  + " (psi)" );
	    System.out.println("Absolute Open Flow (qAOF)       = " + st.qAOF + " (Mcf/day)" );
 	    System.out.println( "   Specific Gravity (G) = " + st.G);
	  }

 	  System.out.println( "   Flow Time (Tf)       = " + st.Tf           + " min");
   	  System.out.println( "   Porosity             = " + st.phi);
 	  System.out.println( "   Compressibilty (c)   = " + st.c            + " vol/vol/psi");

   	  System.out.println( "   Well Bore Radius(rw) = " + st.rw           + " ft");
   	  System.out.println( "   Effective Radius(rw) = " + st.re           + " ft");
	}
  }

  /** Method printShutIn()
   * <p> This method will print the contents of the structure
   */

  public static void printShutIn(int in, dstStruct st)
  {
	int i,j;
	int    iShutIn     = 0;
	double dShutIn[][] = null;

	if (st != null)
	{
	  switch (in)
	  {
		case 1:
		  iShutIn = st.iShut_In_1;
		  dShutIn = st.dShut_In_1;
		  break;
		case 2:
		  iShutIn = st.iShut_In_2;
		  dShutIn = st.dShut_In_2;
		  break;
	  }

	  for (i=0; i<iShutIn; i++)
	  {
		for (j=0; j<dst.dstStruct._COLUMNS; j++)
		{
		  System.out.print(dShutIn[i][j]+" ");
		}
		System.out.println("");
	  }
	}
  }
}