/*
 * @brineTernaryPlot.java Version 1.1 06/16/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;

/** CLASS brineTernaryPlot
 *  <p> This Class will create the Ternary Plot.
 *
 *  @version 1.1 06/16/2011
 *  @author  John R. Victorine
 */

public class brineTernaryPlot extends Canvas
{
  public static final int NORMAL     = 0;
  public static final int MIRROR     = 1;

  public static final int HORIZONTAL = 0;
  public static final int RIGHT      = 1;
  public static final int LEFT       = 2;

  private int iState = NORMAL;  // The Triangle Plot orientation

  private double dHLower     = 0.0;   // Lower Horizontal-Axis Value
  private double dHUpper     = 100.0; // Upper Horizontal-Axis Value
  private double dHIncrement = 20.0;  // Increment Horizontal-Axis for Linear

  private double dRLower     = 0.0;   // Lower Right-Axis Value
  private double dRUpper     = 100.0; // Upper Right-Axis Value
  private double dRIncrement = 20.0;  // Increment Right-Axis for Linear

  private double dLLower     = 0.0;   // Lower Left-Axis Value
  private double dLUpper     = 100.0; // Upper Left-Axis Value
  private double dLIncrement = 20.0;  // Increment Left-Axis for Linear

  private String sTitle      = "";
  private String sAxisH      = "";
  private String sAxisR      = "";
  private String sAxisL      = "";

  private int iXBegin        = 10;    // Starting X Position
  private int iYBegin        = 10;    // Starting Y Position

  private int iWidth         = 254;   // Width of Drawing Area
  private int iHeight        = 220;   // Height of Drawing Area

  /** CONSTRUCTOR brineTernaryPlot()
   *  <p> This is the constructor for this class.
   *  @param iState = Plot a Normal Ternary plot or a Mirror
   *                  Ternary Plot
   */

  public brineTernaryPlot(int iState)
  {
	this.iState = iState;
	this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA
   *     to reallocate memory
   */

  public void close() {}

  /* =============================================================== *
   * ------------------------ SET METHODS -------------------------- *
   * =============================================================== */

  /** Function setTitle()
   *  <p> This method will set the Title of Plot
   *  @param sTitle = The Title of the Plot
   */

  public void setTitle(String sTitle)
  {
    this.sTitle = new String(sTitle);
  }

  /** Function setVerticesTitle()
   *  <p> This method will set the Vertices Title.
   *  @param iVert = The Verticies to be set
   *  @param sText = The Text Label for the Vertices
   */

  public void setVerticesTitle( int iVert, String sText )
  {
    switch (iVert)
    {
      case HORIZONTAL:
        sAxisH = new String(sText);
        break;

      case RIGHT:
        sAxisR = new String(sText);
        break;

      case LEFT:
        sAxisL = new String(sText);
        break;
    }
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawTitles()
   *  <p> This method will draw the plot titles
   *  @param  depth    = The depth of the frame layer in the movieclip
   *  @return depth    = The depth of the frame layer in the movieclip
   */

  public void drawTitles(Graphics g, int X, int Y)
  {
	int         length = 0;
    Font        fsb    = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm   = g.getFontMetrics(fsb);

    g.setFont( fsb );

    length = sTitle.length();

    g.setColor(Color.black);
    g.drawString(sTitle, X+iXBegin + iWidth/2 - length*4,
                         Y+iYBegin+iHeight+30 );
  }

  /** Method drawHorizontalVerticesTitle()
   *  <p> This method will draw the Horizontal Vertices Title
   *  @param  depth    = The depth of the frame layer in the movieclip
   *  @return depth    = The depth of the frame layer in the movieclip
   */

  public void drawHorizontalVerticesTitle(Graphics g, int X, int Y)
  {
    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);
    g.drawString(sAxisH, X+iXBegin+iWidth/2, Y+iYBegin); //-20);
  }

  /** Method drawRightVerticesTitle()
   *  <p> This method will draw the Right Vertices Title
   *  @param  depth    = The depth of the frame layer in the movieclip
   *  @return depth    = The depth of the frame layer in the movieclip
   */

  public void drawRightVerticesTitle(Graphics g, int X, int Y)
  {
    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);
    g.drawString(sAxisR, X+iXBegin+iWidth-20, Y+iYBegin+iHeight+10);
  }

  /** Method drawLeftVerticesTitle()
   *  <p> This method will draw the Left Vertices Title
   *  @param  depth    = The depth of the frame layer in the movieclip
   *  @return depth    = The depth of the frame layer in the movieclip
   */

  public void drawLeftVerticesTitle(Graphics g, int X, int Y)
  {
    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);
    g.drawString(sAxisL, X+iXBegin-20, Y+iYBegin+iHeight+10);
  }

  /** Method drawGrid()
   * <p> This method will draw the grid
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawGrid(Graphics g, int X, int Y)
  {
    drawHorizontalGrid(g, X, Y);
    drawRightGrid(g, X, Y);
    drawLeftGrid(g, X, Y);
  }

  /** Method drawHorizontalGrid()
   * <p> This method will draw a Horizontal Linear Grid
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawHorizontalGrid(Graphics g, int X, int Y)
  {
	int    iTotal = 0;
	int    iXl    = 0;
	int    iXu    = 0;
    int    iXm    = X + iXBegin + (iWidth / 2);
    int    iYn    = 0;
    int    istr   = 0;
    double dWn    = 0.0;
    double ratio  = 0.0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    // Build the Horizontal Linear Grid

    iTotal = (int) (1.0+Math.abs(Math.ceil((dHUpper-dHLower)/dHIncrement)));

    for (int i=0; i<iTotal; i++)
    {
      ratio = (double) (iHeight * i) * dHIncrement /
               Math.abs(dHUpper-dHLower);
      iYn = iHeight - (int) ratio;
      dWn = 2.0 * iYn * Math.tan(Math.PI / 6.0);
      iXl = iXm - (int) dWn / 2;
      iXu = iXm + (int) dWn / 2;

      g.setColor(Color.lightGray);
      g.drawLine(iXl, Y + iYBegin+iYn, iXu, Y + iYBegin+iYn);

      if (iState == NORMAL)
        istr = (int) (dHUpper - i * dHIncrement);
      else
        istr = (int) (dHLower + i * dHIncrement);

      g.setColor(Color.black);
      if ((i!=0) && (i<iTotal-1))
        g.drawString( ""+istr, iXl-15, Y+iYBegin+iYn );
//        g.drawString( ""+istr, iXl-25, Y+iYBegin+iYn-10 );
    }
  }

  /** Method drawRightGrid()
   * <p> This method will draw a Right Linear Grid
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawRightGrid(Graphics g, int X, int Y)
  {
	int    iTotal = 0;
	int    iX1    = 0;
	int    iX2    = 0;
	int    iY1    = 0;
	int    iY2    = 0;
    int    istr   = 0;
	double dX     = 0.0;
	double dY     = 0.0;
    double ratio  = 0.0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    // Build the Right Linear Grid

    iTotal = (int) (1.0 + Math.abs(Math.ceil((dRUpper-dRLower)/dRIncrement)));

    for (int i=0; i<iTotal; i++)
    {
      dX = (double) (iWidth * i) * dRIncrement / Math.abs(dRUpper-dRLower);
      dY = (((double) iWidth-dX)/2.0)*Math.tan(Math.PI/3.0);

      iX1 = X + iXBegin + (int) dX;
      iY1 = Y + iYBegin + iHeight;
      iX2 = X + iXBegin + (iWidth+(int) dX)/2;
      iY2 = Y + iYBegin + iHeight-(int) dY;

      g.setColor(Color.lightGray);
      g.drawLine(iX1, iY1, iX2, iY2);

      if (iState == NORMAL)
        istr = (int) (dRUpper - i * dRIncrement);
      else
        istr = (int) (dRLower + i * dRIncrement);

      g.setColor(Color.black);
      if ((i!=0) && (i<iTotal-1))
        g.drawString( ""+istr, iX2, iY2 );
//        g.drawString( ""+istr, iX2+10, iY2-10 );
    }
  }

  /** Method drawLeftGrid()
   * <p> This method will draw a Left Linear Grid
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawLeftGrid(Graphics g, int X, int Y)
  {
	int    iTotal = 0;
	int    iX1    = 0;
	int    iX2    = 0;
	int    iY1    = 0;
	int    iY2    = 0;
    int    istr   = 0;
	double dX     = 0.0;
	double dY     = 0.0;
    double ratio  = 0.0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    // Build the Right Linear Grid

    iTotal = (int) (1.0 + Math.abs(Math.ceil((dLUpper-dLLower)/dLIncrement)));

    for (int i=0; i<iTotal; i++)
    {
      dX = (double) (iWidth * i) * dLIncrement /
           Math.abs(dLUpper-dLLower);
      dY = (dX/2.0)*Math.tan(Math.PI/3.0);

      iX1 = X + iXBegin + (int) dX;
      iY1 = Y + iYBegin + iHeight;
      iX2 = X + iXBegin + (int) dX/2;
      iY2 = Y + iYBegin + iHeight-(int) dY;

      g.setColor(Color.lightGray);
      g.drawLine(iX1, iY1, iX2, iY2);

      if (iState == NORMAL)
        istr = (int) (dLLower + i * dLIncrement);
      else
        istr = (int) (dLUpper - i * dLIncrement);

      g.setColor(Color.black);
      if ((i!=0) && (i<iTotal-1))
        g.drawString( ""+istr, iX1, Y+iYBegin+iHeight+10 );
//        g.drawString( ""+istr, iX1, Y+iYBegin+iHeight );
    }
  }

  /** Method drawPoint()
   * <p> This method will draw the point
   *  @param g  = Graphics pointer.
   *  @param X  = Starting X Point on plot
   *  @param Y  = Starting Y Point on plot
   *  @param dX = X data Point on plot
   *  @param dY = Y data Point on plot
   */

  public void drawPoint(Graphics g, int X, int Y,
                        double dX, double dY)
  {
	int    iX     = 0;
	int    iY     = 0;
	double dRx    = 0.0;
	double dRy    = 0.0;
	double dTemp  = 0.0;
	double ratio  = 0.0;

    // Set up Graphics Parameters

    dRy = iHeight*(dY-dHLower)/(dHUpper-dHLower);
    iY    = (int) dRy;

    dTemp = dX;
    if (iState != NORMAL)
      dTemp = dLUpper - dX;

    dRx = iWidth*(dTemp-dLLower)/(dLUpper-dLLower);

    if (iState == NORMAL)
    {
      if (dTemp < (dLUpper-dLLower)/2)
        iX = (int) (dRx + dRy * Math.tan(Math.PI/6));
      else
        iX = (int) (dRx - dRy * Math.tan(Math.PI/6));
    }
    else
      iX = (int) (dRx - dRy* Math.tan(Math.PI/6));

    g.setColor(Color.red);
    g.fillRect(X+iXBegin+iX-3, Y+iYBegin+iHeight-iY-3, 6, 6);
  }

  /** Method draw()
   * <p> This method will draw Diamond Plot
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void draw(Graphics g, int X, int Y)
  {
	drawTitles( g, X, Y );
	drawHorizontalVerticesTitle( g, X, Y );
	drawRightVerticesTitle( g, X, Y );
	drawLeftVerticesTitle( g, X, Y );
    drawGrid( g, X, Y );
  }

  /** Method paint( Graphics g )
   * <p> This method will paint the stiff diagram Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 06/16/2011
 *  @author  John Victorine
 */
