/*
 * @brineStiffPlot.java Version 1.1 06/10/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;

import brine.plot.brinePlotStruct;     // Common Brine Data Sample

/** CLASS brineStiffPlot
 *  <p> This Class will create the Stiff Diagram Plot.
 *
 *  @version 1.1 06/10/2011
 *  @author  John R. Victorine
 */

public class brineStiffPlot extends Canvas
{
  private brinePlotStruct st      = null; // Common Brine Data

  public static final Color cCA   = new Color( 160, 160, 160 );
  public static final Color cMG   = new Color(  34, 139,  34 );
  public static final Color cNA   = new Color(   0, 191, 255 );
  public static final Color cK    = new Color(   0,   0, 180 );

  public static final Color CLR_CATIONS[] = { cCA, cMG, cNA, cK };

  public static final Color cCL   = new Color( 173, 255,  47 );
  public static final Color cBR   = new Color( 180,   0,   0 );
  public static final Color cI    = new Color( 180,   0, 180 );
  public static final Color cSO4  = new Color( 102, 102,   0 );
  public static final Color cHCO3 = new Color( 250, 128, 114 );
  public static final Color cCO3  = new Color( 255, 182, 193 );

  public static final Color CLR_ANIONS[] = { cCL, cBR, cI, cSO4, cHCO3, cCO3 };

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

  private int iWidth   = 260; // Width of Plot
  private int iHeight  = 135; // Height of Plot with the Titles

  private int iXBegin  =  40; // Left Edge of Plot Area
  private int iYBegin  =  60; // Top Edge of Plot Area

  private int iXWidth  = 200; // Left Edge of Plot Area
  private int iYHeight =  75; // Top Edge of Plot Area

  private int iX[]     = null;
  private int iY[]     = null;

  /** CONSTRUCTOR brineStiffPlot()
   *  <p> This is the constructor for this class.
   *  @param st  = Brine Plot Data Structure
   */

  public brineStiffPlot( brinePlotStruct st, int X, int Y )
  {
    this.st = st;

    convertDataToPixels(X, Y);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	if (st != null)
	  st.delete();
	st = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;
    int w = this.getWidth();
    int h = this.getHeight();

    image = new BufferedImage( w, h, BufferedImage.TYPE_INT_RGB );

    Graphics2D g2 = image.createGraphics();
    g2.setColor( Color.white );
    g2.fillRect( 0, 0, w, h );
    this.paint(g2);
    g2.dispose();

    return (image);
  }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setData()
   *  <p> This method will set the Brine Plot Data Structure
   *  @param st = Brine Plot Data Structure
   */

  public void setData( brinePlotStruct st )
  {
	this.st = st;

    if (this.st != null)
    {
      convertDataToPixels( 0, 0 );
	}
  }

  /* ================================================================== *
   * ------------------------ SUPPORT METHODS ------------------------- *
   * ================================================================== */

  /** Method void convertDataToPixels()
   *  <p> This method will set the Plot Data
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  private void convertDataToPixels(int X, int Y)
  {
    int    i,j;
    int    iRatio = 0;
    int    iRows  = 6;
    int    idraw  = 0;
    double dx     =   0.0;
    double dy     =   0.0;
    double dXMin  =   0.0;
    double dXMax  = 100.0;
    double dYMax  =  75.0;
    double dYMin  =   0.0;
    double dYIncr =  25.0;

    iX = new int [iRows];
    iY = new int [iRows];

    i = 0;

    if (st != null)
    {
      for (j=1; j<4; j++)
      {
        // Set the Y-Axis Data

        dy = dYMin + (double) j * dYIncr;
        iRatio = (int)((double)iYHeight * (dy - dYMin) / (dYMax - dYMin));
        idraw = iYBegin + iRatio;

	    switch (j)
  	    {
	      case 1:
	        // Na + K
	        dx = 100.0 - (st.per_meql_cations[2] + st.per_meql_cations[3]);
            iX[0]  = X + iXBegin + (int) dx;
            iY[0]  = Y + idraw;

	        // Cl + Br(?) + I(?)
	        dx = st.per_meql_anions[0];
	        if (st.per_meql_anions[1] > 0.0)
	          dx = dx + st.per_meql_anions[1];
	        if (st.per_meql_anions[2] > 0.0)
	          dx = dx + st.per_meql_anions[2];

            iX[1]  = X + iXBegin + 100 + (int) dx;
            iY[1]  = Y + idraw;
	        break;

	      case 2:
	        // HCO3 + CO3
	        dx = st.per_meql_anions[4] + st.per_meql_anions[5];
            iX[2]  = X + iXBegin + 100 + (int) dx;
            iY[2]  = Y + idraw;

            // Ca
	        dx = 100.0 - st.per_meql_cations[0];
            iX[5]  = X + iXBegin + (int) dx;
            iY[5]  = Y + idraw;
	        break;

	      case 3:
	        // SO4
	        dx = st.per_meql_anions[3];
            iX[3]  = X + iXBegin + 100 + (int) dx;
            iY[3]  = Y + idraw;

	        // Mg
	        dx = 100.0 - st.per_meql_cations[1];
            iX[4]  = X + iXBegin + (int) dx;
            iY[4]  = Y + idraw;
	        break;
	    }
      }
    }
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawLinearXAxisGrid()
   *  <p> This method will draw the Linear X-Axis Grid Lines
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawLinearXAxisGrid(Graphics g, int X, int Y)
  {
    int    i      = 0;
    int    idraw  = 0;
    int    idel   = 0;
    int    iRatio = 0;
    int    iXinc  = 0;
    int    itmp   = 0;
    double dx     = 0.0;
    double del    = 0.0;
    double dXMax  = 200.0;
    double dXMin  =   0.0;
    double dXIncr =  25.0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    iXinc = (int) ((double)iXWidth * dXIncr / (dXMax - dXMin));
    idraw = iXBegin;

    while (idraw < iXBegin+iXWidth+5)
    {
      dx     = dXMin + (double) i * dXIncr;
      iRatio = (int) ((double) iXWidth *
                      (dx - dXMin) / (dXMax - dXMin));
      idraw  = iXBegin + iRatio;

      if (idraw < iXBegin+iXWidth+5)
      {
        g.setColor(Color.lightGray);
        g.drawLine(X+idraw, Y+iYBegin, X+idraw, Y+iYBegin+iYHeight);

        g.setColor(Color.black);
        dx = -100.0 + Math.round(dx*1000)/1000.0;
        itmp = (int) Math.abs(dx);
        g.drawString(""+itmp, X+idraw, Y+iYBegin);
      }

      i++;
    }
  }

  /** Method drawLinearYAxisGrid()
   *  <p> This method will draw the Linear Y-Axis Grid Lines
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawLinearYAxisGrid(Graphics g, int X, int Y)
  {
    int    i = 0;
    int    idraw  = 0;
    int    idel   = 0;
    int    iRatio = 0;
    int    iYinc  = 0;
    String stmp   = "";
    double dy     = 0.0;
    double del    = 0.0;
    double dYMax  =  75.0;
    double dYMin  =   0.0;
    double dYIncr =  25.0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    iYinc = (int) ((double)iYHeight *dYIncr / (dYMax - dYMin));
    idraw = iYBegin + iYHeight;

    for (i=0; i<4; i++)
    {
      dy = dYMin + (double) i * dYIncr;
      iRatio = (int)((double)iYHeight * (dy - dYMin) / (dYMax - dYMin));
      idraw = iYBegin + iRatio;

      if (idraw > iYBegin-5)
      {
        g.setColor(Color.lightGray);
        g.drawLine(X+iXBegin, Y+idraw, X+iXBegin+iXWidth, Y+idraw);

        g.setColor(Color.black);
        switch (i)
        {
		  case 0:
		    break;
		  case 1:
            stmp = new String("Na+K");
            g.drawString(stmp, X+5, Y+idraw);

            stmp = new String("Cl");
	        if (st.per_meql_anions[1] > 1.0)
	          stmp = new String( stmp + "+Br" );
	        if (st.per_meql_anions[2] > 1.0)
	          stmp = new String( stmp + "+I" );

            g.drawString(stmp, X+iXBegin+iXWidth+5, Y+idraw);
		    break;
		  case 2:
            stmp = new String("Ca");
            g.drawString(stmp, X+5, Y+idraw);

            stmp = new String("HCO3+CO3");
            g.drawString(stmp, X+iXBegin+iXWidth+5, Y+idraw);
		    break;
		  case 3:
            stmp = new String("Mg");
            g.drawString(stmp, X+5, Y+idraw);

            stmp = new String("SO4");
            g.drawString(stmp, X+iXBegin+iXWidth+5, Y+idraw);
		    break;
		}
      }
    }
  }

  /** Method drawPlotData()
   * <p> This method will plot the data
   *  @param g = Graphics pointer.
   */

  public void drawPlotData(Graphics g)
  {
	g.setColor( new Color( 0, 180, 0 ) );

    if (iX != null)
      g.fillPolygon(iX, iY, 6);
  }

  /** Method draw()
   * <p> This method will draw the stiff diagram
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void draw(Graphics g, int X, int Y)
  {
	String sLabel = "";
	int    length = 0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);

    sLabel = new String( "Stiff Diagram" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+100-length*4), Y+iYBegin-40 );

    sLabel = new String( "Cations" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+30-length*4),  Y+iYBegin-20 );

    sLabel = new String( "% meq/l" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+100-length*4), Y+iYBegin-20 );

    sLabel = new String( "Anions" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+170-length*4), Y+iYBegin-20 );

    drawLinearXAxisGrid(g, X, Y);
    drawLinearYAxisGrid(g, X, Y);
    drawPlotData(g);
  }

  /** Method paint( Graphics g )
   * <p> This method will paint the stiff diagram Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
    draw(g, 0, 0);
  }
}

/*
 *  @version 1.1 06/10/2011
 *  @author  John Victorine
 */
