/*
 * @brineSamplePlot.java Version 1.1 06/13/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;

import brine.brineStruct;           // Raw Brine Data
import brine.brineListStruct;       // Raw Brine Data List
import brine.plot.brinePlotStruct;  // Common Brine Data Sample
import brine.plot.brineStiffPlot;   // Stiff Plot
import brine.plot.brineCollinsPlot; // Collins Bar Diagram

/** CLASS brineSamplePlot
 *  <p> This Class will create the Stiff Diagram Plot.
 *
 *  @version 1.1 06/13/2011
 *  @author  John R. Victorine
 */

public class brineSamplePlot extends Canvas
{
  private int             iRow    = 0;
  private brineListStruct stBrine = null; // Common Brine Data
  private brinePlotListStruct st  = null; // Plot Brine Data List

  private brineStiffPlot   plotStiff   = null; // Stiff Plot Diagram
  private brineCollinsPlot plotCollins = null; // Collins Diagram

  // Brine Math Variables

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

  // X Plot Position

  private static final int _HEADER = 200;

  private static final int _TEXT   =  20;
  private static final int _TEXT2  = 270;

  private static final int _UNIT   =  10;

  private static final int _CA     =  60;
  private static final int _MG     = 115;
  private static final int _NA     = 170;
  private static final int _K      = 225;

  private static final int _CL     = 280;
  private static final int _BR     = 335;
  private static final int _I      = 390;
  private static final int _SO4    = 445;
  private static final int _HCO3   = 500;
  private static final int _CO3    = 555;

  // Y Plot Position

  private static final int _TOP    =  20;  // Brine Sample Plot
  private static final int _LINE_0 =  40;  // Brine Record
  private static final int _LINE_1 =  60;
  private static final int _LINE_2 =  80;
  private static final int _LINE_3 = 100;
  private static final int _LINE_4 = 120;

  private static final int _LINE_5 = 140;  // Source

  private static final int _LINE_6 = 160;  // Table Header
  private static final int _LINE_7 = 180;  // Table - mg/l
  private static final int _LINE_8 = 200;  // Table - meq/l
  private static final int _LINE_9 = 220;  // Table - % meq/l

  private static final int _LINE_10 = 510;  // General Analysis
  private static final int _LINE_11 = 530;
  private static final int _LINE_12 = 550;
  private static final int _LINE_13 = 570;
  private static final int _LINE_14 = 590;

  // Stiff Plot Position

  private static final int _X_STIFF = 10;
  private static final int _Y_STIFF = 290;

  // Collins Bar Plot Position

  private static final int _X_COLLINS = 325;
  private static final int _Y_COLLINS = 230;

  // Plot Dimensions

  private int iWidth  = 625;  // Width of Plot
  private int iHeight = 650;  // Height of Plot with the Titles

  /** CONSTRUCTOR brineSamplePlot()
   *  <p> This is the constructor for this class.
   *  @param stBrine = Brine Data List Structure
   */

  public brineSamplePlot( brineListStruct stBrine )
  {
    this.stBrine = stBrine;
    setBrineData();
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	if (st != null)
	  st.delete();
	st = null;

	stBrine = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;
    int w = this.getWidth();
    int h = this.getHeight();

    image = new BufferedImage( w, h, BufferedImage.TYPE_INT_RGB );

    Graphics2D g2 = image.createGraphics();
    g2.setColor( Color.white );
    g2.fillRect( 0, 0, w, h );
    this.paint(g2);
    g2.dispose();

    return (image);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setBrineData( brineListStruct stBrine )
   * <p> This method will set the Brine Data List Structure
   * @param stBrine - Brine Data List Structure
   */

  public void setBrineData()
  {
    int i,j;
    int lenAnions  = 0;
    int lenCations = 0;

	st = new brinePlotListStruct();

	if (stBrine != null)
	{
	  lenAnions  = brine.math.brineMath.getRows( _ANIONS );
	  lenCations = brine.math.brineMath.getRows( _CATIONS );

	  st         = new brinePlotListStruct();
	  st.iCount  = stBrine.iCount;
	  st.stItem = new brinePlotStruct[st.iCount];

	  for (i=0; i<stBrine.iCount; i++)
	  {
	    st.stItem[i] = new brinePlotStruct();

        st.stItem[i].mgl_cations      = new double[lenCations];
        st.stItem[i].meql_cations     = new double[lenCations];
        st.stItem[i].per_meql_cations = new double[lenCations];

        st.stItem[i].mgl_anions       = new double[lenAnions];
        st.stItem[i].meql_anions      = new double[lenAnions];
        st.stItem[i].per_meql_anions  = new double[lenAnions];
	  }

	  for (i=0; i<stBrine.iCount; i++)
	  {
		st.stItem[i].depthStart  = stBrine.stItem[i].dTOP;
		st.stItem[i].depthEnd    = stBrine.stItem[i].dBASE;

	    st.stItem[i].mgl_cations =
	      brine.math.brineMath.get_mg_per_liter( _CATIONS, i, stBrine );

	    st.stItem[i].mgl_anions =
	      brine.math.brineMath.get_mg_per_liter( _ANIONS, i, stBrine );

        st.stItem[i].meql_cations =
          brine.math.brineMath.mg_to_meq_per_liter(
  		    _CATIONS, st.stItem[i].mgl_cations );

        st.stItem[i].meql_anions =
          brine.math.brineMath.mg_to_meq_per_liter(
	  	      _ANIONS, st.stItem[i].mgl_anions );

        st.stItem[i].per_meql_cations =
          brine.math.brineMath.percent_meq_per_liter(
		      _CATIONS, st.stItem[i].meql_cations );

        st.stItem[i].per_meql_anions =
          brine.math.brineMath.percent_meq_per_liter(
		      _ANIONS, st.stItem[i].meql_anions );
      }
	}
  }

  /** Method setRow()
   * <p> This method will set the plot data to plot
   * @param iRow = The Data Row
   */

  public void setRow(int iRow)
  {
    this.iRow = iRow;

    plotStiff = new brineStiffPlot( st.stItem[iRow],
                                    _X_STIFF, _Y_STIFF );
    plotCollins = new brineCollinsPlot( st.stItem[iRow] );
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method draw()
   * <p> This method will draw the stiff diagram
   *  @param g - Graphics pointer.
   */

  public void draw(Graphics g)
  {
	String sLabel = "";
	int    length = 0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 14);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);

    sLabel = new String( "Brine Sample Plot" );
    length = sLabel.length();
    g.drawString( sLabel, iWidth/2 - length*4 , _TOP );

    fsb  = new Font("Monospaced", Font.BOLD, 12);
    g.setFont( fsb );

    if (stBrine != null)
    {
	  if (stBrine.stItem[iRow] != null)
	  {
        if (stBrine.stItem[iRow].sName.length() > 0)
        {
          sLabel = new String( stBrine.stItem[iRow].sName );
          if (stBrine.stItem[iRow].sAPI.length() > 0)
            sLabel = new String( sLabel + " (" + stBrine.stItem[iRow].sAPI + ")" );
          sLabel = new String( "Well: " + sLabel );
          length = sLabel.length();
          g.drawString( sLabel, iWidth/2 - length*4 , _LINE_0 );
        }

        g.drawString( "State:  " + stBrine.stItem[iRow].state,   _TEXT, _LINE_1 );
        g.drawString( "County: " + stBrine.stItem[iRow].sCounty, _TEXT, _LINE_2 );
        g.drawString( "Field:  " + stBrine.stItem[iRow].sField,  _TEXT, _LINE_3 );

        g.drawString( "Formation: " +
                      stBrine.stItem[iRow].sFORM,   _TEXT2, _LINE_1 );

        sLabel = new String( "" + stBrine.stItem[iRow].dTOP );
        if (stBrine.stItem[iRow].dBASE > 0.0)
          sLabel = new String( sLabel +
                               " - " + stBrine.stItem[iRow].dBASE );
        g.drawString( "Depth:     " +  sLabel,   _TEXT2, _LINE_2 );
        g.drawString( "Latitude:  " +
                      stBrine.stItem[iRow].dLatitude,   _TEXT2, _LINE_3 );
        g.drawString( "Longitude: " +
                      stBrine.stItem[iRow].dLongitude,  _TEXT2, _LINE_4 );

        plotTable( g );

        drawDiagrams( g );

        fsb  = new Font("Monospaced", Font.BOLD, 12);
        g.setFont( fsb );

        g.setColor(Color.black);

        sLabel = new String( "General Analysis" );
        length = sLabel.length();
        g.drawString( sLabel, iWidth/2 - length*4 , _LINE_10 );

        if (stBrine.stItem[iRow].dPH > 0.0)
          g.drawString( "PH: "   +
                      stBrine.stItem[iRow].dPH,   _TEXT, _LINE_11 );
        else
          g.drawString( "PH: ",   _TEXT, _LINE_11 );

        if (stBrine.stItem[iRow].dDEG > 0.0)
          g.drawString( "Temp. F deg: " +
                      stBrine.stItem[iRow].dDEG,  _TEXT, _LINE_12 );
        else
          g.drawString( "Temp. F deg: ", _TEXT, _LINE_12 );

        if (stBrine.stItem[iRow].dSPGR > 0.0)
          g.drawString( "Specific Gravity: "   +
                      stBrine.stItem[iRow].dSPGR, _TEXT, _LINE_13 );
        else
          g.drawString( "Specific Gravity: ", _TEXT, _LINE_13 );

        if (stBrine.stItem[iRow].dOHM > 0.0)
          g.drawString( "Rw: "   +
                      stBrine.stItem[iRow].dOHM, _TEXT2, _LINE_11 );
        else
          g.drawString( "Rw: ", _TEXT2, _LINE_11 );

        if (stBrine.stItem[iRow].dOHM75 > 0.0)
          g.drawString( "Rw @ 75 F deg: " +
                      stBrine.stItem[iRow].dOHM75, _TEXT2, _LINE_12 );
        else
          g.drawString( "Rw @ 75 F deg: ", _TEXT2, _LINE_12 );

        if (stBrine.stItem[iRow].dOHME > 0.0)
          g.drawString( "Est. Rw: "   +
                      stBrine.stItem[iRow].dOHME, _TEXT2, _LINE_13 );
        else
          g.drawString( "Est. Rw: ", _TEXT2, _LINE_13 );


        if (stBrine.stItem[iRow].dTDS > 0.0)
          sLabel = new String( "Total Dissolved Solids: " +
                     stBrine.stItem[iRow].dTDS + " " + "mg/l" );
        else
          sLabel = new String( "Total Dissolved Solids: " );

        length = sLabel.length();
        g.drawString( sLabel, iWidth/2 - length*4 , _LINE_14 );
      }
    }
  }

  /** Method plotTable( Graphics g )
   *  <p> This method will paint the analyte Data Table
   */

  public void plotTable( Graphics g )
  {
	String sLabel = "";
	int    length = 0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);
    if (stBrine.stItem[iRow].source.length() > 0)
    {
      sLabel = new String( "Source: " +
                           stBrine.stItem[iRow].source.length() );
      length = sLabel.length();
      g.drawString( sLabel, iWidth/2 - length*4, _LINE_5 );
    }

    g.setColor(new Color( 0, 180, 0 ));
    g.drawLine(0, _LINE_6-13, iWidth, _LINE_6-13);

    g.setColor(Color.black);

    g.drawString( " Unit ",  _UNIT, _LINE_6 );

    g.drawString( "Ca    ",  _CA,   _LINE_6 );
    g.drawString( "Mg    ",  _MG,   _LINE_6 );
    g.drawString( "Na    ",  _NA,   _LINE_6 );
    g.drawString( "K     ",  _K,    _LINE_6 );

    g.drawString( "Cl    ",  _CL,   _LINE_6 );
    g.drawString( "Br    ",  _BR,   _LINE_6 );
    g.drawString( "I     ",  _I,    _LINE_6 );
    g.drawString( "SO4   ",  _SO4,  _LINE_6 );
    g.drawString( "HCO3  ",  _HCO3, _LINE_6 );
    g.drawString( "CO3   ",  _CO3,  _LINE_6 );

    g.setColor(new Color( 0, 180, 0 ));
    g.drawLine(0, _LINE_6+7, iWidth, _LINE_6+7);
    g.drawLine(_CA-4, _LINE_6-13, _CA-4, _LINE_9+2);
    g.drawLine(_CL-4, _LINE_6-13, _CL-4, _LINE_9+2);


    fsb  = new Font("Monospaced", Font.BOLD, 10);
    g.setFont( fsb );

    g.setColor(Color.black);

    g.drawString( "   mg/l",  _UNIT, _LINE_7 );

    if (st.stItem[iRow].mgl_cations[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_cations[0], _CA, _LINE_7 );
    if (st.stItem[iRow].mgl_cations[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_cations[1], _MG, _LINE_7 );
    if (st.stItem[iRow].mgl_cations[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_cations[2], _NA, _LINE_7 );
    if (st.stItem[iRow].mgl_cations[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_cations[3], _K,  _LINE_7 );

    if (st.stItem[iRow].mgl_anions[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[0], _CL,   _LINE_7 );
    if (st.stItem[iRow].mgl_anions[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[1], _BR,   _LINE_7 );
    if (st.stItem[iRow].mgl_anions[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[2], _I,    _LINE_7 );
    if (st.stItem[iRow].mgl_anions[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[3], _SO4,  _LINE_7 );
    if (st.stItem[iRow].mgl_anions[4] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[4], _HCO3, _LINE_7 );
    if (st.stItem[iRow].mgl_anions[5] > 0.0)
      g.drawString( "" + st.stItem[iRow].mgl_anions[5], _CO3,  _LINE_7 );

    g.drawString( "  meq/l",  _UNIT, _LINE_8 );

    if (st.stItem[iRow].meql_cations[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_cations[0], _CA, _LINE_8 );
    if (st.stItem[iRow].meql_cations[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_cations[1], _MG, _LINE_8 );
    if (st.stItem[iRow].meql_cations[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_cations[2], _NA, _LINE_8 );
    if (st.stItem[iRow].meql_cations[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_cations[3], _K,  _LINE_8 );

    if (st.stItem[iRow].meql_anions[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[0], _CL,   _LINE_8 );
    if (st.stItem[iRow].meql_anions[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[1], _BR,   _LINE_8 );
    if (st.stItem[iRow].meql_anions[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[2], _I,    _LINE_8 );
    if (st.stItem[iRow].meql_anions[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[3], _SO4,  _LINE_8 );
    if (st.stItem[iRow].meql_anions[4] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[4], _HCO3, _LINE_8 );
    if (st.stItem[iRow].meql_anions[5] > 0.0)
      g.drawString( "" + st.stItem[iRow].meql_anions[5], _CO3,  _LINE_8 );

    g.drawString( "% meq/l",  _UNIT, _LINE_9 );

    if (st.stItem[iRow].per_meql_cations[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_cations[0], _CA, _LINE_9 );
    if (st.stItem[iRow].per_meql_cations[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_cations[1], _MG, _LINE_9 );
    if (st.stItem[iRow].per_meql_cations[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_cations[2], _NA, _LINE_9 );
    if (st.stItem[iRow].per_meql_cations[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_cations[3], _K,  _LINE_9 );

    if (st.stItem[iRow].per_meql_anions[0] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[0], _CL,   _LINE_9 );
    if (st.stItem[iRow].per_meql_anions[1] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[1], _BR,   _LINE_9 );
    if (st.stItem[iRow].per_meql_anions[2] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[2], _I,    _LINE_9 );
    if (st.stItem[iRow].per_meql_anions[3] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[3], _SO4,  _LINE_9 );
    if (st.stItem[iRow].per_meql_anions[4] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[4], _HCO3, _LINE_9 );
    if (st.stItem[iRow].per_meql_anions[5] > 0.0)
      g.drawString( "" + st.stItem[iRow].per_meql_anions[5], _CO3,  _LINE_9 );

    g.setColor(new Color( 0, 180, 0 ));
    g.drawLine(0, _LINE_9+2, iWidth, _LINE_9+2);
  }

  /** Method drawDiagrams()
   * <p> This method will draw the stiff diagram
   *  @param g - Graphics pointer.
   */

  public void drawDiagrams(Graphics g)
  {
    plotStiff.draw( g, _X_STIFF, _Y_STIFF );
    plotCollins.draw( g, _X_COLLINS, _Y_COLLINS );
  }

  /** Method paint( Graphics g )
   * <p> This method will paint the stiff diagram Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
    draw(g);
  }
}

/*
 *  @version 1.1 06/13/2011
 *  @author  John Victorine
 */
