/*
 * @brinePlotTrack.java Version 1.1 06/08/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;

import brine.brineListStruct;          // Raw Brine Data
import brine.plot.brinePlotListStruct; // Common Brine Data List
import brine.plot.brinePlotStruct;     // Common Brine Data Sample

/** CLASS brinePlotTrack
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot Brine Tracks.
 *
 *  @version 1.1 06/08/2011
 *  @author  John R. Victorine
 */

public class brinePlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private double dIncrementDepth  = 0.0;
  private int    iStartDepthLabel = 10;

  public static final Color cCA   = new Color( 160, 160, 160 );
  public static final Color cMG   = new Color(  34, 139,  34 );
  public static final Color cNA   = new Color(   0, 191, 255 );
  public static final Color cK    = new Color(   0,   0, 180 );

  public static final Color CLR_CATIONS[] = { cCA, cMG, cNA, cK };

  public static final Color cCL   = new Color( 173, 255,  47 );
  public static final Color cBR   = new Color( 180,   0,   0 );
  public static final Color cI    = new Color( 180,   0, 180 );
  public static final Color cSO4  = new Color( 102, 102,   0 );
  public static final Color cHCO3 = new Color( 250, 128, 114 );
  public static final Color cCO3  = new Color( 255, 182, 193 );

  public static final Color CLR_ANIONS[] = { cCL, cBR, cI, cSO4, cHCO3, cCO3 };

/*
  public static final Color cRED     = new Color( 180,   0,   0 );
  public static final Color cGREEN   = new Color( 0,   180,   0 );
  public static final Color cBLUE    = new Color( 0,     0, 180 );
  public static final Color cCYAN    = new Color(  0,  180, 180 );
  public static final Color cMAGENTA = new Color( 180,   0, 180 );
  public static final Color cYELLOW  = new Color( 180, 180,   0 );
  public static final Color cORANGE  = new Color( 204, 102,   0 );
  public static final Color cBROWN   = new Color( 102, 102,   0 );
//  public static final Color cBROWN   = new Color(  51,  51,   0 );

  public static final Color COLORS[] =  // Color for Log Curves
      { cRED, cGREEN, cBLUE, cBROWN, cMAGENTA, cORANGE, cYELLOW, cCYAN };
*/
//  public static final Color COLORS[] =   // Color for Log Curves
//            {Color.red,     Color.green,  Color.blue,   Color.cyan,
//             Color.magenta, Color.yellow, Color.orange, Color.pink};

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

//  private brineListStruct     stBrine = null; // Brine List Structure
  private brinePlotListStruct   st     = null; // Common Brine Data List

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR brinePlotTrack( int iDataType,
   *                              double depthStart,
   *                              double depthEnd,
   *                              int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public brinePlotTrack( int iDataType,
                         double depthStart,
                         double depthEnd,
                         int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
//	stBrine  = null;

	if (st != null)
	  st.delete();
	st = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];
  }

  /** Method setBrineData( brineListStruct stBrine )
   * <p> This method will set the Brine Data List Structure
   * @param stBrine - Brine Data List Structure
   */

  public void setBrineData(brineListStruct stBrine)
  {
//	this.stBrine = stBrine;
    int i,j;
    int lenAnions  = 0;
    int lenCations = 0;

	st = new brinePlotListStruct();

	if (stBrine != null)
	{
	  lenAnions  = brine.math.brineMath.getRows( _ANIONS );
	  lenCations = brine.math.brineMath.getRows( _CATIONS );

	  st         = new brinePlotListStruct();
	  st.iCount  = stBrine.iCount;
	  st.stItem = new brinePlotStruct[st.iCount];

	  for (i=0; i<stBrine.iCount; i++)
	  {
	    st.stItem[i] = new brinePlotStruct();

        st.stItem[i].mgl_cations      = new double[lenCations];
        st.stItem[i].meql_cations     = new double[lenCations];
        st.stItem[i].per_meql_cations = new double[lenCations];

        st.stItem[i].mgl_anions       = new double[lenAnions];
        st.stItem[i].meql_anions      = new double[lenAnions];
        st.stItem[i].per_meql_anions  = new double[lenAnions];
	  }

	  for (i=0; i<stBrine.iCount; i++)
	  {
		st.stItem[i].depthStart  = stBrine.stItem[i].dTOP;
		st.stItem[i].depthEnd    = stBrine.stItem[i].dBASE;

	    st.stItem[i].mgl_cations =
	      brine.math.brineMath.get_mg_per_liter( _CATIONS, i, stBrine );

	    st.stItem[i].mgl_anions =
	      brine.math.brineMath.get_mg_per_liter( _ANIONS, i, stBrine );

        st.stItem[i].meql_cations =
          brine.math.brineMath.mg_to_meq_per_liter(
  		    _CATIONS, st.stItem[i].mgl_cations );

        st.stItem[i].meql_anions =
          brine.math.brineMath.mg_to_meq_per_liter(
	  	      _ANIONS, st.stItem[i].mgl_anions );

        st.stItem[i].per_meql_cations =
          brine.math.brineMath.percent_meq_per_liter(
		      _CATIONS, st.stItem[i].meql_cations );

        st.stItem[i].per_meql_anions =
          brine.math.brineMath.percent_meq_per_liter(
		      _ANIONS, st.stItem[i].meql_anions );
      }
	}
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g,
   *                   int iROCK,
   *                   int iStartTrack,
   *                   int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iBRINE      = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g,
                        int iBRINE,
                        int iStartTrack,
                        int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc      = 0;
    int    iLog       = 0;
    int    iCycle     = 1;
    double dXinc      = 0.0;
    double dTemp      = 0.0;
    int    iIncrement = iWidthTrack/5;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iTrack     = iBRINE - iqstrat.iqstratTracksStruct.BRINE_START;
    int    iValue     = 0;
    int    iY1        = 0;
    int    iY2        = 0;
    double dMinimum   = 0.0;
    double dMaximum   = 100.0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.gray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
      else
      {
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawTrackLabels( Graphics g,
   *                          int iBRINE,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iBRINE       = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackLabels( Graphics g,
                               int iBRINE,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    iContinue  = 0;
    int    iLine      = -1;
    int    length     = 0;
    String sLabel     = "";
    String sTemp      = "";
    double dMinimum   = 0.0;
    double dMaximum   = 100.0;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iTrack     = iBRINE - iqstrat.iqstratTracksStruct.BRINE_START;
    int    ANALYTE[]  = null;
    Color  COLORS[]   = null;
    int    len        = 0;
    int    iX         = 0;
    int    iY         = 0;

    Font fsb = new Font("Serif", Font.BOLD, 10);
    g.setFont( fsb );

    switch (iTrack)
    {
	  case _CATIONS:
	    ANALYTE = CATIONS;
	    COLORS  = CLR_CATIONS;
	    len     = brine.math.brineMath.getRows( _CATIONS );
	    break;

	  case _ANIONS:
	    ANALYTE = ANIONS;
	    COLORS  = CLR_ANIONS;
	    len     = brine.math.brineMath.getRows( _ANIONS );
	    break;
	}

    sLabel = new String( "% mg/l" );
    length = sLabel.length();

    g.setColor(Color.black);
    g.drawString( sLabel,
                  (iStartTrack + iWidthTrack / 2 - length * 4),
                  LABELSTART + 28 );

    g.drawString( "" + dMinimum,
                  iStartTrack + 1,
                  LABELSTART + 28 );

    sTemp = new String( "" + dMaximum );
    length = sTemp.length();
    g.drawString( "" + dMaximum, iEndTrack - length * 5,
                  LABELSTART + 28 );

    iLine = 1;
    for (i=0; i<len; i++)
    {
	  switch (i)
	  {
	    case 1:
	    case 3:
	    case 5:
          iX = iStartTrack + 10;
	      break;
	    case 0:
          iX = iStartTrack + 60;
	      break;
	    case 2:
          iX = iStartTrack + 60;
	      iLine = 2;
	      break;
	    case 4:
          iX = iStartTrack + 60;
	      iLine = 3;
	      break;
	  }

      g.setColor( COLORS[i] );

      sLabel = new String( brine.brineStandardTools.CURVES[ANALYTE[i]][1] );
      length = sLabel.length();

      iY = LABELSTART  + (iLine + 2) * 14;

      g.drawString( sLabel, iX, iY );
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    sLabel = new String( iqstrat.iqstratTracksStruct.BRINE_TRACK_DESC[iTrack] );
    length = sLabel.length();
    g.drawString( sLabel,
                  iStartTrack+iWidthTrack/2-(9*length)/4,
                  LABELSTART+14 );
  }

  /** Method drawTrackCurves( Graphics g,
   *                          int iBRINE,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the curves on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iBRINE       = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackCurves( Graphics g,
                               int iBRINE,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    i,j;
    int    iPlot      = 0;
    int    iX1        = 0;
    int    iX2        = 0;
    int    iY1        = 0;
    int    iY2        = 0;
    double dRatio     = 0.0;
    double dMinimum   = 0.0;
    double dMaximum   = 100.0;
    double dLog[]     = null;
    double depth      = 0.0;
    double data       = 0.0;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iTrack     = iBRINE - iqstrat.iqstratTracksStruct.BRINE_START;
    int    ANALYTE[]  = null;
    Color  COLORS[]   = null;
    int    len        = 0;

    switch (iTrack)
    {
	  case _CATIONS:
	    ANALYTE = CATIONS;
	    COLORS  = CLR_CATIONS;
	    len     = brine.math.brineMath.getRows( _CATIONS );
	    break;

	  case _ANIONS:
	    ANALYTE = ANIONS;
	    COLORS  = CLR_ANIONS;
	    len     = brine.math.brineMath.getRows( _ANIONS );
	    break;
	}

    if (st != null)
    {
	  for (i=0; i<st.iCount; i++)
	  {
        switch (iTrack)
        {
    	  case _CATIONS:
            dLog = st.stItem[i].per_meql_cations;
      	    break;

      	  case _ANIONS:
            dLog = st.stItem[i].per_meql_anions;
      	    break;
       	}

        depth = st.stItem[i].depthStart;
        dRatio = (double) iLogHeight * (depth - depthStart) /
                 (depthEnd - depthStart);
        iY1 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY1 = iHeight - (int) dRatio;

        iY2 = 4;
        if (st.stItem[i].depthEnd > st.stItem[i].depthStart)
        {
          depth = st.stItem[i].depthEnd;
          dRatio = (double) iLogHeight * (depth - depthStart) /
                   (depthEnd - depthStart);
          iY2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iY2 = iHeight - (int) dRatio;
	    }

        if ((iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
            (iY1 < iHeight) && (iY2 < iHeight))
        {
          iPlot = 0;
          if ((iY1 >= PLOT_TITLES) && (iY1 < iHeight))
          {
            iPlot = 1;
          }

          if ((iY2 > PLOT_TITLES) && (iY2 <= iHeight))
          {
            iPlot = 2;
		  }

	      for (j=0; j<len; j++)
	      {
            if (dLog[j] > 0.0)
            {
              data  = dLog[j];
              if (data < 0.0)   data = 0.0;
              if (data >= 98.0) data = 98.0;

              dRatio = (double) iWidthTrack * data / dMaximum;
              iX1 = iStartTrack + (int) dRatio;

              g.setColor(COLORS[j]);

              if ((iPlot == 1) || (iPlot == 2))
              {
                if (iPlot == 2)
                {
                  g.fillRect(iX1, iY1, 4, iY2-iY1);
		   	    }
		  	    else
                  g.fillRect(iX1, iY1, 4, 4);
			  }
		    }
          }
		}
	  }
	}
  }

  /** Method draw( Graphics g, int iBRINE, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iBRINE       = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iBRINE, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_BRINE;

    if ((iBRINE == iqstrat.iqstratTracksStruct._BRINE_LAS_CATIONS) ||
        (iBRINE == iqstrat.iqstratTracksStruct._BRINE_LAS_ANIONS))
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iBRINE, iStartTrack, iWidthTrack );
      drawTrackLabels( g, iBRINE, iStartTrack, iWidthTrack );
      drawTrackCurves( g, iBRINE, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 06/08/2011
 *  @author  John Victorine
 */
