/*
 * @brinePlotImageTrack.java Version 1.1 06/14/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

//import cmn.cmnStruct;                   // Common Data Structure
//import util.utilBrowser;

import brine.brineListStruct;           // Raw Brine Data
import brine.plot.brinePlotListStruct;  // Common Brine Data List
import brine.plot.brinePlotStruct;      // Common Brine Data Sample

import brine.gui.brineStiffPlotFrame;   // Stiff Diagram Dialog
import brine.gui.brineCollinsPlotFrame; // Collins Bar Diagram Dialog
import brine.gui.brineSamplePlotFrame;  // Brine Sample Plot Dialog
import brine.gui.brinePiperPlotFrame;   // Brine Piper Plot Dialog

/** CLASS brinePlotImageTrack
 *  <p> This Class will create the Brine Plot Image Track Selection.
 *
 *  @version 1.1 06/14/2011
 *  @author  John R. Victorine
 */

public class brinePlotImageTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

//  private cmnStruct           stCMN   = null; // Common Data Structure
  private brineListStruct     stBrine = null; // Brine List Structure
  private brinePlotListStruct st      = null; // Common Brine Data List

  // -- Diagrams

  private brineStiffPlotFrame   pStiff   = null; // Stiff Plot Diagram
  private brineCollinsPlotFrame pCollins = null; // Collins Plot Diagram
  private brineSamplePlotFrame  pSample  = null; // Sample Plot Diagram
  private brinePiperPlotFrame   piper    = null; // Piper Plot Diagram

  // Brine Image Variables

  private int  iSelected[] = null;

  private int  iTotal     = 0;
  private int  iYMouse[]  = null;
  private int  iYMouse2[] = null;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  private double dIncrementDepth  = 0.0;

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR brinePlotImageTrack( int iDataType,
   *                                   double depthStart,
   *                                   double depthEnd,
   *                                   int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   */

  public brinePlotImageTrack( int iDataType,
                              double depthStart,
                              double depthEnd,
                              int iScale )
  {
    this.iDataType = iDataType;

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stBrine   = null;

    iSelected = null;
    iYMouse   = null;
    iYMouse2  = null;

	if (st != null)
	  st.delete();
	st = null;

	if (pStiff != null)
	  pStiff.close();
	pStiff = null;

	if (pCollins != null)
	  pCollins.close();
	pCollins = null;

    if (pSample != null)
      pSample.close();
    pSample = null;

    if (piper != null)
      piper.close();
    piper = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight         = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];

    convertCoreToPixels();
  }

  /** Method setBrineData( brineListStruct stBrine )
   * <p> This method will set the Brine Data List Structure
   * @param stBrine - Brine Data List Structure
   */

  public void setBrineData(brineListStruct stBrine)
  {
	this.stBrine = stBrine;

    int i,j;
    int lenAnions  = 0;
    int lenCations = 0;

	st = new brinePlotListStruct();

	if (stBrine != null)
	{
	  lenAnions  = brine.math.brineMath.getRows( _ANIONS );
	  lenCations = brine.math.brineMath.getRows( _CATIONS );

	  st         = new brinePlotListStruct();
	  st.iCount  = stBrine.iCount;
	  st.stItem = new brinePlotStruct[st.iCount];

	  for (i=0; i<stBrine.iCount; i++)
	  {
	    st.stItem[i] = new brinePlotStruct();

        st.stItem[i].mgl_cations      = new double[lenCations];
        st.stItem[i].meql_cations     = new double[lenCations];
        st.stItem[i].per_meql_cations = new double[lenCations];

        st.stItem[i].mgl_anions       = new double[lenAnions];
        st.stItem[i].meql_anions      = new double[lenAnions];
        st.stItem[i].per_meql_anions  = new double[lenAnions];
	  }

	  for (i=0; i<stBrine.iCount; i++)
	  {
		st.stItem[i].depthStart  = stBrine.stItem[i].dTOP;
		st.stItem[i].depthEnd    = stBrine.stItem[i].dBASE;

	    st.stItem[i].mgl_cations =
	      brine.math.brineMath.get_mg_per_liter( _CATIONS, i, stBrine );

	    st.stItem[i].mgl_anions =
	      brine.math.brineMath.get_mg_per_liter( _ANIONS, i, stBrine );

        st.stItem[i].meql_cations =
          brine.math.brineMath.mg_to_meq_per_liter(
  		    _CATIONS, st.stItem[i].mgl_cations );

        st.stItem[i].meql_anions =
          brine.math.brineMath.mg_to_meq_per_liter(
	  	      _ANIONS, st.stItem[i].mgl_anions );

        st.stItem[i].per_meql_cations =
          brine.math.brineMath.percent_meq_per_liter(
		      _CATIONS, st.stItem[i].meql_cations );

        st.stItem[i].per_meql_anions =
          brine.math.brineMath.percent_meq_per_liter(
		      _ANIONS, st.stItem[i].meql_anions );
      }

      convertCoreToPixels();

      if (st != null)
      {
		if (st.iCount > 0)
		{
          pStiff   = new brineStiffPlotFrame( st.stItem[0] );
          pCollins = new brineCollinsPlotFrame( st.stItem[0] );
	    }
      }
	}
  }

  /** Method convertCoreToPixels()
   *  <p> This method will convert the postion of the core into pixel location
   *      so the core can be represented by a button.
   */

  private void convertCoreToPixels()
  {
    double dStart = 0.0;
    double dEnd = 0.0;
    double dRatio = 0.0;
    int    iTemp  = 0;

    if (st != null)
    {
      iTotal = st.iCount;

      iYMouse  = new int[iTotal];
      iYMouse2 = new int[iTotal];

      iSelected = new int[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        iSelected[i] = 0;

        dStart = st.stItem[i].depthStart;
        dEnd   = st.stItem[i].depthEnd;

        dRatio = (double) iLogHeight * (dStart-depthStart)/(depthEnd-depthStart);
        iYMouse[i] = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iYMouse[i] = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight * (dEnd-depthStart)/(depthEnd-depthStart);
        iYMouse2[i] = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iYMouse2[i] = iHeight - (int) dRatio;

        if ((iYMouse2[i]-iYMouse[i]) < 10)
        {
          iYMouse2[i] = iYMouse[i] + 10;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iYMouse2[i] = iYMouse[i] - 10;
        }

        if (iYMouse2[i] < iYMouse[i])
        {
          iTemp       = iYMouse[i];
          iYMouse[i]  = iYMouse2[i];
          iYMouse2[i] = iTemp;
        }
      }
    }
  }

  /** Method computeMouseDepth( int iYm )
   * <p> This method will compute the Mouse Depth
   * @param iYm = the cursor y-coordinate
   * @return depth = depth computed from mouse cursor location
   */

  public double computeMouseDepth( int iYm )
  {
    double depth = 0.0;

    if (iYm > PLOT_TITLES)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        depth = depthStart + (depthEnd-depthStart) * (iHeight-iYm)/iLogHeight;
      else
        depth = depthStart + (depthEnd-depthStart) * (iYm-PLOT_TITLES)/iLogHeight;
    }

    return (depth);
  }

  /** Method displayDiagram( int iXm, int iYm,
   *                         int iView,
   *                         int iStartTrack, int iWidthTrack )
   * <p> This method will display the core image if it is selected
   * @param iXm         = the x position
   * @param iYm         = the y position
   * @param iView       = The Plot
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void displayDiagram( int iXm, int iYm,
                              int iView,
                              int iStartTrack, int iWidthTrack )
  {
    int iFound = -1;

    pStiff.setVisible(false);
    pCollins.setVisible(false);

    for (int k=0; k<iTotal; k++)
    {
      if ((iXm > iStartTrack && iXm < iStartTrack+iWidthTrack) &&
          (iYm > iYMouse[k] && iYm < iYMouse2[k]))
      {
        switch (iView)
        {
          case cmn.cmnStruct.STIFF:
            if (pStiff != null)
            {
			  pStiff.setData( st.stItem[k] );
			  pStiff.setVisible(true);
			}
            break;
          case cmn.cmnStruct.COLLINS:
            if (pCollins != null)
            {
			  pCollins.setData( st.stItem[k] );
			  pCollins.setVisible(true);
			}
            break;
          case cmn.cmnStruct.PIPER:
            break;
        }
      }
    }
  }

  /** Method displaySamplePlot( int iXm, int iYm,
   *                            int iView,
   *                            int iStartTrack, int iWidthTrack )
   * <p> This method will display the Brine Sample Plot if it is selected
   * @param iXm         = the x position
   * @param iYm         = the y position
   * @param iView       = The Plot
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void displaySamplePlot( int iXm, int iYm,
                                 int iView,
                                 int iStartTrack, int iWidthTrack )
  {
    if (pSample != null)
      pSample.close();
    pSample = null;

    for (int k=0; k<iTotal; k++)
    {
      if ((iXm > iStartTrack && iXm < iStartTrack+iWidthTrack) &&
          (iYm > iYMouse[k] && iYm < iYMouse2[k]))
      {
        switch (iView)
        {
          case cmn.cmnStruct.STIFF:
          case cmn.cmnStruct.COLLINS:
            if (stBrine != null)
            {
  	          if (stBrine.stItem != null)
	          {
	            pSample = new brineSamplePlotFrame( stBrine );
	            pSample.setRow(k);
	          }
	        }
	        break;
          case cmn.cmnStruct.PIPER:
            break;
	    }
      }
    }
  }

  /** Method displayPiperPlot( int iXm, int iYm,
   *                           int iView,
   *                           int iStartTrack, int iWidthTrack )
   * <p> This method will display the Piper Diagram if it is selected
   * @param iXm         = the x position
   * @param iYm         = the y position
   * @param iView       = The Plot
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

 public void displayPiperPlot( int iXm, int iYm,
                               int iView,
                               int iStartTrack, int iWidthTrack )
 {
    if (piper != null)
      piper.close();
    piper = null;

    for (int k=0; k<iTotal; k++)
    {
      if ((iXm > iStartTrack && iXm < iStartTrack+iWidthTrack) &&
          (iYm > iYMouse[k] && iYm < iYMouse2[k]))
      {
        switch (iView)
        {
          case cmn.cmnStruct.STIFF:
          case cmn.cmnStruct.COLLINS:
	        break;
          case cmn.cmnStruct.PIPER:

            if (stBrine != null)
            {
  	          if (stBrine.stItem != null)
	          {
	            piper = new brinePiperPlotFrame( stBrine );
	            piper.setRow(k);
	          }
	        }
            break;
	    }
      }
    }
 }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawRanges( Graphics g,
   *                     int iView,
   *                     int iStartTrack,
   *                     int iWidthTrack )
   *  <p> This method will draw the correlation Grid.
   * @param g           = Graphics Parameter.
   * @param iView       = The Core Image Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawRanges( Graphics g,
                          int iView,
                          int iStartTrack,
                          int iWidthTrack )
  {
    int iEndTrack = iStartTrack + iWidthTrack;
    int iDraw     = 0;

    for (int i=0; i<iTotal; i++)
    {
      switch (iView)
      {
        case cmn.cmnStruct.STIFF:
          if (iSelected[i] == 0)
            g.setColor(Color.blue);
          else
            g.setColor(Color.cyan);
          break;
        case cmn.cmnStruct.COLLINS:
          if (iSelected[i] == 0)
            g.setColor(Color.green);
          else
            g.setColor(Color.yellow);
          break;
        case cmn.cmnStruct.PIPER:
          if (iSelected[i] == 0)
            g.setColor(Color.red);
          else
            g.setColor(Color.pink);
          break;
      }

      if ((iYMouse[i] > PLOT_TITLES) && (iYMouse2[i] > PLOT_TITLES) &&
          (iYMouse[i] < iHeight)     && (iYMouse2[i] < iHeight))

//      if (iDraw == 1)
      {
        g.fill3DRect( iStartTrack,
                      iYMouse[i],
                      iWidthTrack,
                      (iYMouse2[i] - iYMouse[i]),
                      true );

        g.setColor(Color.darkGray);
        g.drawLine(iStartTrack + 1,               iYMouse[i]  + 1,
                   iStartTrack + iWidthTrack - 1, iYMouse[i]  + 1);
        g.drawLine(iStartTrack + 1,               iYMouse[i]  + 1,
                   iStartTrack + 1,               iYMouse2[i] - 1);
        g.drawLine(iStartTrack + 1,               iYMouse2[i] - 1,
                   iStartTrack + iWidthTrack - 1, iYMouse2[i] - 1);
        g.drawLine(iStartTrack + iWidthTrack - 1, iYMouse[i]  + 1,
                   iStartTrack + iWidthTrack - 1, iYMouse2[i] - 1);
      }
    }
  }

  /** Method drawGrid( Graphics g,
   *                   int iView,
   *                   int iStartTrack,
   *                   int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iView       = The Core Image Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawGrid( Graphics g,
                        int iView,
                        int iStartTrack,
                        int iWidthTrack )
  {
    int    i=0;
    int    iDepth = 0;
    int    iY1    = 0;
    double depth  = 0.0;
    String sTitle = "";
    int    iEndTrack = iStartTrack + iWidthTrack;

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      depth  = depthStart + i * dIncrementDepth / iIncrementY;
      iDepth = (int) (10.0 * depth);
      depth  = (double) iDepth / 10.0;

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      g.setColor(Color.black);
      g.drawLine(iStartTrack, iY1, iStartTrack+iWidthTrack, iY1);
    }

    switch (iView)
    {
      case cmn.cmnStruct.STIFF:
        sTitle = "Stiff Diagram";
        break;
      case cmn.cmnStruct.COLLINS:
        sTitle = "Collins Bar";
        break;
      case cmn.cmnStruct.PIPER:
        sTitle = "Piper Diagram";
        break;
    }

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 11);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
//    g2.drawString(sTitle, LABELSTART+14, -1*(iStartTrack+2*iWidthTrack/3));
    g2.drawString(sTitle, LABELSTART, -1*(iStartTrack+2*iWidthTrack/3));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g,
   *               int iSelected,
   *               int iStartTrack,
   *               int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g - Graphics Parameter.
   * @param iSelected   = The Image Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iView = -1;
    int iColor = iqstrat.iqstratTracksStruct._SRC_BRINE;

    if ((iSelected == iqstrat.iqstratTracksStruct._BRINE_STIFF) ||
        (iSelected == iqstrat.iqstratTracksStruct._BRINE_COLLINS) ||
        (iSelected == iqstrat.iqstratTracksStruct._BRINE_PIPER))
    {
      switch (iSelected)
      {
        case iqstrat.iqstratTracksStruct._BRINE_STIFF:
          iView = cmn.cmnStruct.STIFF;
          break;
        case iqstrat.iqstratTracksStruct._BRINE_COLLINS:
          iView = cmn.cmnStruct.COLLINS;
          break;
        case iqstrat.iqstratTracksStruct._BRINE_PIPER:
          iView = cmn.cmnStruct.PIPER;
          break;
      }

      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iView, iStartTrack, iWidthTrack );
      drawRanges( g, iView, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 06/14/2011
 *  @author  John Victorine
 */
