/*
 * @brinePiperPlot.java Version 1.1 06/16/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;

import brine.brineListStruct;       // Brine Data List Structure
import brine.brineStruct;           // Brine Data Structure
import brine.plot.brinePlotStruct;  // Brine Plot Data Structure
import brine.plot.brineTernaryPlot; // Ternary Plot
import brine.plot.brineDiamondPlot; // Diamond Plot

/** CLASS brinePiperPlot
 *  <p> This Class will create the Collins Bar Diagram.
 *
 *  @version 1.1 06/16/2011
 *  @author  John R. Victorine
 */

public class brinePiperPlot extends Canvas
{
  private int             iRow    = -1;
  private brineListStruct stBrine = null; // Common Brine Data
  private brinePlotListStruct st  = null; // Plot Brine Data List

  private brineTernaryPlot pCations = null;
  private brineTernaryPlot pAnions  = null;
  private brineDiamondPlot pDiamond = null;

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

  private static final int _TOP     =  20;  // Piper Diagram
  private static final int _TOP2    =  40;  // Lease & api-number

  private int iWidth   = 650; // Width of Plot
  private int iHeight  = 650; //700; // Height of Plot with the Titles

  /** CONSTRUCTOR brinePiperPlot()
   *  <p> This is the constructor for this class.
   *  @param stBrine = Brine Data List Structure
   */

  public brinePiperPlot( brineListStruct stBrine )
  {
    this.stBrine = stBrine;
    setBrineData();
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	if (st != null)
	  st.delete();
	st = null;

	stBrine = null;

    if (pCations != null)
      pCations.close();
    pCations = null;

    if (pAnions != null)
      pAnions.close();
    pAnions  = null;

    if (pDiamond != null)
      pDiamond.close();
    pDiamond = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;
    int w = this.getWidth();
    int h = this.getHeight();

    image = new BufferedImage( w, h, BufferedImage.TYPE_INT_RGB );

    Graphics2D g2 = image.createGraphics();
    g2.setColor( Color.white );
    g2.fillRect( 0, 0, w, h );
    this.paint(g2);
    g2.dispose();

    return (image);
  }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setBrineData( brineListStruct stBrine )
   * <p> This method will set the Brine Data List Structure
   * @param stBrine - Brine Data List Structure
   */

  public void setBrineData()
  {
    int i,j;
    int lenAnions  = 0;
    int lenCations = 0;

	st = new brinePlotListStruct();

	if (stBrine != null)
	{
	  lenAnions  = brine.math.brineMath.getRows( _ANIONS );
	  lenCations = brine.math.brineMath.getRows( _CATIONS );

	  st         = new brinePlotListStruct();
	  st.iCount  = stBrine.iCount;
	  st.stItem = new brinePlotStruct[st.iCount];

	  for (i=0; i<stBrine.iCount; i++)
	  {
	    st.stItem[i] = new brinePlotStruct();

        st.stItem[i].mgl_cations      = new double[lenCations];
        st.stItem[i].meql_cations     = new double[lenCations];
        st.stItem[i].per_meql_cations = new double[lenCations];

        st.stItem[i].mgl_anions       = new double[lenAnions];
        st.stItem[i].meql_anions      = new double[lenAnions];
        st.stItem[i].per_meql_anions  = new double[lenAnions];
	  }

	  for (i=0; i<stBrine.iCount; i++)
	  {
		st.stItem[i].depthStart  = stBrine.stItem[i].dTOP;
		st.stItem[i].depthEnd    = stBrine.stItem[i].dBASE;

	    st.stItem[i].mgl_cations =
	      brine.math.brineMath.get_mg_per_liter( _CATIONS, i, stBrine );

	    st.stItem[i].mgl_anions =
	      brine.math.brineMath.get_mg_per_liter( _ANIONS, i, stBrine );

        st.stItem[i].meql_cations =
          brine.math.brineMath.mg_to_meq_per_liter(
  		    _CATIONS, st.stItem[i].mgl_cations );

        st.stItem[i].meql_anions =
          brine.math.brineMath.mg_to_meq_per_liter(
	  	      _ANIONS, st.stItem[i].mgl_anions );

        st.stItem[i].per_meql_cations =
          brine.math.brineMath.percent_meq_per_liter(
		      _CATIONS, st.stItem[i].meql_cations );

        st.stItem[i].per_meql_anions =
          brine.math.brineMath.percent_meq_per_liter(
		      _ANIONS, st.stItem[i].meql_anions );
      }
	}
  }

  /** Method setRow()
   * <p> This method will set the plot data to plot
   * @param iRow = The Data Row
   */

  public void setRow(int iRow) { this.iRow = iRow; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawTitle()
   * <p> This method will draw the Title
   *  @param g = Graphics pointer.
   */

  public void drawTitle(Graphics g)
  {
	int i=0;
	String sTitle    = "";
	String sLabel    = "";
	String sLabel2   = "";
	String sLabel3   = "";
	String sName     = "";
	int    iName     = 0;
	String sAPI      = "";
	String sCounty   = "";
	int    iCounty   = 0;
	String state     = "";
	String sLocation = "";
	int    iLocation = 0;
	String sForm     = "";
	int    iForm     = 0;
	int    length    = 0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 16);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);

    sTitle  = new String( "Piper Diagram" );
    length  = sTitle.length();
    g.drawString( sTitle, iWidth/2 - length*4 , _TOP );

    fsb  = new Font("Monospaced", Font.BOLD, 14);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    if ((stBrine != null) && (iRow > -1))
    {
	  if (stBrine.stItem[iRow].sName.length() > 0)
	  {
		sLabel = new String( stBrine.stItem[iRow].sName );

		if (stBrine.sAPI.length() > 0)
		{
  		  sLabel = new String( sLabel +
  		                       " (" + stBrine.stItem[iRow].sAPI + ")" );
		}
	  }

      if (stBrine.iCount == 1)
      {
        sLabel2 = new String("" + stBrine.stItem[iRow].dTOP);
        if (stBrine.stItem[iRow].dBASE > stBrine.stItem[iRow].dTOP)
          sLabel2 = new String( "" + stBrine.stItem[iRow].dTOP + " - " +
                                     stBrine.stItem[iRow].dBASE);

        if (stBrine.stItem[iRow].sFORM.length() > 0)
        {
          sLabel3 = new String(stBrine.stItem[iRow].sFORM);
  	    }
  	  }
	}
    else if ((stBrine != null) && (iRow == -1))
    {
	  sName     = "";
	  iName     = 1;
	  sCounty   = "";
	  iCounty   = 1;
	  sLocation = "";
	  iLocation = 1;

	  for (i=0; i<stBrine.iCount; i++)
	  {
		if (i==0)
		{
		  sName     = new String( stBrine.stItem[i].sName );
		  sAPI      = new String( stBrine.stItem[i].sAPI );
		  sCounty   = new String( stBrine.stItem[i].sCounty );
		  state     = new String( stBrine.stItem[i].state );
		  sLocation = new String( stBrine.stItem[i].sLocation );
		}

		if (!sName.equals(stBrine.stItem[i].sName))
		{
	      iName = 0;
		}

		if (!sCounty.equals(stBrine.stItem[i].sCounty))
		{
	      iCounty = 0;
		}

		if (!sLocation.equals(stBrine.stItem[i].sLocation))
		{
	      iLocation = 0;
		}
	  }

	  if ((iName == 1) && (sName.length() > 0))
	  {
	    sLabel = new String( sName );

	    if (sAPI.length() > 0)
	    {
	      sLabel = new String( sLabel + " (" + sAPI + ")" );
	    }
	  }
	  else if ((iLocation == 1) && (sLocation.length() > 8))
	  {
	    sLabel = new String( sLocation );

	    if (sCounty.length() > 0)
	    {
	      sLabel = new String( sLabel + ", " + sCounty );
	    }

	    if (state.length() > 0)
	    {
	      sLabel = new String( sLabel + ", " + state );
	    }
	  }
	  else if ((iCounty == 1) && (sCounty.length() > 0))
	  {
	    sLabel = new String( sCounty );

	    if (state.length() > 0)
	    {
	      sLabel = new String( sLabel + ", " + state );
	    }
	  }
	}

    length = sLabel.length();
    g.drawString( sLabel, iWidth/2 - length*4 , _TOP2 );
    g.drawString( sLabel2, 400, _TOP2+20 );
    g.drawString( sLabel3, 400, _TOP2+40 );
  }

  /** Method drawData()
   * <p> This method will draw the Piper Diagram with Data
   *  @param g = Graphics pointer.
   */

  public void drawData(Graphics g)
  {
    Font        fsb  = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.drawString( "Sodium (Na)",    75, 150);
    g.drawString( "Potassium (K)",  75, 170);
    g.drawString( "Magnesium (Mg)", 75, 190);
    g.drawString( "Calcium (Ca)",   75, 210);

    pCations = new brineTernaryPlot(brine.plot.brineTernaryPlot.MIRROR);
    pCations.setTitle("CATIONS");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.HORIZONTAL, "Mg");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.RIGHT,      "Na+K");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.LEFT,       "Ca");
    pCations.draw(g, 25, 300); //.drawGrid(g, 25, 300);

    g.drawString( "Bicarbonate (HCO3)", 450, 150);
    g.drawString( "Carbonate (CO3)",    450, 170);
    g.drawString( "Chloride (Cl)",      450, 190);
    g.drawString( "Sulfate (SO4)",      450, 210);

    pAnions  = new brineTernaryPlot(brine.plot.brineTernaryPlot.NORMAL);
    pAnions.setTitle("ANIONS");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.HORIZONTAL, "SO4");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.RIGHT,      "Cl");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.LEFT,       "HCO3+CO3");
    pAnions.draw(g, 325, 300); //.drawGrid(g, 325, 300);

    pDiamond = new brineDiamondPlot();
    pDiamond.setUpperLeftTitle("SO4+Cl");
    pDiamond.setLowerLeftTitle("Na+K");
    pDiamond.setUpperRightTitle("Ca+Mg");
    pDiamond.setLowerRightTitle("CO3+HCO3");
    pDiamond.draw(g, 175, 40); //drawGrid(g, 175, 40);

    drawPoint(g);
  }

  /** Method draw()
   * <p> This method will draw the stiff diagram
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void draw(Graphics g)
  {
	drawTitle(g);
	drawData(g);
  }

  /** Method draw()
   * <p> This method will draw the stiff diagram
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   *

  public void draw(Graphics g)
  {
	String sLabel = "";
	int    length = 0;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 14);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);

    sLabel = new String( "Piper Diagram" );
    length = sLabel.length();
    g.drawString( sLabel, iWidth/2 - length*4 , _TOP );

    if (stBrine != null)
    {
	  if (stBrine.sName.length() > 0)
	  {
		sLabel = new String( stBrine.sName );

		if (stBrine.sAPI.length() > 0)
		{
  		  sLabel = new String( sLabel +
  		                       " (" + stBrine.sAPI + ")" );
		}

        length = sLabel.length();
        g.drawString( sLabel, iWidth/2 - length*4 , _TOP2 );
	  }
	}

    sLabel = new String("" + stBrine.depthStart);
    if (stBrine.depthEnd > stBrine.depthStart)
      sLabel = new String( "" + stBrine.depthStart + " - " +
                           stBrine.depthEnd);
    g.drawString(sLabel, 400, _TOP2+20);

    if (stBrine.sFormation.length() > 0)
    {
      g.drawString(stBrine.sFormation, 400, _TOP2+40);
	}

    fsb  = new Font("Monospaced", Font.BOLD, 12);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.drawString( "Sodium (Na)",    75, 150);
    g.drawString( "Potassium (K)",  75, 170);
    g.drawString( "Magnesium (Mg)", 75, 190);
    g.drawString( "Calcium (Ca)",   75, 210);

    pCations = new brineTernaryPlot(brine.plot.brineTernaryPlot.MIRROR);
    pCations.setTitle("CATIONS");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.HORIZONTAL, "Mg");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.RIGHT,      "Na+K");
    pCations.setVerticesTitle(brine.plot.brineTernaryPlot.LEFT,       "Ca");
    pCations.draw(g, 25, 300); //.drawGrid(g, 25, 300);

    g.drawString( "Bicarbonate (HCO3)", 450, 150);
    g.drawString( "Carbonate (CO3)",    450, 170);
    g.drawString( "Chloride (Cl)",      450, 190);
    g.drawString( "Sulfate (SO4)",      450, 210);

    pAnions  = new brineTernaryPlot(brine.plot.brineTernaryPlot.NORMAL);
    pAnions.setTitle("ANIONS");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.HORIZONTAL, "SO4");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.RIGHT,      "Cl");
    pAnions.setVerticesTitle(brine.plot.brineTernaryPlot.LEFT,       "HCO3+CO3");
    pAnions.draw(g, 325, 300); //.drawGrid(g, 325, 300);

    pDiamond = new brineDiamondPlot();
    pDiamond.setUpperLeftTitle("SO4+Cl");
    pDiamond.setLowerLeftTitle("Na+K");
    pDiamond.setUpperRightTitle("Ca+Mg");
    pDiamond.setLowerRightTitle("CO3+HCO3");
    pDiamond.draw(g, 175, 40); //drawGrid(g, 175, 40);

    drawPoint(g);
  }
*/
  /** Method drawPoint()
   * <p> This method will draw the point
   *  @param g  = Graphics pointer.
   */

  public void drawPoint(Graphics g)
  {
	double dCa    = 0.0;
	double dMg    = 0.0;
	double dNaK   = 0.0;

	double dCl    = 0.0;
	double dSO4   = 0.0;
	double dClSO4 = 0.0;

	if (iRow > -1)
	{
      dCa  = st.stItem[iRow].per_meql_cations[0]; // Calcium
      dMg  = st.stItem[iRow].per_meql_cations[1]; // Magnesium
      dNaK = st.stItem[iRow].per_meql_cations[2] +
             st.stItem[iRow].per_meql_cations[3];

      dCl    = st.stItem[iRow].per_meql_anions[0]; // Chloride
      dSO4   = st.stItem[iRow].per_meql_anions[3]; // Sulphate
      dClSO4 = dCl + dSO4;

      pCations.drawPoint(g, 25,  300, dCa,  dMg );
      pAnions.drawPoint(g,  325, 300, dCl,  dSO4);
      pDiamond.drawPoint(g, 175,  40, dNaK, dClSO4);
	}
	else
	{
      for (int i=0; i<st.iCount; i++)
	  {
        dCa  = st.stItem[i].per_meql_cations[0]; // Calcium
        dMg  = st.stItem[i].per_meql_cations[1]; // Magnesium
        dNaK = st.stItem[i].per_meql_cations[2] +
               st.stItem[i].per_meql_cations[3];

        dCl    = st.stItem[i].per_meql_anions[0]; // Chloride
        dSO4   = st.stItem[i].per_meql_anions[3]; // Sulphate
        dClSO4 = dCl + dSO4;

        pCations.drawPoint(g, 25,  300, dCa,  dMg );
        pAnions.drawPoint(g,  325, 300, dCl,  dSO4);
        pDiamond.drawPoint(g, 175,  40, dNaK, dClSO4);
	  }
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint the stiff diagram Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
    draw(g);
  }
}

/*
 *  @version 1.1 06/16/2011
 *  @author  John Victorine
 */
