/*
 * @brineCollinsPlot.java Version 1.1 06/10/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;

import brine.plot.brinePlotStruct;     // Common Brine Data Sample

/** CLASS brineCollinsPlot
 *  <p> This Class will create the Collins Bar Diagram.
 *
 *  @version 1.1 06/10/2011
 *  @author  John R. Victorine
 */

public class brineCollinsPlot extends Canvas
{
  private brinePlotStruct st      = null; // Common Brine Data

  public static final Color cCA   = new Color( 160, 160, 160 );
  public static final Color cMG   = new Color(  34, 139,  34 );
  public static final Color cNA   = new Color(   0, 191, 255 );
  public static final Color cK    = new Color(   0,   0, 180 );

  public static final Color CLR_CATIONS[] = { cCA, cMG, cNA, cK };

  public static final Color cCL   = new Color( 173, 255,  47 );
  public static final Color cBR   = new Color( 180,   0,   0 );
  public static final Color cI    = new Color( 180,   0, 180 );
  public static final Color cSO4  = new Color( 102, 102,   0 );
  public static final Color cHCO3 = new Color( 250, 128, 114 );
  public static final Color cCO3  = new Color( 255, 182, 193 );

  public static final Color CLR_ANIONS[] = { cCL, cBR, cI, cSO4, cHCO3, cCO3 };

  public static final int _CATIONS  = brine.math.brineMath._CATIONS;
  public static final int _ANIONS   = brine.math.brineMath._ANIONS;

  public static final int CATIONS[] = brine.math.brineMath.CATIONS;
  public static final int ANIONS[]  = brine.math.brineMath.ANIONS;

  private int iWidth   = 200; //150; // Width of Plot
  private int iHeight  = 260; //160; // Height of Plot with the Titles

  private int iXBegin  =  40; // Left Edge of Plot Area
  private int iYBegin  =  60; // Top Edge of Plot Area

  private int iXWidth  =  80; //40; // Left Edge of Plot Area
  private int iYHeight = 200; //100; // Top Edge of Plot Area

  private int iX[]     = null;
  private int iY[]     = null;

  /** CONSTRUCTOR brineCollinsPlot()
   *  <p> This is the constructor for this class.
   *  @param st  = Brine Plot Data Structure
   */

  public brineCollinsPlot( brinePlotStruct st )
  {
    this.st = st;
    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	if (st != null)
	  st.delete();
	st = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;
    int w = this.getWidth();
    int h = this.getHeight();

    image = new BufferedImage( w, h, BufferedImage.TYPE_INT_RGB );

    Graphics2D g2 = image.createGraphics();
    g2.setColor( Color.white );
    g2.fillRect( 0, 0, w, h );
    this.paint(g2);
    g2.dispose();

    return (image);
  }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setData()
   *  <p> This method will set the Brine Plot Data Structure
   *  @param st = Brine Plot Data Structure
   */

  public void setData( brinePlotStruct st ) { this.st = st; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawPlotData()
   * <p> This method will plot the data
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void drawPlotData(Graphics g, int X, int Y)
  {
	int i=0;
	int iY1   = 0;
	int iY    = 0;
	int idiff = 0;
	double dTotal = 0.0;

	int    iw   = 40;
	double dfac = 2.0;

    if (st != null)
    {
      dTotal = 0.0;
	  for (i=0; i<4; i++)
	  {
	    dTotal = dTotal + st.per_meql_cations[i];
	  }

	  // Ca

	  iY1 = 0;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_cations[0] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_cations[0] );

      g.setColor(cCA);
      g.fillRect(X+iXBegin, Y+iYBegin+iY1, iw, iY);

      // Mg

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_cations[1] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_cations[1] );

      g.setColor(cMG);
      g.fillRect(X+iXBegin, Y+iYBegin+iY1, iw, iY);

      // Na

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_cations[2] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_cations[2] );

      g.setColor(cNA);
      g.fillRect(X+iXBegin, Y+iYBegin+iY1, iw, iY);

      // K

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_cations[3] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_cations[3] );

      g.setColor(cK);
      g.fillRect(X+iXBegin, Y+iYBegin+iY1, iw, iY);

      dTotal = 0.0;
	  for (i=0; i<4; i++)
	  {
	    dTotal = dTotal + st.per_meql_anions[i];
	  }

      // Cl

	  iY1 = 0;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[0] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[0] );

      g.setColor(cCL);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);

      // Br

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[1] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[1] );

      g.setColor(cBR);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);

      // I

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[2] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[2] );

      g.setColor(cI);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);

      // SO4

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[3] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[3] );

      g.setColor(cSO4);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);

      // HCO3

	  iY1 = iY + iY1;
//	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[4] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[4] );

      g.setColor(cHCO3);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);

      // CO3

  	  iY1 = iY + iY1;
//  	  iY  = (int) ( dfac * 100.0 * st.per_meql_anions[5] / dTotal);
	  iY  = (int) ( dfac * st.per_meql_anions[5] );

      g.setColor(cCO3);
      g.fillRect(X+iXBegin+iw, Y+iYBegin+iY1, iw, iY);
    }
  }

  /** Method draw()
   * <p> This method will draw the stiff diagram
   *  @param g = Graphics pointer.
   *  @param X = Starting X Point on plot
   *  @param Y = Starting Y Point on plot
   */

  public void draw(Graphics g, int X, int Y)
  {
	String sLabel = "";
	int    length = 0;
	int    iw     = 40;

    Font        fsb  = new Font("Monospaced", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    g.setColor(Color.black);

    sLabel = new String( "Collins Bar Diagram" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+40-length*4), Y+iYBegin-40 );

    sLabel = new String( "% meq/l" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin+40-length*4), Y+iYBegin-20 );

    sLabel = new String( "100" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin );

    sLabel = new String( "80" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin+40 );

    sLabel = new String( "60" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin+80 );

    sLabel = new String( "40" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin+120 );

    sLabel = new String( "20" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin+160 );

    sLabel = new String( "0" );
    length = sLabel.length();
    g.drawString( sLabel, (X+iXBegin-length*8), Y+iYBegin+200 );

    g.setColor(cCA);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Calcium (Ca)", X+iXBegin+2*iw+20, Y+iYBegin+8 );

    g.setColor(cMG);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+20, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Magnesium (Mg)", X+iXBegin+2*iw+20, Y+iYBegin+20+8 );

    g.setColor(cNA);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+40, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Sodium (Na)", X+iXBegin+2*iw+20, Y+iYBegin+40+8 );

    g.setColor(cK);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+60, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Potassium (K)", X+iXBegin+2*iw+20, Y+iYBegin+60+8 );

    g.setColor(cCL);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+80, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Chloride (Cl)", X+iXBegin+2*iw+20, Y+iYBegin+80+8 );

    g.setColor(cBR);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+100, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Bromide (Br)", X+iXBegin+2*iw+20, Y+iYBegin+100+8 );

    g.setColor(cI);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+120, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Iodide (I)", X+iXBegin+2*iw+20, Y+iYBegin+120+8 );

    g.setColor(cSO4);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+140, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Sulphate (SO4)", X+iXBegin+2*iw+20, Y+iYBegin+140+8 );

    g.setColor(cCO3);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+160, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Carbonate (CO3)", X+iXBegin+2*iw+20, Y+iYBegin+160+8 );

    g.setColor(cHCO3);
    g.fillRect(X+iXBegin+2*iw+5, Y+iYBegin+180, 12, 12);
    g.setColor(Color.black);
    g.drawString( "Bicarbonate (HCO3)", X+iXBegin+2*iw+20, Y+iYBegin+180+8 );

    drawPlotData(g, X, Y);
  }

  /** Method paint( Graphics g )
   * <p> This method will paint the stiff diagram Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
    draw(g, 0, 0);
  }
}

/*
 *  @version 1.1 06/10/2011
 *  @author  John Victorine
 */
