/*
 * @brineMath.java Version 1.1 06/07/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.math;

import brine.brineListStruct;

/** Class brineMath
 *  <p> This Class will perform the basic math for the brine plots.
 *
 *  @version 1.1 06/07/2011
 *  @author  John R. Victorine
 */

public class brineMath
{
  public static final int _CATIONS = 0;
  public static final int _ANIONS  = 1;

  public static final int CATIONS[] =
  {
    brine.brineStandardTools._CA,
    brine.brineStandardTools._MG,
    brine.brineStandardTools._NA,
    brine.brineStandardTools._K,
  };

  public static final int ANIONS[]  =
  {
    brine.brineStandardTools._CL,
    brine.brineStandardTools._BR,
    brine.brineStandardTools._I,
    brine.brineStandardTools._SO4,
    brine.brineStandardTools._HCO3,
    brine.brineStandardTools._CO3,
  };

  /** Method getRows()
   *  <p> This method will return the number of cations or anions
   *  @param  iType = Cations or Anions Indicator
   *  @return iRows = Number of rows;
   */

  public static int getRows( int iType )
  {
    int iRows = 0;

    switch (iType)
    {
	  case _CATIONS:
	    iRows = CATIONS.length;
	    break;

	  case _ANIONS:
	    iRows = ANIONS.length;
	    break;
	}

    return (iRows);
  }

  /** Method mg_to_meq_per_liter()
   *  <p> This method will get the mg/l for either Cations
   *      or Anions from the brine data list structure.
   *  @param  iType = Cations or Anions Indicator
   *  @param  iRow  = Brine Data Row Number
   *  @param  st    = Brine Data List Structure
   *  @return dMG   = Array of mg/l values.
   */

  public static double[] get_mg_per_liter(
	  int iType, int iRow, brineListStruct st )
  {
    double dMG[]  = null;
    int    i      = 0;
    int    len    = 0;

    len = getRows( iType );

    dMG = new double[len];

	// initialize the meq/liter data array

	for (i=0; i<len; i++)
	{
	  dMG[i] = 0.0;
	}

    if (st != null)
    {
	  for (i=0; i<len; i++)
	  {
        switch (iType)
        {
  	      case _CATIONS:
	        dMG[i] = st.getData( CATIONS[i], iRow );
	        break;

	      case _ANIONS:
	        dMG[i] = st.getData( ANIONS[i], iRow );
	        break;
	    }
  	  }
    }

    return (dMG);
  }

  /** Method mg_to_meq_per_liter()
   *  <p> This method will compute meq/l from mg/l for either Cations
   *      or Anions
   *  @param  iType = Cations or Anions Indicator
   *  @param  dMG   = Array of mg/l values.
   *  @return dMEQ  = Array of meq/l values.
   */

  public static double[] mg_to_meq_per_liter(
	  int iType, double dMG[] )
  {
    double dMEQ[]  = null;
    int    i       = 0;
    int    len     = 0;
    int    itmp    = 0;
    double dme_meq = 0.0;

    len  = getRows( iType );

    dMEQ = new double[len];

	// initialize the meq/liter data array

	for (i=0; i<len; i++)
	{
	  dMEQ[i] = 0.0;
	}

    for (i=0; i<len; i++)
    {
      switch (iType)
      {
  	    case _CATIONS:
	      dme_meq = brine.brineStandardTools.getGMperMole(CATIONS[i]) /
	                brine.brineStandardTools.getValence(CATIONS[i]);
	      break;

        case _ANIONS:
          dme_meq = brine.brineStandardTools.getGMperMole(ANIONS[i]) /
                    brine.brineStandardTools.getValence(ANIONS[i]);
	      break;
	  }

      if (dMG[i] > 0.0)
      {
	    dMEQ[i] = dMG[i] / dme_meq;
	    itmp    = (int) (100.0 *dMEQ[i]);
	    dMEQ[i] = itmp / 100.0;
	  }
    }

    return (dMEQ);
  }

  /** Method percent_meq_per_liter()
   *  <p> This method will compute % meq/l from mg/l for either Cations
   *      or Anions
   *  @param  iType = Cations or Anions Indicator
   *  @return dMEQ  = Array of meq/l values.
   *  @return dPER  = Array of % meq/l values.
   */

  public static double[] percent_meq_per_liter(
	  int iType, double dMEQ[] )
  {
    double dPER[]  = null;
    int    itmp    = 0;
    int    i       = 0;
    int    len     = 0;
    double dTotal  = 0.0;

    len  = getRows( iType );

    dPER = new double[len];

	// initialize the Percent data array

	for (i=0; i<len; i++)
	{
	  dPER[i] = 0.0;
	}

    if (dMEQ != null)
    {
	  // Compute the total meq/liter

	  for (i=0; i<len; i++)
	  {
	    dTotal = dTotal + dMEQ[i];
	  }

      // Compute the Percent meq/liter

	  for (i=0; i<len; i++)
	  {
	    dPER[i] = 100.0 * Math.abs(dMEQ[i] / dTotal);
	    itmp    = (int) (100.0 *dPER[i]);
	    dPER[i] = itmp / 100.0;
	  }
	}

	return (dPER);
  }
}

/*
 *  @version 1.1 06/07/2011
 *  @author  John Victorine
 */
