/*
 * @brineCSVFile.java Version 1.1 06/15/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.io;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import util.utilFileIO;
import cmn.cmnString;
import brine.brineListStruct;
import brine.brineStruct;

/** Class brineCSVFile
 *  <p> This Class is designed to open an ascii text brine data CSV file and read
 *      the contents and parse the data into the brine Structure.
 *
 *  @version 1.1 06/15/2011
 *  @author  John R. Victorine
 */

public class brineCSVFile
{
  private utilFileIO      pFILE  = null;

  private String          sLine1 = "";
  private String          sLine2 = "";
  private brineListStruct stList = null;

  private int    iRows    = 0;
  private int    iColumns = 0;

  private int    iError   = util.utilFileIO.NOERROR;
  private String sError   = new String("");

  /** Construct brineCSVFile()
   *  <p> This is the Constructor for this class.
   */

  public brineCSVFile() { }

  /* ====================================================================== *
   * ---------------------------- FILE METHODS ---------------------------- *
   * ====================================================================== */

  /** Method Open()
    * <p> This is the Constructor for this class.
    * @param iFileID    = To access the file as URL or FILE.
    * @param sDirectory = File URL or Directory Path
    * @param sFilename  = File Name
    */

  public void Open(int iFileID, String sDirectory, String sFilename)
  {
    pFILE  = new utilFileIO();
    pFILE.Open(iFileID, util.utilFileIO.READ, sDirectory, sFilename);
    iError = pFILE.getErrorID();
    sError = pFILE.getError();

    iRows             = 0;
    iColumns          = 0;
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close() { pFILE.Close(); }

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    sLine1   = null;
    sLine2   = null;
    sError   = null;

    iRows    = 0;

    if (pFILE != null)
      pFILE.delete();
    pFILE = null;
  }

  /** Method get2Rows()
   * <p> This method will read and parse the Rock File and get the first 2 lines.
   */

  public void get2Rows()
  {
    int             iCount  = 0;
    String          sTemp   = "";
    StringTokenizer stToken = null;

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        sTemp = pFILE.getString();
        if (iCount == 0)
        {
          sLine1   = new String(sTemp);
          stToken  = new StringTokenizer(sTemp, ",");
          iColumns = stToken.countTokens();
        }

        if (iCount == 1)
          sLine2 = new String(sTemp);

        iCount++;
        iRows++;
      }
    }
  }

  /** Method getData()
   * <p> This method will read and parse the Rock File into its data structures.
   * @param iStart  = the starting row number
   * @param iColumn = Array of file column mapping to the brine Data structure
   * @param iUnit   = Array of Units for analytes.
   */

  public void getData(int iStart, int iColumn[], int iUnit[])
  {
    int    i=0;
    int    iCount = 1;
    String sTemp  = "";
    brineStruct st = null;
    String sKEY = cmn.cmnString.UniqueName();
    String sKID = "0";
    int    iValue = 0;
    int    iBegin = 0;

    stList = new brineListStruct();

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (iCount >= iStart)
          {
            sTemp  = pFILE.getString();
            sKID   = new String(sKEY + iCount);
            st     = parseRow(sKID, sTemp, iColumn, iUnit);
//brine.brineUtility.print(st);

            if (st.dTOP != brine.brineStruct.dNULL)
              stList = brine.brineUtility.add(st, stList);

            sTemp = "";
            st.delete();
            st = null;
          }

          iCount++;
        }
      }
    }

    stList = brine.brineUtility.computeDepthRange(stList);
//    stList = brine.brineUtility.bubbleSort(stList, 0);
    stList = brine.brineUtility.addBaseDepth( stList );
  }

  /** Method parseRow()
   * <p> This method will parse a Rock File row into the brine structure.
   * @param  sKEY    = Unique key for formation name
   * @param  sData   = Data Row String
   * @param  iColumn = The mapped columns
   * @return st      = the brine Data Structure
   */

  private brineStruct parseRow(String sKEY, String sData,
                               int iColumn[], int iUnit[])
  {
    brineStruct st = new brineStruct();
    int    iCount = 0;
    int    iFound = 0;
    int    iNext  = 0;
    int    iChng  = 0;
    String sTemp  = "";
    StringTokenizer stToken = null;
    String sTokens[] = null;
    String sDelimiter = new String("[,]+");

    st.sKEY  = new String(sKEY);

    sData    = addspace(sData);
    sTemp    = new String(sData.replace('\t', ' '));
//    sTokens  = las3.las3Parse.splitDataLine( sTemp, "," );
    sTokens = sTemp.split( sDelimiter );

    iColumns = 0;
    if (sTokens != null)
      iColumns = sTokens.length;

    for (int i=0; i<iColumns; i++)
    {
      sTemp = sTokens[i];
      sTemp = new String(sTemp.replace('<', ' '));
      sTemp = new String(sTemp.replace('>', ' '));
      sTemp = new String(sTemp.trim());

      iNext  = i+1;
      iFound = -1;
      iChng  = 0;
      for (int j=0; j<brine.brineStandardTools.TOTAL_A; j++)
      {
        if (iNext == iColumn[j])
        {
		  if (iUnit[j] == 1) { iChng = 1; }
          iFound = j;
	    }
      }

      if (iFound > -1)
      {
        st = mapData(
	  	       sTemp, brine.brineStandardTools.CURVES[iFound][1], iChng, st);
	  }
    }

    return (st);
  }

  /** Method removeLT()
   * <p> This method will remove the '<' from the String
   * @param  str  = The string to be converted
   * @return sNew = The text converted
   */

  public static String removeLT( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if ((c[i] != '<') && (c[i] != ' '))
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method mapData()
   * <p> This method will parse a Rock File row into the brine structure.
   * @param  sData       = Data Row String
   * @param  sIdentifier = the column idenfier
   * @param  iChange     = Indicates the unit is not mg/l but ug/l
   * @param  st          = the brine Data Structure
   * @return st          = the brine Data Structure
   */

  private brineStruct mapData( String sData,
                               String sIdentifier,
                               int    iChange,
                               brineStruct st)
  {
	double dFactor = 1.0;
    String sID     = "";

	if (iChange == 1)
	  dFactor = 0.001;

    sID = new String(sIdentifier.toUpperCase());

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.KEY))
    {
	  stList._KEY++;
	  st.sKID = new String(sData);
	  st.sKEY = new String(sData);
	}

//System.out.println( sIdentifier + " " + sData );

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.STATE))  // "State"
    {
      st.state = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.COUNTY)) // "County"
    {
      st.sCounty = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.FIELD))  // "Field"
    {
      st.sField = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.LOC))  // "Location"
    {
      st.sLocation = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.API))    // "Well API-Number"
    {
      st.sAPI = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.NAMEA))   // "Well Name"
    {
      st.sName = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.LAT))    // "Latitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLatitude = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.LONG))   // "Longitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLongitude = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.KB))   // "Kelly Bushing"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dKB = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.GL))   // "Ground Level"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dGL = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.DF))   // "Derrick Floor"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dDF = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.TGT))   // "TGT"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dTGT = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.SRC))    // "Sample Source"
    {
      st.source = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.DATE))   // "Sample Date"
    {
      st.sRecovery = new String(sData);
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.TOP))     // "Depth Top" unit="ft"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TOP++;
        st.dTOP     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.BASE))   // "Depth Base" unit="ft"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BASE++;
        st.dBASE     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.FORM))   // "Formation"
    {
      st.sFORM = new String(sData);
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.AGE))    // "Formation Age"
    {
      st.sAGE = new String(sData);
    }

    // CATIONS

    // .. Alkali Metal

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._LI))    // "Lithium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LI++;
        st.dLI = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NA))    // "Sodium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NA++;
        st.dNA = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._K))     // "Potassium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K++;
        st.dK = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._RB))    // "Rubidium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RB++;
        st.dRB = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CS))    // "Cesium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CS++;
        st.dCS = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NA_K))  // "Sodium/Potassium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NA_K++;
        st.dNA_K = dFactor * cmn.cmnString.stringToDouble(sData);

        if (st.dNA_K > 0.0)
        {
          stList._NA++;
          st.dNA = st.dNA_K/2.0;
          stList._K++;
          st.dK  = st.dNA_K/2.0;
		}
      }
    }

    // ..  Alkali Earth Metal

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BE))    // "BERYLLIUM"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BE++;
        st.dBE = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MG))    // "Magnesium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MG++;
        st.dMG = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CA))    // "Calcium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CA++;
        st.dCA = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SR))    // "Strontium"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SR++;
        st.dSR = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BA))    // "BARIUM"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BA++;
        st.dBA = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIB

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CR)) ||   // "CHROMIUM"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CRII)))   // "CHROMIUM II"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR_II++;
        st.dCR_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CRIII))   // "CHROMIUM III"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR_III++;
        st.dCR_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIIB

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MN)) ||   // "MANGANESE"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MNII)))   // "MANGANESE II"
    {
//      sData = removeLT(sData);
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MN_II++;
        st.dMN_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MNIII))    // "MANGANESE III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MN_III++;
        st.dMN_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIIIB

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._FE)) ||   // "Iron"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._FEII)))   // "Iron II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._FE_II++;
        st.dFE_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._FEIII))    // "Iron III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._FE_III++;
        st.dFE_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CO)) ||  // "COBALT"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._COII)))  // "COBALT II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO_II++;
        st.dCO_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._COIII))    // "COBALT III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO_III++;
        st.dCO_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NI)) ||   // "NICKEL"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NIII)))   // "NICKEL II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NI_II++;
        st.dNI_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NIIII))    // "NICKEL III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NI_III++;
        st.dNI_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IB

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CU))    // "COPPER"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CU_I++;
        st.dCU_I = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CUII))    // "COPPER II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CU_II++;
        st.dCU_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._AG))    // "SILVER"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AG++;
        st.dAG = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._AU))    // "GOLD"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AU++;
        st.dAU = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._AUIII))    // "GOLD III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AU_III++;
        st.dAU_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IIB

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._ZN))    // "ZINC"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._ZN++;
        st.dZN = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CD))    // "CADMIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CD++;
        st.dCD = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HG))    // "MERCURY"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HG++;
        st.dHG = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HGII))    // "MERCURY II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HG_II++;
        st.dHG_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IIIA

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._AL))    // "ALUMINUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AL++;
        st.dAL = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IVA

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SN)) ||   // "Tin"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SNII)))    // "Tin II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SN_II++;
        st.dSN_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SNIV))    // "Tin IV"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SN_IV++;
        st.dSN_IV = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._PB)) ||   // "LEAD"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._PBII)))    // "LEAD II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PB_II++;
        st.dPB_II = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._PBIV))    // "LEAD IV"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PB_IV++;
        st.dPB_IV = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    //.. VA

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SB)) ||   // "ANTIMONY"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SBIII)))  // "ANTIMONY III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SB_III++;
        st.dSB_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SBV))    // "ANTIMONY V"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SB_V++;
        st.dSB_V = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BI)) ||   // "Bismuth"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BIIII)))  // "Bismuth III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BI_III++;
        st.dBI_III = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BIV))    // "Bismuth V"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BI_V++;
        st.dBI_V = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // .. Other

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NH4))   // "AMMONIA"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NH4++;
        st.dNH4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    // ANIONS

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._F))     // "FLUORIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._F++;
        st.dF = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CL))    // "Chloride"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CL++;
        st.dCL = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BR))    // "Bromine"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BR++;
        st.dBR = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._I))     // "Iodine"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._I++;
        st.dI = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._OH))    // "HYDROXIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OH++;
        st.dOH = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BO3))   // "Borate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BO3++;
        st.dBO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CO3))   // "Carbonate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO3++;
        st.dCO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HCO3))  // "Bicarbonate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HCO3++;
        st.dHCO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CLO))   // "HYPOCHLORITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO++;
        st.dCLO = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CLO2))  // "CHLORITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO2++;
        st.dCLO2 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CLO3))  // "CHLORATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO3++;
        st.dCLO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CLO4))  // "PERCHLORATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO4++;
        st.dCLO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CN))    // "CYANIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CN++;
        st.dCN = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NCO))    // "Cyanate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NCO++;
        st.dNCO = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._OCN))    // "Isocyanate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OCN++;
        st.dOCN = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SCN))    // "Thiocyanite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SCN++;
        st.dSCN = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._N))   // "Nitride"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._N++;
        st.dN = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._N3))   // "Azide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._N3++;
        st.dN3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NO2))   // "NITRITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NO2++;
        st.dNO2 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._NO3))   // "NITRATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NO3++;
        st.dNO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CRO4))   // "Chromium (II)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CRO4++;
        st.dCRO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._CR2O7))   // "Chromium (III)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR2O7++;
        st.dCR2O7 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MNO4))   // "Permanganate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MNO4++;
        st.dMNO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._P))     // "Phosphide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._P++;
        st.dP = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._PO3))     // "PHOSPHITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PO3++;
        st.dPO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._PO4))     // "PHOSPHATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PO4++;
        st.dPO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HPO4))     // "Monohydrogen Phosphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HPO4++;
        st.dHPO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._H2PO4))     // "Dihydrogen Phosphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._H2PO4++;
        st.dH2PO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._AS))    // "ARSENIC"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AS++;
        st.dAS = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SE))    // "SELENIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SE++;
        st.dSE = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._S))   // "Sulfide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._S++;
        st.dS = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SO3))   // "Sulphite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SO3++;
        st.dSO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HSO3))   // "Hydrogen Sulphite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HSO3++;
        st.dHSO3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._S2O3))   // "Thiosulphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._S2O3++;
        st.dS2O3 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(brine.io.ReadBrineDataXMLFile._H2S)) ||  // "Hydrogen Sulfide"
        (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HS)))     // "Hydrogen Sulfide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HS++;
        st.dHS = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._SO4))   // "Sulfate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SO4++;
        st.dSO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._HSO4))  // "BISULFATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HSO4++;
        st.dHSO4 = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._B))     // "BORON"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._B++;
//        st.dB = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._BRO3))  // "BROMATE"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._BRO3++;
//        st.dBRO3 = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._MO))    // "MOLYBDENUM"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._MO++;
//        st.dMO = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._TL))    // "THALLIUM"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._TL++;
//        st.dTL = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._TH))    // "THORIUM"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._TH++;
//        st.dTH = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile._V))     // "VANADIUM"
//    {
//        sData  = removeLT(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._V++;
//        st.dV = dFactor * cmn.cmnString.stringToDouble(sData);
//      }
//    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.SOLID))  // "Total Solids"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SOLID++;
        st.dSOLID = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.TDS))    // "Total Solids Computed"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TDS++;
        st.dTDS = dFactor * cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.SPGR))   // "Specific Gravity"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SPGR++;
        st.dSPGR = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.SIGMA))  // "Specific Conductivity"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SIGMA++;
        st.dSIGMA = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.PH))     // "PH"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PH++;
        st.dPH = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.DEG))    // "Temperature (F)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._DEG++;
        st.dDEG = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.OHM))    // "Resistivity (Rw)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHM++;
        st.dOHM = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.OHM75))  // "Resistivity at 75 deg"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHM75++;
        st.dOHM75 = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(brine.io.ReadBrineDataXMLFile.OHME))   // "Estimated Rw"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHME++;
        st.dOHME = cmn.cmnString.stringToDouble(sData);
      }
    }

    return (st);
  }

  /** Method addspace()
   * <p> This method will add a space character after each comma in the String
   * @param  str  = The string to be converted
   * @param  ch   = the character to be removed from string
   * @return sNew = The text converted
   */

  public static String addspace( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if (c[i] == ',')
          sNew = new String(sNew + c[i] + ' ');
        else
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /* ====================================================================== *
   * ----------------------------- GET METHODS ---------------------------- *
   * ====================================================================== */

  /** Method getLine1()
   * <p> This method will return the 1st Line of the delimited file
   * @return sLine1 = 1st line of delimited file
   */

  public String getLine1() { return (sLine1); }

  /** Method getLine2()
   * <p> This method will return the 2nd Line of the delimited file
   * @return sLine2 = 2nd line of delimited file
   */

  public String getLine2() { return (sLine2); }

  /** Method getRock()
   * <p> This method will return brine data list
   * @return stList = brine Data list structure
   */

  public brineListStruct getRock() { return (stList); }

  /* ====================================================================== *
   * -------------------------- GET ERROR METHODS ------------------------- *
   * ====================================================================== */

  /** METHOD getErrorID()
   *  <p> This method will return the error number.
   *  @return iError = Error Indicator
   */

  public int getErrorID() { return (iError); }

  /** METHOD getError()
   *  <p> This method will return the error string.
   *  @return sError = Error String
   */

  public String getError() { return (sError); }
}

/*
 *  @version 1.1 06/15/2011
 *  @author  John Victorine
 */
