/*
 * @ReadBrineDataXMLFile.java Version 1.0 06/03/2011
 *
 * Copyright (c) 2011 John R. Victorine
 * All Rights Reserved.
 */

package brine.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import brine.brineListStruct;
import brine.brineStruct;
import brine.brineColumnsStruct;

/** Class ReadBrineDataXMLFile()
 * <p> This Class will parse the Measured Brine XML Data Stream
 *     directly to the Measured Brine List Data Structure.
 *
 *  @version 1.1 06/03/2011
 *  @author  John R. Victorine
 */

public class ReadBrineDataXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int              iType  = FILE;
  private int              iRows  = 0;
  private int              iCount = -1;

  private int              iComments = 0;

  public static final String ROOTNAME = "fluid_data"; // ROOT NAME
  public static final String RECORDS  = "records";   // Attribute

  // Brine Column Data

  public static final String COLUMNS  = "columns";   // ELEMENT
  public static final String COLUMN   = "column";    // ELEMENT

  public static final String MNEMONIC = "mnemonic";  // Attribute
  public static final String NAME     = "name";      // Attribute
  public static final String UNIT     = "unit";      // Attribute
  public static final String TYPE     = "type";      // Attribute

  // Brine Data

  public static final String DATA   = "data";   // ELEMENT
  public static final String ROW    = "row";    // ELEMENT

  public static final String KEY    = "KEY";    // "Fluid Primary Key"
  public static final String STATE  = "STATE";  // "State"
  public static final String COUNTY = "COUNTY"; // "County"
  public static final String FIELD  = "FIELD";  // "Field"
  public static final String LOC    = "LOC";    // "Location"
  public static final String NAMEA  = "NAME";   //  Name of Well
  public static final String API    = "API";    // "Well API-Number"
  public static final String LAT    = "LAT";    // "Latitude"
  public static final String LONG   = "LONG";   // "Longitude"

  public static final String KB     ="KB";      // "Kelly Bushing"
  public static final String GL     ="GL";      // "Ground Level"
  public static final String DF     ="DF";      // "Derrick Floor"
  public static final String TGT    ="TGT";     // "TGT"

  public static final String FORM   = "FORM";   // "Formation"
  public static final String AGE    = "AGE";    // "Formation Age"

  public static final String SRC    = "SRC";    // "Sample Source"
  public static final String DATE   = "DATE";   // "Sample Date"

  public static final String TOP    = "TOP";    // "Depth Top" unit="ft"
  public static final String BASE   = "BASE";   // "Depth Base" unit="ft"

  // CATIONS

  // .. Alkali Metal
  public static final String _LI    = "Li";     // "Lithium"
  public static final String _NA    = "Na";     // "Sodium"
  public static final String _K     = "K";      // "Potassium"
  public static final String _RB    = "RB";     // "Rubidium"
  public static final String _CS    = "CS";     // "Cesium"
  public static final String _NA_K  = "Na_K";   // "Sodium/Potassium"

  // ..  Alkali Earth Metal
  public static final String _BE    = "Be";     // "BERYLLIUM"
  public static final String _MG    = "Mg";     // "Magnesium"
  public static final String _CA    = "Ca";     // "Calcium"
  public static final String _SR    = "Sr";     // "Strontium"
  public static final String _BA    = "Ba";     // "Barium"

  // ..  VIB
  public static final String _CR    = "Cr";     // "CHROMIUM"
  public static final String _CRII  = "CrII";   // "CHROMIUM II"
  public static final String _CRIII = "CrIII";  // "CHROMIUM III"

  // ..  VIIB
  public static final String _MN    = "Mn";     // "MANGANESE"
  public static final String _MNII  = "MnII";   // "MANGANESE II"
  public static final String _MNIII = "MnIII";  // "MANGANESE III"

  // ..  VIIIB
  public static final String _FE    = "Fe";     // "Iron"
  public static final String _FEII  = "FeII";   // "Iron II"
  public static final String _FEIII = "FeIII";  // "Iron II"
  public static final String _CO    = "Co";     // "COBALT"
  public static final String _COII  = "CoII";   // "COBALT II"
  public static final String _COIII = "CoIII";  // "COBALT III"
  public static final String _NI    = "Ni";     // "NICKEL"
  public static final String _NIII  = "NiII";   // "NICKEL II"
  public static final String _NIIII = "NiIII";  // "NICKEL III"

  // ..  IB
  public static final String _CU    = "Cu";     // "COPPER"
  public static final String _CUII  = "CuII";   // "COPPER II"
  public static final String _AG    = "Ag";     // "SILVER"
  public static final String _AU    = "Au";     // "GOLD"
  public static final String _AUIII = "AuIII";  // "GOLD III"

    // ..  IIB
  public static final String _ZN    = "Zn";     // "ZINC"
  public static final String _CD    = "Cd";     // "CADMIUM"
  public static final String _HG    = "Hg";     // "MERCURY"
  public static final String _HGII  = "HgII";   // "MERCURY II"

  // ..  IIIA
  public static final String _AL    = "Al";     // "ALUMINUM"

  // ..  IVA
  public static final String _SN    = "Sn";     // "Tin"
  public static final String _SNII  = "SnII";   // "Tin II"
  public static final String _SNIV  = "SnIV";   // "Tin IV"
  public static final String _PB    = "Pb";     // "LEAD"
  public static final String _PBII  = "PbII";   // "LEAD II"
  public static final String _PBIV  = "PbIV";   // "LEAD IV"

  //.. VA
  public static final String _SB    = "Sb";     // "ANTIMONY"
  public static final String _SBIII = "SbIII";  // "ANTIMONY III"
  public static final String _SBV   = "SbV";    // "ANTIMONY V"
  public static final String _BI    = "Bi";     // "Bismuth"
  public static final String _BIIII = "BiIII";  // "Bismuth III"
  public static final String _BIV   = "BiV";    // "Bismuth V"

  // .. Other
  public static final String _NH4   = "NH4";    // "AMMONIA"

  // ANIONS

  public static final String _F     = "F";      // "FLUORIDE"
  public static final String _CL    = "Cl";     // "Chloride"
  public static final String _BR    = "Br";     // "Bromine"
  public static final String _I     = "I";      // "Iodine"

  public static final String _OH    = "OH";     // "HYDROXIDE"

  public static final String _BO3   = "BO3";    // "Borate"
  public static final String _CO3   = "CO3";    // "Carbonate"
  public static final String _HCO3  = "HCO3";   // "Bicarbonate"

  public static final String _CLO   = "ClO";    // "HYPOCHLORITE"
  public static final String _CLO2  = "ClO2";   // "CHLORITE"
  public static final String _CLO3  = "ClO3";   // "CHLORATE"
  public static final String _CLO4  = "ClO4";   // "PERCHLORATE"

  public static final String _CN    = "CN";     // "CYANIDE"
  public static final String _NCO   = "NCO";    // "Cyanate"
  public static final String _OCN   = "OCN";    // "Isocyanate"
  public static final String _SCN   = "SCN";    // "Thiocyanite"

  public static final String _N     = "N";      // "Nitride"
  public static final String _N3    = "N3";     // "Azide"
  public static final String _NO2   = "NO2";    // "NITRITE"
  public static final String _NO3   = "NO3";    // "NITRATE"

  public static final String _CRO4  = "CrO4";  // Chromium (II)
  public static final String _CR2O7 = "Cr2O7"; // Chromium (III)

  public static final String _MNO4  = "MnO4"; // Permanganate

  public static final String _P     = "P";    // "Phosphide"
  public static final String _PO3   = "PO3";  // "PHOSPHITE"
  public static final String _PO4   = "PO4";  // "PHOSPHATE"
  public static final String _HPO4  = "HPO4"; // "Monohydrogen Phosphate"
  public static final String _H2PO4 = "H2PO4";// "Dihydrogen Phosphate"

  public static final String _AS    = "As";   // "ARSENIC"
  public static final String _SE    = "Se";   // "SELENIUM"

  public static final String _S     = "S";    // Sulfide
  public static final String _HS    = "HS";   // Hydrogen Sulfide
  public static final String _SO3   = "SO3";  // Sulphite
  public static final String _HSO3  = "HSO3"; // Hydrogen Sulphite
  public static final String _S2O3  = "S2O3"; // Thiosulphate
  public static final String _H2S   = "H2S";  // "Hydrogen Sulfide"
  public static final String _SO4   = "SO4";  // "Sulfate"
  public static final String _HSO4  = "HSO4"; // "BISULFATE"

  public static final String _B     = "B";      // "BORON"
  public static final String _BRO3  = "BrO3";   // "BROMATE"
  public static final String _MO    = "Mo";     // "MOLYBDENUM"
  public static final String _TL    = "Tl";     // "THALLIUM"
  public static final String _TH    = "Th";     // "THORIUM"
  public static final String _V     = "V";      // "VANADIUM"

  public static final String SOLID  = "SOLID";  // "Total Solids"
  public static final String TDS    = "TDS";    // "Total Solids Computed"

  public static final String SPGR   = "SPGR";   // "Specific Gravity"
  public static final String SIGMA  = "SIGMA";  // "Specific Conductivity"
  public static final String PH     = "PH";     // "PH"
  public static final String DEG    = "DEG";    // "Temperature (F)"
  public static final String OHM    = "OHM";    // "Resistivity (Rw)"
  public static final String OHM75  = "OHM75";  // "Resistivity at 75 deg"
  public static final String OHME   = "OHME";   // "Estimated Rw"

  private static final int _NONE    = 0;
  private static final int _COLUMNS = 1;
  private static final int _DATA    = 2;
  private static final int _ROW     = 3;

  private int iData = _NONE;

  private brineColumnsStruct  stColumn = new brineColumnsStruct();
  private brineListStruct     stList   = new brineListStruct();

  private int    iError = 0;
  private String sError = "";

  /** Method ReadBrineDataXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadBrineDataXMLFile() { iType = FILE; }

  /** Method ReadBrineDataXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadBrineDataXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public brineListStruct Process(String filename)
  {
 	double dTotal = 0.0;
 	double data   = 0.0;
 	int    iTotal = 0;

    Read(filename);

    stList = brine.brineUtility.computeDepthRange( stList );
    stList = brine.brineUtility.bubbleSort( stList, 1 );
    stList = brine.brineUtility.addBaseDepth( stList );

    for (int i=0; i<stList.iCount; i++)
    {
	  dTotal = 0.0;
      for (int j=brine.brineStandardTools._LI; j<brine.brineStandardTools._SOLID; j++)
      {
	    if (stList.checkData(j))
	    {
	      data = stList.getData(j, i);
	      if (data > 0.0)
	        dTotal = dTotal + data;
	    }
	  }

      iTotal = (int) (dTotal * 10.0);
      dTotal = (double) iTotal / 10.0;

  	  stList.setData(brine.brineStandardTools._SOLID, i, dTotal);
	  stList.setData(brine.brineStandardTools._TDS,   i, dTotal);
	}

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadBrineDataXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadBrineDataXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(COLUMNS))
    {
      iRows  = 0;
      iCount = -1;
      iData  = _COLUMNS;
    }

    if (eName.equals(COLUMN))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stColumns[iCount] = new brineColumnsStruct();
      }
    }

    if (eName.equals(DATA))
    {
      iRows  = 0;
      iCount = -1;
      iData  = _DATA;
    }

    if (eName.equals(ROW))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new brineStruct();
      }
      iData  = _ROW;
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _COLUMNS:
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stColumns = new brineColumnsStruct[iRows];
              stList.iColumns  = iRows;
              break;

            case _DATA:
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stItem = new brineStruct[iRows];
              stList.iCount = iRows;
              break;
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _COLUMNS:
              if (iCount < iRows)
              {
                stList.stColumns[iCount] = ParseColumns( aName,
                                                         sOut.trim(),
                                                         stList.stColumns[iCount] );
              }
              break;

            case _ROW:
              if (iCount < iRows)
              {
//                stList = ParseData( aName, sOut.trim(), stList );
                stList.stItem[iCount] = ParseData( aName,
                                                   sOut.trim(),
                                                   stList.stItem[iCount] );
              }
              break;
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    String s = new String(buf, offset, len);
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseColumns()
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Measured Brine Data String
   * @param  st          = The Measured Brine Data Data Structure
   * @return st          = The Measured Brine Data Data Structure
   */

  private brineColumnsStruct ParseColumns( String             sIdentifier,
                                           String             sData,
                                           brineColumnsStruct st )
  {
    if (sIdentifier.equals(MNEMONIC))
    {
      st.sMnemonic = new String(sData);
    }

    if (sIdentifier.equals(NAME))
    {
      st.sName = new String(sData);
    }

    if (sIdentifier.equals(UNIT))
    {
      st.sUnit = new String(sData);
    }

    if (sIdentifier.equals(TYPE))
    {
      st.sKEY = new String(sData);
    }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Measured Brine Data String
   * @param  st          = Brine Data List Structure
   * @return st          = Brine Data List Structure
   */

  private brineListStruct ParseData( String          sIdentifier,
                                     String          sData,
                                     brineListStruct st )
  {
    if (sIdentifier.equals(STATE))  // "State"
    {
      st.state = new String(sData);
    }
    if (sIdentifier.equals(COUNTY)) // "County"
    {
      st.sCounty = new String(sData);
    }
    if (sIdentifier.equals(FIELD))  // "Field"
    {
      st.sField = new String(sData);
    }
    if (sIdentifier.equals(API))    // "Well API-Number"
    {
      st.sAPI = new String(sData);
    }
    if (sIdentifier.equals(NAMEA))  // "Well Name
    {
      st.sName = new String(sData);
    }
    if (sIdentifier.equals(LAT))    // "Latitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLatitude = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(LONG))   // "Longitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLongitude = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(KB))   // "Kelly Bushing"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dKB = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(GL))   // "Ground Level"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dGL = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(DF))   // "Derrick Floor"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dDF = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(TGT))   // "TGT"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dTGT = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(SRC))    // "Sample Source"
    {
      st.source = new String(sData);
    }
    if (sIdentifier.equals(DATE))   // "Sample Date"
    {
      st.sAnalyzed = new String(sData);
    }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Measured Brine Data String
   * @param  st          = Brine Data Structure
   * @return st          = Brine Data Structure
   */

  private brineStruct ParseData( String      sIdentifier,
                                 String      sData,
                                 brineStruct st )
  {
    if (sIdentifier.equals(KEY))
    {
	  stList._KEY++;
	  st.sKID = new String(sData);
	  st.sKEY = new String(sData);
	}

    if (sIdentifier.equals(STATE))  // "State"
    {
      st.state = new String(sData);
    }
    if (sIdentifier.equals(COUNTY)) // "County"
    {
      st.sCounty = new String(sData);
    }
    if (sIdentifier.equals(FIELD))  // "Field"
    {
      st.sField = new String(sData);
    }
    if (sIdentifier.equals(LOC))  // "Location"
    {
      st.sLocation = new String(sData);
    }
    if (sIdentifier.equals(API))    // "Well API-Number"
    {
      st.sAPI = new String(sData);
    }
    if (sIdentifier.equals(NAMEA))   // "Well Name"
    {
      st.sName = new String(sData);
    }
    if (sIdentifier.equals(LAT))    // "Latitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLatitude = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(LONG))   // "Longitude"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dLongitude = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(KB))   // "Kelly Bushing"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dKB = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(GL))   // "Ground Level"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dGL = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(DF))   // "Derrick Floor"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dDF = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(TGT))   // "TGT"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dTGT = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(SRC))    // "Sample Source"
    {
      st.source = new String(sData);
    }
    if (sIdentifier.equals(DATE))   // "Sample Date"
    {
      st.sRecovery = new String(sData);
    }

    if (sIdentifier.equals(TOP))    // "Depth Top" unit="ft"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TOP++;
        st.dTOP     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(BASE))   // "Depth Base" unit="ft"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BASE++;
        st.dBASE     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(FORM))   // "Formation"
    {
      st.sFORM = new String(sData);
    }
    if (sIdentifier.equals(AGE))    // "Formation Age"
    {
      st.sAGE = new String(sData);
    }

    // CATIONS
    // .. Alkali Metal

    if (sIdentifier.equals(_LI))    // "LITHIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LI++;
        st.dLI = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NA))    // "Sodium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NA++;
        st.dNA = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_K))     // "Potassium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K++;
        st.dK = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_RB))     // "Rubidium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RB++;
        st.dK = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CS))     // "Cesium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CS++;
        st.dK = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NA_K))  // "Sodium/Potassium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NA_K++;
        st.dNA_K = cmn.cmnString.stringToDouble(sData);

        stList._NA++;
        st.dNA = st.dNA_K/2.0;
        stList._K++;
        st.dK  = st.dNA_K/2.0;
      }
    }

    // ..  Alkali Earth Metal

    if (sIdentifier.equals(_BE))    // "BERYLLIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BE++;
        st.dBE = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_MG))    // "Magnesium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MG++;
        st.dMG = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CA))    // "Calcium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CA++;
        st.dCA = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SR))    // "Strontium"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SR++;
        st.dSR = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_BA))    // "BARIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BA++;
        st.dBA = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIB

    if ((sIdentifier.equals(_CR)) ||   // "CHROMIUM"
        (sIdentifier.equals(_CRII)))   // "CHROMIUM II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR_II++;
        st.dCR_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CRIII))   // "CHROMIUM III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR_III++;
        st.dCR_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIIB

    if ((sIdentifier.equals(_MN)) ||   // "MANGANESE"
        (sIdentifier.equals(_MNII)))   // "MANGANESE II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MN_II++;
        st.dMN_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_MNIII))    // "MANGANESE III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MN_III++;
        st.dMN_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  VIIIB

    if ((sIdentifier.equals(_FE)) ||   // "Iron"
        (sIdentifier.equals(_FEII)))   // "Iron II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._FE_II++;
        st.dFE_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_FEIII))    // "Iron III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._FE_III++;
        st.dFE_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(_CO)) ||  // "COBALT"
        (sIdentifier.equals(_COII)))  // "COBALT II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO_II++;
        st.dCO_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_COIII))    // "COBALT III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO_III++;
        st.dCO_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(_NI)) ||   // "NICKEL"
        (sIdentifier.equals(_NIII)))   // "NICKEL II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NI_II++;
        st.dNI_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NIIII))    // "NICKEL III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NI_III++;
        st.dNI_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IB

    if (sIdentifier.equals(_CU))    // "COPPER"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CU_I++;
        st.dCU_I = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CUII))    // "COPPER II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CU_II++;
        st.dCU_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_AG))    // "SILVER"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AG++;
        st.dAG = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_AU))    // "GOLD"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AU++;
        st.dAU = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_AUIII))    // "GOLD III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AU_III++;
        st.dAU_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IIB

    if (sIdentifier.equals(_ZN))    // "ZINC"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._ZN++;
        st.dZN = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CD))    // "CADMIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CD++;
        st.dCD = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HG))    // "MERCURY"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HG++;
        st.dHG = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HGII))    // "MERCURY II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HG_II++;
        st.dHG_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IIIA

    if (sIdentifier.equals(_AL))    // "ALUMINUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AL++;
        st.dAL = cmn.cmnString.stringToDouble(sData);
      }
    }

    //.. VA

    if ((sIdentifier.equals(_SB)) ||   // "ANTIMONY"
        (sIdentifier.equals(_SBIII)))  // "ANTIMONY III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SB_III++;
        st.dSB_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SBV))    // "ANTIMONY V"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SB_V++;
        st.dSB_V = cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(_BI)) ||   // "Bismuth"
        (sIdentifier.equals(_BIIII)))  // "Bismuth III"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BI_III++;
        st.dBI_III = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_BIV))    // "Bismuth V"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BI_V++;
        st.dBI_V = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ..  IVA

    if ((sIdentifier.equals(_SN)) ||   // "Tin"
        (sIdentifier.equals(_SNII)))    // "Tin II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SN_II++;
        st.dSN_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SNIV))    // "Tin IV"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SN_IV++;
        st.dSN_IV = cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(_PB)) ||   // "LEAD"
        (sIdentifier.equals(_PBII)))    // "LEAD II"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PB_II++;
        st.dPB_II = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_PBIV))    // "LEAD IV"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PB_IV++;
        st.dPB_IV = cmn.cmnString.stringToDouble(sData);
      }
    }

    // .. Other

    if (sIdentifier.equals(_NH4))   // "AMMONIA"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NH4++;
        st.dNH4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    // ANIONS

    if (sIdentifier.equals(_F))     // "FLUORIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._F++;
        st.dF = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CL))    // "Chloride"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CL++;
        st.dCL = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_BR))    // "Bromine"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BR++;
        st.dBR = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_I))     // "Iodine"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._I++;
        st.dI = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_OH))    // "HYDROXIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OH++;
        st.dOH = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_BO3))   // "Borate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BO3++;
        st.dBO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CO3))   // "Carbonate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CO3++;
        st.dCO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HCO3))  // "Bicarbonate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HCO3++;
        st.dHCO3     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CLO))   // "HYPOCHLORITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO++;
        st.dCLO = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CLO2))  // "CHLORITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO2++;
        st.dCLO2 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CLO3))  // "CHLORATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO3++;
        st.dCLO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CLO4))  // "PERCHLORATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CLO4++;
        st.dCLO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CN))    // "CYANIDE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CN++;
        st.dCN = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NCO))    // "Cyanate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NCO++;
        st.dNCO = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_OCN))    // "Isocyanate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OCN++;
        st.dOCN = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SCN))    // "Thiocyanite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SCN++;
        st.dSCN = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_N))   // "Nitride"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._N++;
        st.dN = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_N3))   // "Azide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._N3++;
        st.dN3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NO2))   // "NITRITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NO2++;
        st.dNO2 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_NO3))   // "NITRATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NO3++;
        st.dNO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CRO4))   // "Chromium (II)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CRO4++;
        st.dCRO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_CR2O7))   // "Chromium (III)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CR2O7++;
        st.dCR2O7 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_MNO4))   // "Permanganate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MNO4++;
        st.dMNO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_P))     // "Phosphide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._P++;
        st.dP = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_PO3))     // "PHOSPHITE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PO3++;
        st.dPO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_PO4))     // "PHOSPHATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PO4++;
        st.dPO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HPO4))     // "Monohydrogen Phosphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HPO4++;
        st.dHPO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_H2PO4))     // "Dihydrogen Phosphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._H2PO4++;
        st.dH2PO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_AS))    // "ARSENIC"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._AS++;
        st.dAS = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SE))    // "SELENIUM"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SE++;
        st.dSE = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_S))   // "Sulfide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._S++;
        st.dS = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SO3))   // "Sulphite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SO3++;
        st.dSO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HSO3))   // "Hydrogen Sulphite"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HSO3++;
        st.dHSO3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_S2O3))   // "Thiosulphate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._S2O3++;
        st.dS2O3 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if ((sIdentifier.equals(_H2S)) ||  // "Hydrogen Sulfide"
        (sIdentifier.equals(_HS)))     // "Hydrogen Sulfide"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HS++;
        st.dHS = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_SO4))   // "Sulfate"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SO4++;
        st.dSO4 = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(_HSO4))  // "BISULFATE"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._HSO4++;
        st.dHSO4 = cmn.cmnString.stringToDouble(sData);
      }
    }
//    if (sIdentifier.equals(_B))     // "BORON"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._B++;
//        st.dB = cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(_BRO3))  // "BROMATE"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._BRO3++;
//        st.dBRO3 = cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(_MO))    // "MOLYBDENUM"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._MO++;
//        st.dMO = cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(_TL))    // "THALLIUM"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._TL++;
//        st.dTL = cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(_TH))    // "THORIUM"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._TH++;
//        st.dTH = cmn.cmnString.stringToDouble(sData);
//      }
//    }
//    if (sIdentifier.equals(_V))     // "VANADIUM"
//    {
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        stList._V++;
//        st.dV = cmn.cmnString.stringToDouble(sData);
//      }
//    }


    if (sIdentifier.equals(SOLID))  // "Total Solids"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SOLID++;
        st.dSOLID = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(TDS))    // "Total Solids Computed"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TDS++;
        st.dTDS = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(SPGR))   // "Specific Gravity"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SPGR++;
        st.dSPGR = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(SIGMA))  // "Specific Conductivity"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SIGMA++;
        st.dSIGMA = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(PH))     // "PH"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PH++;
        st.dPH = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(DEG))    // "Temperature (F)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._DEG++;
        st.dDEG = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(OHM))    // "Resistivity (Rw)"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHM++;
        st.dOHM = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(OHM75))  // "Resistivity at 75 deg"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHM75++;
        st.dOHM75 = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(OHME))   // "Estimated Rw"
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._OHME++;
        st.dOHME = cmn.cmnString.stringToDouble(sData);
      }
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
