/*
 * @brineSamplePlotFrame.java Version 1.1 06/13/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine.gui;

import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import gui.guiSearchDirectoryFrame;

import iqstrat.iqstratStruct;          // Global IQSTRAT Data Structure

import brine.brineListStruct;          // Common Brine Data Sample
import brine.gui.brineSamplePlotPanel; // Brine Stiff Plot Panel

/** Class brineSamplePlotFrame
 *  <p> This Class will create the Stiff Frame.
 *
 *  @version 1.1 06/13/2011
 *  @author  John R. Victorine
 */

public class brineSamplePlotFrame extends JFrame implements ActionListener, Observer
{
  // Variables passed to this panel

  private int                     iControl = 0;
  private brineListStruct         st       = null;

  // Global Variables

  private iqstratStruct           stStruct = null;

  private Observable              notifier = null; // Observalbles Object
  private brineSamplePlotPanel    plot     = null; // Sample Plot Panel
  private guiSearchDirectoryFrame pSearch  = null; // Search for Directory

  // Global Widgets

  private JButton  btnPDF   = new JButton();  // Create PDF of Sample Plot
  private JButton  btnDef   = new JButton();  // Display Definitions Web Page
  private JButton  btnClose = new JButton();  // Close Dialog

  /** Constructor brineSamplePlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param st  = Brine Data Structure
   */

  public brineSamplePlotFrame( brineListStruct st )
  {
    try
    {
      this.st = st;
      this.stStruct = new iqstratStruct();

      jbInit();

      addWindowListener( new brineSamplePlotFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor brineSamplePlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param iControl = Control Dialog Position
   *  @param st       = Brine Data Structure
   */

  public brineSamplePlotFrame( int iControl, brineListStruct st )
  {
    try
    {
	  this.iControl = iControl;
      this.st       = st;
      this.stStruct = new iqstratStruct();

      jbInit();

      addWindowListener( new brineSamplePlotFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	JPanel pnl = new JPanel();

    this.setTitle("Brine Sample Plot");
    this.getContentPane().setLayout(new BorderLayout());

    // Create an Observable

    notifier = new brineSamplePlotFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Buttons

    pnl.setLayout(new GridLayout());

    btnPDF.setFont(new java.awt.Font("Dialog", 1, 11));
    btnPDF.setText("Create PDF Document");
    btnPDF.addActionListener(this);

    btnDef.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDef.setText("Brine Definitions");
    btnDef.addActionListener(this);

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // Sample Plot Panel

    plot = new brineSamplePlotPanel( st );

    // Attach Widgets to Frame

    this.getContentPane().add(plot, BorderLayout.CENTER);
    this.getContentPane().add(pnl,  BorderLayout.SOUTH);
    pnl.add(btnPDF,                 null);
    pnl.add(btnDef,                 null);
    pnl.add(btnClose,               null);

    // Display the Frame

//    this.setSize(new Dimension(500, 400));
    this.setSize(new Dimension(650, 700));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    if (iControl > 0)
    {
      this.setLocation(iControl, 10);
	}
	else
	{
      this.setLocation((d.width  - this.getSize().width) / 2,
                       (d.height - this.getSize().height) / 2);
    }

    this.setResizable(true);
    this.setVisible(true);
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    st       = null;
    stStruct = null;

    if (plot != null)
      plot.close();
    plot = null;

    if (pSearch != null)
      pSearch.close();
    pSearch = null;

    btnPDF   = null;  // Create PDF of Sample Plot
    btnDef   = null;  // Display Definitions Web Page
    btnClose = null;  // Close Dialog

    dispose();
  }

  /** METHOD brineSamplePlotFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class brineSamplePlotFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /* ======================================================================= *
   * --------------------------------- PDF --------------------------------- *
   * ======================================================================= */

  /** Method createPDF()
   *  <p> This method will create a PDF Image of Profile Plot
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  private void createPDF()
  {
    iqstrat.io.iqstratIO_Files.createImage( stStruct, plot.getImage() );
    iqstrat.io.iqstratIO_Files.createHTML( stStruct );
//    iqstrat.io.iqstratIO_PDF.createImage( stStruct, plot.getImage() );
//    iqstrat.io.iqstratIO_PDF.createPDF( stStruct );
  }

  /* ======================================================================= *
   * ----------------------------- GET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int  getPlotWidth()  { return ( plot.getPlotWidth() ); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return ( plot.getPlotHeight() ); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;

    if (plot != null) image = plot.getImage();

    return (image);
  }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setRow()
   * <p> This method will set the plot data to plot
   */

  public void setRow(int iRow)
  {
	if (plot != null) plot.setRow(iRow);
  }

  /* ======================================================================= *
   * ------------------------------- ACTIONS ------------------------------- *
   * ======================================================================= */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnPDF)
    {
      String sPath        = kgs.io.kgsIOUser.getHomeDir();
      stStruct.sDirectory = new String( sPath );
      stStruct.sName      = new String( "SamplePlot" );

      pSearch = new guiSearchDirectoryFrame(notifier, sPath, "SamplePlot", ".png");
	}

    if (event.getSource() == btnDef)
    {
      util.BrowserControl.displayURL( cmn.cmnStruct.BRINE_DEFINITON );
	}

    if (event.getSource() == btnClose) { close(); }
  }

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String        sArg     = new String((String) arg);

    if (sArg.equals("Directory Path Selected"))
    {
      if (pSearch != null)
      {
        stStruct.sDirectory = new String( pSearch.getPath() );
        stStruct.sName      = new String( pSearch.getFile() );
      }

      createPDF();

      if (pSearch != null)
        pSearch.close();
      pSearch = null;
	}
  }
}

/** CLASS brineSamplePlotFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class brineSamplePlotFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 06/13/2011
 *  @author  John Victorine
 */
