/*
 * @brineUtility.java Version 1.1 05/27/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine;

import brine.brineStruct;
import brine.brineListStruct;

import iqstrat.iqstratHeadersStruct;

/** Class brineUtility
 *  <p> This Class will provide basic utilities for the Rock data structures.
 *
 *  @version 1.1 05/27/2011
 *  @author  John R. Victorine
 */

public class brineUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the brine data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Brine List Data Structure
   * @return st       = The Brine List Data Structure.
   */

  public static brineListStruct addHeaderData( iqstratHeadersStruct stHeader,
                                               brineListStruct   st)
  {
	String sUKey = new String(cmn.cmnString.UniqueName());

    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      st.sField     = new String(stHeader.sField);      // Field
      st.sCounty    = new String(stHeader.sCounty);     // County
      st.state      = new String(stHeader.state);       // State
      st.sLocation  = new String(stHeader.sLocation);   // Location (TRS)

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dKB   = stHeader.dKB;   // Kelly Bushing
      st.dGL   = stHeader.dGL;   // Ground Level
      st.dDF   = stHeader.dDF;   // Derrick Floor

      for (int i=0; i<st.iCount; i++)
      {
		if (st.stItem[i] != null)
		{
		  if (st.stItem[i].sKID.length() < 2)
		    st.stItem[i].sKID     = new String(sUKey + "_" + i);

		  if (st.stItem[i].sKEY.length() < 2)
		    st.stItem[i].sKEY     = new String(sUKey + "_" + i);

          st.stItem[i].sAPI       = new String(stHeader.sAPI);
          st.stItem[i].sName      = new String(stHeader.sName);

          // XY Position Information

          st.stItem[i].dLatitude  = stHeader.dLatitude;
          st.stItem[i].dLongitude = stHeader.dLongitude;

          st.stItem[i].sField     = new String(stHeader.sField);
          st.stItem[i].sCounty    = new String(stHeader.sCounty);
          st.stItem[i].state      = new String(stHeader.state);
          st.stItem[i].sLocation  = new String(stHeader.sLocation);

          st.stItem[i].dKB        = stHeader.dKB;
          st.stItem[i].dGL        = stHeader.dGL;
          st.stItem[i].dDF        = stHeader.dDF;


          if (st.stItem[i].source.length() == 0)
            st.stItem[i].source   = new String(st.source);
          if (st.stItem[i].sRecovery.length() == 0)
            st.stItem[i].sRecovery = new String(st.sRecovery);

          // AGE

          if (st.stItem[i].sFORM.length() == 0)
            st.stItem[i].sFORM    = new String(st.sFormation);
		}
	  }
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the brine data
   * @param   st = the Brine data list structure
   * @return  st = the Brine data list structure
   */

  public static brineListStruct computeDepthRange(brineListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;
    int    iBegin     = 0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].dTOP;
        depthEnd   = st.stItem[i].dBASE;
        if (depthEnd  == brine.brineStruct.dNULL)
          depthEnd = depthStart;

//        if (i==0)
        if ((iBegin==0) && (depthStart != brine.brineStruct.dNULL))
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
          iBegin = 1;
        }
        else
        {
          if (st.depthStart > depthStart)
            st.depthStart = depthStart;

          if (st.depthEnd < depthEnd)
            st.depthEnd   = depthEnd;
        }
      }
    }

    return (st);
  }

  /** Method addBaseDepth()
   * <p> This method will add a base depth if it is null by determining the
   *     difference from one sample depth with the next and picking the smallest
   *     interval between them
   * @param  stList = Brine Data list data structure
   * @return stList = Brine Data list data structure
   */

  public static brineListStruct addBaseDepth( brineListStruct stList )
  {
    int    i     = 0;
    double diff  = 0.0;
    double diffo = 2.0;

    if (stList != null)
    {
      if (stList.iCount > 0)
      {
        for (i=0; i<stList.iCount; i++)
        {
          if (i < stList.iCount-1)
          {
            diff = stList.stItem[i+1].dTOP - stList.stItem[i].dTOP;
            if (diff < diffo) diffo = diff;
          }
        }

        for (i=0; i<stList.iCount; i++)
        {
          if ((stList.stItem[i].dBASE == brine.brineStruct.dNULL) &&
              (stList.stItem[i].dTOP != brine.brineStruct.dNULL))
          {
            stList.stItem[i].dBASE = stList.stItem[i].dTOP + diffo;
          }
        }
      }
    }

    return (stList);
  }

  /** Method copyList()
   * <p> This method will copy one Brine Data list structure to another
   * @param  stOld = Old Brine Data list data structure
   * @return stNew = New Brine Data list data structure
   */

  public static brineListStruct copyList(brineListStruct stOld)
  {
    int i,j;
    brineListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new brineListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new brineStruct[stOld.iCount];

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID);   // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
        stNew.sName  = new String(stOld.sName);  // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        stNew.sField    = new String(stOld.sField);      // Field
        stNew.sCounty   = new String(stOld.sCounty);     // County
        stNew.state     = new String(stOld.state);       // State
        stNew.sLocation = new String(stOld.sLocation);   // Location (TRS)

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL;   // Ground Level
        stNew.dKB   = stOld.dKB;   // Kelly Bushing
        stNew.dDF   = stOld.dDF;   // Derrick Floor
        stNew.dTGT  = stOld.dTGT;  // TGT

        // -- LAS 3 Output Required Variables

        stNew.sKGS       = new String(stOld.sKGS);       // KGS Saved Data Indicator
        stNew.source     = new String(stOld.source);     // Core Source
        stNew.sRecovery  = new String(stOld.sRecovery);  // Recovery Date
        stNew.depthStart = stOld.depthStart;             // Brine Top Depth
        stNew.depthEnd   = stOld.depthEnd;               // Brine Base Depth
        stNew.sFormation = new String(stOld.sFormation); // Primary Formation
        stNew.sCompany   = new String(stOld.sCompany);   // Analyzing Company
        stNew.sAnalyzed  = new String(stOld.sAnalyzed);  // Analysis Date

        // Brine Data Identifiers

        stNew._KEY    = stOld._KEY;    // KEY

        // AGE
        stNew._FORM   = stOld._FORM;   // Primary Formation
        stNew._AGE    = stOld._AGE;    // Age of Formation

        // Depth
        stNew._TOP    = stOld._TOP;    // Top Depth
        stNew._BASE   = stOld._BASE;   // Base Depth

        // Other Measured Well Data
        stNew._SPGR   = stOld._SPGR;   // Specific Gravity
        stNew._SIGMA  = stOld._SIGMA;  // Specific Conductivity
        stNew._PH     = stOld._PH;     // PH
        stNew._DEG    = stOld._DEG;    // Degree F
        stNew._OHM    = stOld._OHM;    // Resistivity (Rw)
        stNew._OHM75  = stOld._OHM75;  // Resistivity at 75 deg
        stNew._OHME   = stOld._OHME;   // Estimated Rw

        // .. Alkali Metal
        stNew._LI     = stOld._LI;     // Lithium
        stNew._NA     = stOld._NA;     // Sodium
        stNew._K      = stOld._K;      // Potassium
        stNew._RB     = stOld._RB;     // Rubidium
        stNew._CS     = stOld._CS;     // Cesium
        stNew._NA_K   = stOld._NA_K;   // Sodium/Potasium

        // ..  Alkali Earth Metal
        stNew._BE     = stOld._BE;     // Beryllium
        stNew._MG     = stOld._MG;     // Magnesium
        stNew._CA     = stOld._CA;     // Calcium
        stNew._SR     = stOld._SR;     // Strontium
        stNew._BA     = stOld._BA;     // Barium

        // ..  VIB
        stNew._CR_II  = stOld._CR_II;  // Chromium (II)
        stNew._CR_III = stOld._CR_III; // Chromium (iii)

        // ..  VIIB
        stNew._MN_II  = stOld._MN_II;  // Manganese (II)
        stNew._MN_III = stOld._MN_III; // Manganese (III)

        // ..  VIIIB
        stNew._FE_II  = stOld._FE_II;  // Iron (II)
        stNew._FE_III = stOld._FE_III; // Iron (III)
        stNew._CO_II  = stOld._CO_II;  // Cobalt (II)
        stNew._CO_III = stOld._CO_III; // Cobalt (III)
        stNew._NI_II  = stOld._NI_II;  // Nickel (II)
        stNew._NI_III = stOld._NI_III; // Nickel (III)

        // ..  IB
        stNew._CU_I   = stOld._CU_I;   // Copper (I)
        stNew._CU_II  = stOld._CU_II;  // Copper (II)
        stNew._AG     = stOld._AG;     // Silver
        stNew._AU     = stOld._AU;     // Gold
        stNew._AU_III = stOld._AU_III; // Gold (III)

        // ..  IIB
        stNew._ZN     = stOld._ZN;    // Zinc
        stNew._CD     = stOld._CD;    // Cadmium
        stNew._HG     = stOld._HG;    // Mercury (I)
        stNew._HG_II  = stOld._HG_II; // Mercury (II)

        // ..  IIIA
        stNew._AL     = stOld._AL;    // Aluminium

        // ..
        stNew._SB_III = stOld._SB_III; // Antimonous(III)
        stNew._SB_V   = stOld._SB_V;   // Antimonic(V)
        stNew._BI_III = stOld._BI_III; // Bismuthous(III)
        stNew._BI_V   = stOld._BI_V;   // Bismuth(V)

        // ..  IVA
        stNew._SN_II  = stOld._SN_II; // Tin (II)
        stNew._SN_IV  = stOld._SN_IV; // Tin (IV)
        stNew._PB_II  = stOld._PB_II; // Lead (II)
        stNew._PB_IV  = stOld._PB_IV; // Lead (IV)

        // .. Other
        stNew._NH4    = stOld._NH4;   // Ammonium

        // Common Anions

        // ..Halogen
        stNew._F      = stOld._F;     // Flouride
        stNew._CL     = stOld._CL;    // Chloride
        stNew._BR     = stOld._BR;    // Bromide
        stNew._I      = stOld._I;     // Iodide

        stNew._OH     = stOld._OH;    // Hydroxide

        stNew._BO3    = stOld._BO3;   // Borate
        stNew._CO3    = stOld._CO3;   // Carbonate
        stNew._HCO3   = stOld._HCO3;  // Bicarbonate

        stNew._CLO    = stOld._CLO;   // Hypochlorite
        stNew._CLO2   = stOld._CLO2;  // Chlorite
        stNew._CLO3   = stOld._CLO3;  // Chlorate
        stNew._CLO4   = stOld._CLO4;  // Per chlorate

        stNew._CN     = stOld._CN;    // Cyanide
        stNew._NCO    = stOld._NCO;   // Cyanate
        stNew._OCN    = stOld._OCN;   // Isocyanate
        stNew._SCN    = stOld._SCN;   // Thicyanite

        stNew._N      = stOld._N;     // Nitride
        stNew._N3     = stOld._N3;    // Azide
        stNew._NO2    = stOld._NO2;   // Nitrite
        stNew._NO3    = stOld._NO3;   // Nitrate

        stNew._CRO4   = stOld._CRO4;  // Chromium (II)
        stNew._CR2O7  = stOld._CR2O7; // Chromium (III)

        stNew._MNO4   = stOld._MNO4;  // Permanganate

        stNew._P      = stOld._P;     // Phosphide
        stNew._PO3    = stOld._PO3;   // Phosphite
        stNew._PO4    = stOld._PO4;   // Phosphate
        stNew._HPO4   = stOld._HPO4;  // Monohydrogen Phosphate
        stNew._H2PO4  = stOld._H2PO4; // Dihydrogen Phosphate

        stNew._AS     = stOld._AS;    // Arsinide
        stNew._SE     = stOld._SE;    // Selenide

        stNew._S      = stOld._S;     // Sulfide
        stNew._HS     = stOld._HS;    // Hydrogen Sulfide
        stNew._SO3    = stOld._SO3;   // Sulphite
        stNew._HSO3   = stOld._HSO3;  // Hydrogen Sulphite
        stNew._S2O3   = stOld._S2O3;  // Thiosulphate
        stNew._SO4    = stOld._SO4;   // Sulfate
        stNew._HSO4   = stOld._HSO4;  // Bisulfate

        stNew._SOLID  = stOld._SOLID; // Total Solids
        stNew._TDS    = stOld._TDS;   // Total Solids Computed

        for (i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static brineListStruct transfer(brineListStruct stOld)
  {
    brineListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copyBrineHeader()
   * <p> This method will copy one Brine Data list structure to another
   * @param  stOld = Old Brine Data list data structure
   * @return stNew = New Brine Data list data structure
   */

  public static brineListStruct copyBrineHeader(brineListStruct stOld)
  {
    int i,j;
    brineListStruct stNew = null;

    if (stOld != null)
    {
        stNew        = new brineListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new brineStruct[stOld.iCount];

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID); // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY); // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI); // API-Number of Location
        stNew.sName  = new String(stOld.sName); // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        stNew.sField    = new String(stOld.sField);      // Field
        stNew.sCounty   = new String(stOld.sCounty);     // County
        stNew.state     = new String(stOld.state);       // State
        stNew.sLocation = new String(stOld.sLocation);   // Location (TRS)

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL;   // Ground Level
        stNew.dKB   = stOld.dKB;   // Kelly Bushing
        stNew.dDF   = stOld.dDF;   // Derrick Floor
        stNew.dTGT  = stOld.dTGT;  // TGT

        // -- LAS 3 Output Required Variables

        stNew.sKGS       = new String(stOld.sKGS);       // KGS Saved Data Indicator
        stNew.source     = new String(stOld.source);     // Core Source
        stNew.sRecovery  = new String(stOld.sRecovery);  // Recovery Date
        stNew.depthStart = stOld.depthStart;             // Brine Top Depth
        stNew.depthEnd   = stOld.depthEnd;               // Brine Base Depth
        stNew.sFormation = new String(stOld.sFormation); // Primary Formation
        stNew.sCompany   = new String(stOld.sCompany);   // Analyzing Company
        stNew.sAnalyzed  = new String(stOld.sAnalyzed);  // Analysis Date

        // Brine Data Identifiers

        stNew._KEY    = stOld._KEY;    // KEY

        // AGE
        stNew._FORM   = stOld._FORM;   // Primary Formation
        stNew._AGE    = stOld._AGE;    // Age of Formation

        // Depth
        stNew._TOP    = stOld._TOP;    // Top Depth
        stNew._BASE   = stOld._BASE;   // Base Depth

        // Other Measured Well Data
        stNew._SPGR   = stOld._SPGR;   // Specific Gravity
        stNew._SIGMA  = stOld._SIGMA;  // Specific Conductivity
        stNew._PH     = stOld._PH;     // PH
        stNew._DEG    = stOld._DEG;    // Degree F
        stNew._OHM    = stOld._OHM;    // Resistivity (Rw)
        stNew._OHM75  = stOld._OHM75;  // Resistivity at 75 deg
        stNew._OHME   = stOld._OHME;   // Estimated Rw

        // .. Alkali Metal
        stNew._LI     = stOld._LI;     // Lithium
        stNew._NA     = stOld._NA;     // Sodium
        stNew._K      = stOld._K;      // Potassium
        stNew._RB     = stOld._RB;     // Rubidium
        stNew._CS     = stOld._CS;     // Cesium
        stNew._NA_K   = stOld._NA_K;   // Sodium/Potasium

        // ..  Alkali Earth Metal
        stNew._BE     = stOld._BE;     // Beryllium
        stNew._MG     = stOld._MG;     // Magnesium
        stNew._CA     = stOld._CA;     // Calcium
        stNew._SR     = stOld._SR;     // Strontium
        stNew._BA     = stOld._BA;     // Barium

        // ..  VIB
        stNew._CR_II  = stOld._CR_II;  // Chromium (II)
        stNew._CR_III = stOld._CR_III; // Chromium (iii)

        // ..  VIIB
        stNew._MN_II  = stOld._MN_II;  // Manganese (II)
        stNew._MN_III = stOld._MN_III; // Manganese (III)

        // ..  VIIIB
        stNew._FE_II  = stOld._FE_II;  // Iron (II)
        stNew._FE_III = stOld._FE_III; // Iron (III)
        stNew._CO_II  = stOld._CO_II;  // Cobalt (II)
        stNew._CO_III = stOld._CO_III; // Cobalt (III)
        stNew._NI_II  = stOld._NI_II;  // Nickel (II)
        stNew._NI_III = stOld._NI_III; // Nickel (III)

        // ..  IB
        stNew._CU_I   = stOld._CU_I;   // Copper (I)
        stNew._CU_II  = stOld._CU_II;  // Copper (II)
        stNew._AG     = stOld._AG;     // Silver
        stNew._AU     = stOld._AU;     // Gold
        stNew._AU_III = stOld._AU_III; // Gold (III)

        // ..  IIB
        stNew._ZN     = stOld._ZN;    // Zinc
        stNew._CD     = stOld._CD;    // Cadmium
        stNew._HG     = stOld._HG;    // Mercury (I)
        stNew._HG_II  = stOld._HG_II; // Mercury (II)

        // ..  IIIA
        stNew._AL     = stOld._AL;    // Aluminium

        // ..
        stNew._SB_III = stOld._SB_III; // Antimonous(III)
        stNew._SB_V   = stOld._SB_V;   // Antimonic(V)
        stNew._BI_III = stOld._BI_III; // Bismuthous(III)
        stNew._BI_V   = stOld._BI_V;   // Bismuth(V)

        // ..  IVA
        stNew._SN_II  = stOld._SN_II; // Tin (II)
        stNew._SN_IV  = stOld._SN_IV; // Tin (IV)
        stNew._PB_II  = stOld._PB_II; // Lead (II)
        stNew._PB_IV  = stOld._PB_IV; // Lead (IV)

        // .. Other
        stNew._NH4    = stOld._NH4;   // Ammonium

        // Common Anions

        // ..Halogen
        stNew._F      = stOld._F;     // Flouride
        stNew._CL     = stOld._CL;    // Chloride
        stNew._BR     = stOld._BR;    // Bromide
        stNew._I      = stOld._I;     // Iodide

        stNew._OH     = stOld._OH;    // Hydroxide

        stNew._BO3    = stOld._BO3;   // Borate
        stNew._CO3    = stOld._CO3;   // Carbonate
        stNew._HCO3   = stOld._HCO3;  // Bicarbonate

        stNew._CLO    = stOld._CLO;   // Hypochlorite
        stNew._CLO2   = stOld._CLO2;  // Chlorite
        stNew._CLO3   = stOld._CLO3;  // Chlorate
        stNew._CLO4   = stOld._CLO4;  // Per chlorate

        stNew._CN     = stOld._CN;    // Cyanide
        stNew._NCO    = stOld._NCO;   // Cyanate
        stNew._OCN    = stOld._OCN;   // Isocyanate
        stNew._SCN    = stOld._SCN;   // Thicyanite

        stNew._N      = stOld._N;     // Nitride
        stNew._N3     = stOld._N3;    // Azide
        stNew._NO2    = stOld._NO2;   // Nitrite
        stNew._NO3    = stOld._NO3;   // Nitrate

        stNew._CRO4   = stOld._CRO4;  // Chromium (II)
        stNew._CR2O7  = stOld._CR2O7; // Chromium (III)

        stNew._MNO4   = stOld._MNO4;  // Permanganate

        stNew._P      = stOld._P;     // Phosphide
        stNew._PO3    = stOld._PO3;   // Phosphite
        stNew._PO4    = stOld._PO4;   // Phosphate
        stNew._HPO4   = stOld._HPO4;  // Monohydrogen Phosphate
        stNew._H2PO4  = stOld._H2PO4; // Dihydrogen Phosphate

        stNew._AS     = stOld._AS;    // Arsinide
        stNew._SE     = stOld._SE;    // Selenide

        stNew._S      = stOld._S;     // Sulfide
        stNew._HS     = stOld._HS;    // Hydrogen Sulfide
        stNew._SO3    = stOld._SO3;   // Sulphite
        stNew._HSO3   = stOld._HSO3;  // Hydrogen Sulphite
        stNew._S2O3   = stOld._S2O3;  // Thiosulphate
        stNew._SO4    = stOld._SO4;   // Sulfate
        stNew._HSO4   = stOld._HSO4;  // Bisulfate

        stNew._SOLID  = stOld._SOLID; // Total Solids
        stNew._TDS    = stOld._TDS;   // Total Solids Computed
    }

    return (stNew);
  }


  /** Method add()
   * <p> This method will add the Brine Data to an existing list
   * @param  stBrine = The Brine Data Structure
   * @param  st     = The Old Brine Data List Data Structure
   * @return st     = The new Brine Data List Data Structure.
   */

  public static brineListStruct add( brineStruct stBrine,
                                     brineListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    brineListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

//    stTemp        = new brineListStruct();
    stTemp        = copyBrineHeader( st );
    stTemp.stItem = new brineStruct[iRecords];

    iSource       = stTemp.iSource;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new brineStruct();
    stTemp.stItem[iCount] = copy(stBrine);
    iCount++;

    stTemp.iCount = iCount;

//    st        = new brineListStruct();
    st        = copyBrineHeader( stTemp );
    st.stItem = new brineStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a data structrure in an existing list
   * @param  sKEY    = The Primary Key of the data to be removed
   * @param  stBlock = The fossil structure that will be replaced
   * @param  st      = The List Data Structure
   * @return st      = The new List Data Structure.
   */

  public static brineListStruct modify( String sKEY,
                                        brineStruct stBlock,
                                        brineListStruct st)
  {
    int i=0;
    brineListStruct stTemp = null;

    if (st != null)
    {
//      stTemp        = new brineListStruct();
      stTemp        = copyBrineHeader( st );
      stTemp.stItem = new brineStruct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

//      st        = new brineListStruct();
      st        = copyBrineHeader( stTemp );
      st.stItem = new brineStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a data structure from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old List Data Structure
   * @return st     = The new List Data Structure.
   */

  public static brineListStruct remove( String sKEY, brineListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    brineListStruct stTemp = null;

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        iRecords = st.iCount-1;

//        stTemp        = new brineListStruct();
        stTemp        = copyBrineHeader( st );
        stTemp.stItem = new brineStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

//      st        = new brineListStruct();
      st        = copyBrineHeader( stTemp );
      st.stItem = new brineStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one Brine data structure to another
   * @param  stOld = old Brine data structure
   * @return stNew = new Brine data structure
   */

  public static brineStruct copy (brineStruct stOld)
  {
    brineStruct stNew = new brineStruct();

    if (stOld != null)
    {
        // Brine Data Identifiers

        stNew.sKID    = new String( stOld.sKID );    // KID
        stNew.sKEY    = new String( stOld.sKEY );    // KEY

        // Identification Information
        stNew.sAPI       = new String(stOld.sAPI);        // API-Number
        stNew.sName      = new String(stOld.sName);       // Name
        stNew.sField     = new String(stOld.sField);      // Field
        stNew.sCounty    = new String(stOld.sCounty);     // County
        stNew.state      = new String(stOld.state);       // State
        stNew.sLocation  = new String(stOld.sLocation);   // Location (TRS)

        // XY Position Information
        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Elevation Data
        stNew.dGL        = stOld.dGL;   // Ground Level
        stNew.dKB        = stOld.dKB;   // Kelly Bushing
        stNew.dDF        = stOld.dDF;   // Derrick Floor
        stNew.dTGT       = stOld.dTGT;  // TGT

        // Data
        stNew.source     = new String(stOld.source);     // Core Source
        stNew.sRecovery  = new String(stOld.sRecovery);  // Recovery Date

        // AGE
        stNew.sFORM   = new String( stOld.sFORM );   // Primary Formation
        stNew.sAGE    = new String( stOld.sAGE );    // Age of Formation

        // Depth
        stNew.dTOP    = stOld.dTOP;    // Top Depth
        stNew.dBASE   = stOld.dBASE;   // Base Depth

        // Other Measured Well Data
        stNew.dSPGR   = stOld.dSPGR;   // Specific Gravity
        stNew.dSIGMA  = stOld.dSIGMA;  // Specific Conductivity
        stNew.dPH     = stOld.dPH;     // PH
        stNew.dDEG    = stOld.dDEG;    // Degree F
        stNew.dOHM    = stOld.dOHM;    // Resistivity (Rw)
        stNew.dOHM75  = stOld.dOHM75;  // Resistivity at 75 deg
        stNew.dOHME   = stOld.dOHME;   // Estimated Rw

        // .. Alkali Metal
        stNew.dLI     = stOld.dLI;     // Lithium
        stNew.dNA     = stOld.dNA;     // Sodium
        stNew.dK      = stOld.dK;      // Potassium
        stNew.dRB     = stOld.dRB;     // Rubidium
        stNew.dCS     = stOld.dCS;     // Cesium
        stNew.dNA_K   = stOld.dNA_K;   // Sodium/Potasium

        // ..  Alkali Earth Metal
        stNew.dBE     = stOld.dBE;     // Beryllium
        stNew.dMG     = stOld.dMG;     // Magnesium
        stNew.dCA     = stOld.dCA;     // Calcium
        stNew.dSR     = stOld.dSR;     // Strontium
        stNew.dBA     = stOld.dBA;     // Barium

        // ..  VIB
        stNew.dCR_II  = stOld.dCR_II;  // Chromium (II)
        stNew.dCR_III = stOld.dCR_III; // Chromium (iii)

        // ..  VIIB
        stNew.dMN_II  = stOld.dMN_II;  // Manganese (II)
        stNew.dMN_III = stOld.dMN_III; // Manganese (III)

        // ..  VIIIB
        stNew.dFE_II  = stOld.dFE_II;  // Iron (II)
        stNew.dFE_III = stOld.dFE_III; // Iron (III)
        stNew.dCO_II  = stOld.dCO_II;  // Cobalt (II)
        stNew.dCO_III = stOld.dCO_III; // Cobalt (III)
        stNew.dNI_II  = stOld.dNI_II;  // Nickel (II)
        stNew.dNI_III = stOld.dNI_III; // Nickel (III)

        // ..  IB
        stNew.dCU_I   = stOld.dCU_I;   // Copper (I)
        stNew.dCU_II  = stOld.dCU_II;  // Copper (II)
        stNew.dAG     = stOld.dAG;     // Silver
        stNew.dAU     = stOld.dAU;     // Gold
        stNew.dAU_III = stOld.dAU_III; // Gold (III)

        // ..  IIB
        stNew.dZN     = stOld.dZN;    // Zinc
        stNew.dCD     = stOld.dCD;    // Cadmium
        stNew.dHG     = stOld.dHG;    // Mercury (I)
        stNew.dHG_II  = stOld.dHG_II; // Mercury (II)

        // ..  IIIA
        stNew.dAL     = stOld.dAL;    // Aluminium

        // ..
        stNew.dSB_III = stOld.dSB_III; // Antimonous(III)
        stNew.dSB_V   = stOld.dSB_V;   // Antimonic(V)
        stNew.dBI_III = stOld.dBI_III; // Bismuthous(III)
        stNew.dBI_V   = stOld.dBI_V;   // Bismuth(V)

        // ..  IVA
        stNew.dSN_II  = stOld.dSN_II; // Tin (II)
        stNew.dSN_IV  = stOld.dSN_IV; // Tin (IV)
        stNew.dPB_II  = stOld.dPB_II; // Lead (II)
        stNew.dPB_IV  = stOld.dPB_IV; // Lead (IV)

        // .. Other
        stNew.dNH4    = stOld.dNH4;   // Ammonium

        // Common Anions

        // ..Halogen
        stNew.dF      = stOld.dF;     // Flouride
        stNew.dCL     = stOld.dCL;    // Chloride
        stNew.dBR     = stOld.dBR;    // Bromide
        stNew.dI      = stOld.dI;     // Iodide

        stNew.dOH     = stOld.dOH;    // Hydroxide

        stNew.dBO3    = stOld.dBO3;   // Borate
        stNew.dCO3    = stOld.dCO3;   // Carbonate
        stNew.dHCO3   = stOld.dHCO3;  // Bicarbonate

        stNew.dCLO    = stOld.dCLO;   // Hypochlorite
        stNew.dCLO2   = stOld.dCLO2;  // Chlorite
        stNew.dCLO3   = stOld.dCLO3;  // Chlorate
        stNew.dCLO4   = stOld.dCLO4;  // Per chlorate

        stNew.dCN     = stOld.dCN;    // Cyanide
        stNew.dNCO    = stOld.dNCO;   // Cyanate
        stNew.dOCN    = stOld.dOCN;   // Isocyanate
        stNew.dSCN    = stOld.dSCN;   // Thicyanite

        stNew.dN      = stOld.dN;     // Nitride
        stNew.dN3     = stOld.dN3;    // Azide
        stNew.dNO2    = stOld.dNO2;   // Nitrite
        stNew.dNO3    = stOld.dNO3;   // Nitrate

        stNew.dCRO4   = stOld.dCRO4;  // Chromium (II)
        stNew.dCR2O7  = stOld.dCR2O7; // Chromium (III)

        stNew.dMNO4   = stOld.dMNO4;  // Permanganate

        stNew.dP      = stOld.dP;     // Phosphide
        stNew.dPO3    = stOld.dPO3;   // Phosphite
        stNew.dPO4    = stOld.dPO4;   // Phosphate
        stNew.dHPO4   = stOld.dHPO4;  // Monohydrogen Phosphate
        stNew.dH2PO4  = stOld.dH2PO4; // Dihydrogen Phosphate

        stNew.dAS     = stOld.dAS;    // Arsinide
        stNew.dSE     = stOld.dSE;    // Selenide

        stNew.dS      = stOld.dS;     // Sulfide
        stNew.dHS     = stOld.dHS;    // Hydrogen Sulfide
        stNew.dSO3    = stOld.dSO3;   // Sulphite
        stNew.dHSO3   = stOld.dHSO3;  // Hydrogen Sulphite
        stNew.dS2O3   = stOld.dS2O3;  // Thiosulphate
        stNew.dSO4    = stOld.dSO4;   // Sulfate
        stNew.dHSO4   = stOld.dHSO4;  // Bisulfate

        stNew.dSOLID  = stOld.dSOLID; // Total Solids
        stNew.dTDS    = stOld.dTDS;   // Total Solids Computed
    }

    return (stNew);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = brine data list structure
   * @param  By = By Identifier
   * @return st = sorted brine data list structure
   */

  public static brineListStruct bubbleSort(brineListStruct st, int By)
  {
    boolean swappedOnPrevRun  = true;
    brineStruct stTemp   = null;
    double      d1       = 0.0;
    double      d2       = 0.0;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            switch (By)
            {
			  case 0:
			    d1 = st.stItem[i].dTOP;
			    d2 = st.stItem[i+1].dTOP;
                break;
			  case 1:
			    d1 = st.stItem[i].dLatitude;
			    d2 = st.stItem[i+1].dLatitude;
                break;
		    }

            if(d1 > d2)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the brine data
   * @param  st = brine data list structure
   */

  public static void print(brineListStruct st)
  {
    if (st != null)
    {
      System.out.println("API: "    + st.sAPI);   // API-Number of Location
      System.out.println("Name: "   + st.sName);  // Well Name or Outcrop Name

      System.out.println("Field: "  + st.sField);      // Field
      System.out.println("County: " + st.sCounty);     // County
      System.out.println("State: "  + st.state);       // State
      System.out.println("Loc: "    + st.sLocation);   // Location (TRS)

      // XY Position Information

      System.out.println("Latitude: "  + st.dLatitude);   // Latitude
      System.out.println("Longitude: " + st.dLongitude);  // Longitude

      // Z Position - Depth, Elevation

      System.out.println("KB: " + st.dKB);   // Kelly Bushing
      System.out.println("GL: " + st.dGL);   // Ground Level
      System.out.println("DF: " + st.dDF);   // Derrick Floor

	  for (int i=0; i<st.iCount; i++)
	  {
        System.out.println("\n---------------------------------------------");
        System.out.println(st.stItem[i].sName + " " +
                           st.stItem[i].dLatitude + " " +
                           st.stItem[i].dLongitude);

		if ((st.stItem[i].dTOP > -1.0) ||
		    (st.stItem[i].dBASE > -1.0))
		{
          System.out.print("Depth: ");

	      if (st.stItem[i].dTOP > -1.0)
	      {
		    System.out.print("TOP: " + st.stItem[i].dTOP + " ft");
		  }

	      if (st.stItem[i].dBASE > -1.0)
	      {
		    System.out.print(" BASE: " + st.stItem[i].dBASE + " ft");
          }

          System.out.println(" ");
        }

		if ((st.stItem[i].dLI > -1.0) ||
		    (st.stItem[i].dNA > -1.0) ||
		    (st.stItem[i].dK > -1.0)  ||
		    (st.stItem[i].dCS > -1.0))
		{
          System.out.print("Alkali: ");

	      if (st.stItem[i].dLI > -1.0)
	      {
		    System.out.print(" Li: " + st.stItem[i].dLI + " mg/l");
		  }

	      if (st.stItem[i].dNA > -1.0)
	      {
		    System.out.print(" Na: " + st.stItem[i].dNA + " mg/l");
          }

	      if (st.stItem[i].dK > -1.0)
	      {
		    System.out.print(" K: " + st.stItem[i].dK + " mg/l");
		  }

	      if (st.stItem[i].dCS > -1.0)
	      {
		    System.out.print(" Cs: " + st.stItem[i].dCS + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.stItem[i].dMG > -1.0) ||
		    (st.stItem[i].dCA > -1.0) ||
		    (st.stItem[i].dSR > -1.0) ||
		    (st.stItem[i].dBA > -1.0))
		{
          System.out.print("Earth: ");

	      if (st.stItem[i].dMG > -1.0)
	      {
		    System.out.print(" Mg: " + st.stItem[i].dMG + " mg/l");
		  }

	      if (st.stItem[i].dCA > -1.0)
	      {
		    System.out.print(" Ca: " + st.stItem[i].dCA + " mg/l");
          }

	      if (st.stItem[i].dSR > -1.0)
	      {
		    System.out.print(" Sr: " + st.stItem[i].dSR + " mg/l");
		  }

	      if (st.stItem[i].dBA > -1.0)
	      {
		    System.out.print(" Ba: " + st.stItem[i].dBA + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.stItem[i].dF > -1.0) ||
		    (st.stItem[i].dCL > -1.0) ||
		    (st.stItem[i].dBR > -1.0) ||
		    (st.stItem[i].dI > -1.0))
		{
          System.out.print("Halogen: ");

	      if (st.stItem[i].dF > -1.0)
	      {
		    System.out.print(" F: " + st.stItem[i].dF + " mg/l");
		  }

	      if (st.stItem[i].dCL > -1.0)
	      {
		    System.out.print(" Cl: " + st.stItem[i].dCL + " mg/l");
          }

	      if (st.stItem[i].dBR > -1.0)
	      {
		    System.out.print(" Br: " + st.stItem[i].dBR + " mg/l");
		  }

	      if (st.stItem[i].dI > -1.0)
	      {
		    System.out.print(" I: " + st.stItem[i].dI + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.stItem[i].dCO3 > -1.0) ||
		    (st.stItem[i].dHCO3 > -1.0) ||
		    (st.stItem[i].dSO4 > -1.0))
		{
          System.out.print("Carb/Sulf: ");

	      if (st.stItem[i].dCO3 > -1.0)
	      {
		    System.out.print(" CO3: " + st.stItem[i].dCO3 + " mg/l");
		  }

	      if (st.stItem[i].dHCO3 > -1.0)
	      {
		    System.out.print(" HCO3: " + st.stItem[i].dHCO3 + " mg/l");
          }

	      if (st.stItem[i].dSO4 > -1.0)
	      {
		    System.out.print(" SO4: " + st.stItem[i].dSO4 + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.stItem[i].dSOLID > -1.0) ||
		    (st.stItem[i].dTDS > -1.0))
		{
          System.out.print("TDS: ");

	      if (st.stItem[i].dSOLID > -1.0)
	      {
		    System.out.print(" " + st.stItem[i].dSOLID + " mg/l");
		  }

	      if (st.stItem[i].dTDS > -1.0)
	      {
		    System.out.print(" C " + st.stItem[i].dTDS + " mg/l");
          }

          System.out.println(" ");
        }
	  }
	}
  }

  /** Method print()
   * <p> This method will print the brine data
   * @param  st = brine data list structure
   */

  public static void print(brineStruct st)
  {
    if (st != null)
    {
        System.out.println("\n---------------------------------------------");
        System.out.println(st.sName + " " +
                           st.dLatitude + " " +
                           st.dLongitude);

		if ((st.dTOP > -1.0) ||
		    (st.dBASE > -1.0))
		{
          System.out.print("Depth: ");

	      if (st.dTOP > -1.0)
	      {
		    System.out.print("TOP: " + st.dTOP + " ft");
		  }

	      if (st.dBASE > -1.0)
	      {
		    System.out.print(" BASE: " + st.dBASE + " ft");
          }

          System.out.println(" ");
        }

		if ((st.dLI > -1.0) ||
		    (st.dNA > -1.0) ||
		    (st.dK > -1.0)  ||
		    (st.dCS > -1.0))
		{
          System.out.print("Alkali: ");

	      if (st.dLI > -1.0)
	      {
		    System.out.print(" Li: " + st.dLI + " mg/l");
		  }

	      if (st.dNA > -1.0)
	      {
		    System.out.print(" Na: " + st.dNA + " mg/l");
          }

	      if (st.dK > -1.0)
	      {
		    System.out.print(" K: " + st.dK + " mg/l");
		  }

	      if (st.dCS > -1.0)
	      {
		    System.out.print(" Cs: " + st.dCS + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.dMG > -1.0) ||
		    (st.dCA > -1.0) ||
		    (st.dSR > -1.0) ||
		    (st.dBA > -1.0))
		{
          System.out.print("Earth: ");

	      if (st.dMG > -1.0)
	      {
		    System.out.print(" Mg: " + st.dMG + " mg/l");
		  }

	      if (st.dCA > -1.0)
	      {
		    System.out.print(" Ca: " + st.dCA + " mg/l");
          }

	      if (st.dSR > -1.0)
	      {
		    System.out.print(" Sr: " + st.dSR + " mg/l");
		  }

	      if (st.dBA > -1.0)
	      {
		    System.out.print(" Ba: " + st.dBA + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.dF > -1.0) ||
		    (st.dCL > -1.0) ||
		    (st.dBR > -1.0) ||
		    (st.dI > -1.0))
		{
          System.out.print("Halogen: ");

	      if (st.dF > -1.0)
	      {
		    System.out.print(" F: " + st.dF + " mg/l");
		  }

	      if (st.dCL > -1.0)
	      {
		    System.out.print(" Cl: " + st.dCL + " mg/l");
          }

	      if (st.dBR > -1.0)
	      {
		    System.out.print(" Br: " + st.dBR + " mg/l");
		  }

	      if (st.dI > -1.0)
	      {
		    System.out.print(" I: " + st.dI + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.dCO3 > -1.0) ||
		    (st.dHCO3 > -1.0) ||
		    (st.dSO4 > -1.0))
		{
          System.out.print("Carb/Sulf: ");

	      if (st.dCO3 > -1.0)
	      {
		    System.out.print(" CO3: " + st.dCO3 + " mg/l");
		  }

	      if (st.dHCO3 > -1.0)
	      {
		    System.out.print(" HCO3: " + st.dHCO3 + " mg/l");
          }

	      if (st.dSO4 > -1.0)
	      {
		    System.out.print(" SO4: " + st.dSO4 + " mg/l");
          }

          System.out.println(" ");
        }

		if ((st.dSOLID > -1.0) ||
		    (st.dTDS > -1.0))
		{
          System.out.print("TDS: ");

	      if (st.dSOLID > -1.0)
	      {
		    System.out.print(" " + st.dSOLID + " mg/l");
		  }

	      if (st.dTDS > -1.0)
	      {
		    System.out.print(" C " + st.dTDS + " mg/l");
          }

          System.out.println(" ");
	  }
	}
  }
}