/*
 * @brineListStruct.java Version 1.1 06/01/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package brine;

import brine.brineStruct;

/** CLASS brineListStruct
 *  <p> This Class holds the Brine Data List Information.
 *
 *  List composed from,
 *  Common Cations, Anions, Acids, Salts and Hydrate Nomenclature
 *  Missouri University of Science and Technology
 *  http://web.mst.edu/~tbone/subjects/tbone/chem1/Chem%201%20Compound%20Sheet%20FS08.pdf
 *
 *  @version 1.1 06/01/2011
 *  @author  John R. Victorine
 */

public class brineListStruct
{
  public int    iSource = las3.las3Constants._SOURCE_NONE;

  // Stratigraphic Headers Data Information

  public String sKID       = "0";  // Primary Key of the Well
  public String sKEY       = "";   // Primary Key created on user's PC
  public int    iType      = iqstrat.iqstratHeadersStruct.WELL; // Type of Data
  public String sAPI       = "";   // API-Number OR UWI
  public String sName      = "";   // Well Name or Outcrop Name
  public String status     = iqstrat.iqstratStatusStruct.STATUS[0][0]; // Status
  public String sField     = "";   // Field
  public String sCounty    = "";   // County
  public String state      = "";   // State
  public String sLocation  = "";   // Location

  public double dLatitude  = 0.0;  // Latitude of Location
  public double dLongitude = 0.0;  // Longitude of Location

  public double depth      = 0.0;  // Total Depth of Well or Height of Outcrop

  public double dKB        = 0.0;  // Kelly Bushing
  public double dGL        = 0.0;  // Ground Level
  public double dDF        = 0.0;  // Derrick Floor
  public double dTGT       = 0.0;  // TGT

  // LAS 3 Data Structure without the data section

  public String sKGS       = "YES"; // KGS Saved Data Indicator

  public String source     = "";    // Sample Source {S}
  public String sRecovery  = "";    // Recovery Date (Date Core Cut) {D}
  public double depthStart = 0.0;   // Brine Top Depth {F}
  public double depthEnd   = 0.0;   // Brine Base Depth {F}
  public String sFormation = "";    // Primary Formation Cored {S}
  public String sCompany   = "";    // Analyzing Company {S}
  public String sAnalyzed  = "";    // Analysis Date {D}

  // Brine Data

  public int         iCount   = 0;    // Total Number of Records
  public brineStruct stItem[] = null; // Array of Brine Data Structures

  // Brine Data Columns Names

  public int                iColumns    = 0;    // Total Number of Columns
  public brineColumnsStruct stColumns[] = null; // Array of Brine Columns Structures

  // Brine Data Identifiers

  public  int _KEY    = -1;  // KEY

  // AGE
  public  int _FORM   = -1;  // Primary Formation
  public  int _AGE    = -1;  // Age of Formation

  // Depth
  public  int _TOP    = -1;  // Top Depth
  public  int _BASE   = -1;  // Base Depth

  // Other Measured Well Data
  public  int _SPGR   = -1;  // Specific Gravity
  public  int _SIGMA  = -1;  // Specific Conductivity
  public  int _PH     = -1;  // PH
  public  int _DEG    = -1;  // Degree F
  public  int _OHM    = -1;  // Resistivity (Rw)
  public  int _OHM75  = -1; // Resistivity at 75 deg
  public  int _OHME   = -1; // Estimated Rw

  // .. Alkali Metal
  public  int _LI     = -1; // Lithium
  public  int _NA     = -1; // Sodium
  public  int _K      = -1; // Potassium
  public  int _RB     = -1; // Rubidium
  public  int _CS     = -1; // Cesium
  public  int _NA_K   = -1; // Sodium/Potasium

  // ..  Alkali Earth Metal
  public  int _BE     = -1; // Beryllium
  public  int _MG     = -1; // Magnesium
  public  int _CA     = -1; // Calcium
  public  int _SR     = -1; // Strontium
  public  int _BA     = -1; // Barium

  // ..  VIB
  public  int _CR_II  = -1; // Chromium (II)
  public  int _CR_III = -1; // Chromium (iii)

  // ..  VIIB
  public  int _MN_II  = -1; // Manganese (II)
  public  int _MN_III = -1; // Manganese (IV)

  // ..  VIIIB
  public  int _FE_II  = -1; // Iron (II)
  public  int _FE_III = -1; // Iron (III)
  public  int _CO_II  = -1; // Cobalt (II)
  public  int _CO_III = -1; // Cobalt (III)
  public  int _NI_II  = -1; // Nickel (II)
  public  int _NI_III = -1; // Nickel (III)

  // ..  IB
  public  int _CU_I   = -1; // Copper (I)
  public  int _CU_II  = -1; // Copper (II)
  public  int _AG     = -1; // Silver
  public  int _AU     = -1; // Gold
  public  int _AU_III = -1; // Gold (III)

  // ..  IIB
  public  int _ZN     = -1; // Zinc
  public  int _CD     = -1; // Cadmium
  public  int _HG     = -1; // Mercury (I)
  public  int _HG_II  = -1; // Mercury (II)

  // ..  IIIA
  public  int _AL     = -1; // Aluminium

  //..
  public  int _SB_III = -1; // Antimonous(III)
  public  int _SB_V   = -1; // Antimonic(V)
  public  int _BI_III = -1; // Bismuthous(III)
  public  int _BI_V   = -1; // Bismuth(V)


  // ..  IVA
  public  int _SN_II  = -1; // Tin (II)
  public  int _SN_IV  = -1; // Tin (IV)
  public  int _PB_II  = -1; // Lead (II)
  public  int _PB_IV  = -1; // Lead (IV)

  // .. Other
  public  int _NH4    = -1; // Ammonium

  // Common Anions

  // ..Halogen
  public  int _F      = -1; // Flouride
  public  int _CL     = -1; // Chloride
  public  int _BR     = -1; // Bromide
  public  int _I      = -1; // Iodide

  public  int _OH     = -1; // Hydroxide

  public  int _BO3    = -1; // Borate
  public  int _CO3    = -1; // Carbonate
  public  int _HCO3   = -1; // Bicarbonate

  public  int _CLO    = -1; // Hypochlorite
  public  int _CLO2   = -1; // Chlorite
  public  int _CLO3   = -1; // Chlorate
  public  int _CLO4   = -1; // Per chlorate

  public  int _CN     = -1; // Cyanide
  public  int _NCO    = -1; // Cyanate
  public  int _OCN    = -1; // Isocyanate
  public  int _SCN    = -1; // Thicyanite

  public  int _N      = -1; // Nitride
  public  int _N3     = -1; // Azide
  public  int _NO2    = -1; // Nitrite
  public  int _NO3    = -1; // Nitrate

  public  int _CRO4   = -1; // Chromium (II)
  public  int _CR2O7  = -1; // Chromium (III)

  public  int _MNO4   = -1; // Permanganate

  public  int _P      = -1; // Phosphide
  public  int _PO3    = -1; // Phosphite
  public  int _PO4    = -1; // Phosphate
  public  int _HPO4   = -1; // Monohydrogen Phosphate
  public  int _H2PO4  = -1; // Dihydrogen Phosphate

  public  int _AS     = -1; // Arsinide
  public  int _SE     = -1; // Selenide

  public  int _S      = -1; // Sulfide
  public  int _HS     = -1; // Hydrogen Sulfide
  public  int _SO3    = -1; // Sulphite
  public  int _HSO3   = -1; // Hydrogen Sulphite
  public  int _S2O3   = -1; // Thiosulphate
  public  int _SO4    = -1; // Sulfate
  public  int _HSO4   = -1; // Bisulfate

  public  int _SOLID  = -1; // Total Solids
  public  int _TDS    = -1; // Total Solids Computed

  /** Method delete()
   * <p> This method will force set the Objects to null and force Java to
   *     free unwanted memory
   */

  public void delete()
  {
	int i=0;

    sKID      = null;   // Primary KEY of Record
    sKEY      = null;   // User defined primary key

    sAPI       = null;  // API-Number OR UWI
    sName      = null;  // Well Name or Outcrop Name
    status     = null;  // Status

    // LAS 3 Data Structure without the data section

    sKGS       = "YES"; // KGS Saved Data Indicator

    source     = null;  // Sample Source {S}
    sRecovery  = null;  // Recovery Date (Date Core Cut) {D}
    sFormation = null;  // Primary Formation Cored {S}
    sCompany   = null;  // Analyzing Company {S}
    sAnalyzed  = null;  // Analysis Date {D}

    sField     = null;  // Field
    sCounty    = null;  // County
    state      = null;  // State
    sLocation  = null;  // Location

    for (i=0; i<iCount; i++)
    {
      if (stItem[i] != null)
        stItem[i].delete();
      stItem[i] = null;
    }

    iCount = 0;

    // Brine Data Names

    for (i=0; i<iColumns; i++)
    {
      if (stColumns[i] != null)
        stColumns[i].delete();
      stColumns[i] = null;
    }

    iColumns = 0;
  }

  /** Method checkData(int iCurve)
   * <p> This method will return a true if the data is present
   * @param iCurve = the requested curve type
   * @return bValue true the data is present and false it is not there
   */

  public boolean checkData(int iCurve)
  {
    boolean bValue = false;

    switch (iCurve)
    {
      // Brine Data Identifiers

//      case brine.brineStandardTools._KEY:    // KEY
//        if (_KEY > -1) bValue = true;
//        break;

      // AGE
      case brine.brineStandardTools._FORM:   // Primary Formation
        if ( _FORM > -1) bValue = true;
        break;
      case brine.brineStandardTools._AGE:    // Age of Formation
        if ( _AGE > -1) bValue = true;
        break;

      // Depth
      case brine.brineStandardTools._TOP:    // Top Depth
        if ( _TOP > -1) bValue = true;
        break;
      case brine.brineStandardTools._BASE:   // Base Depth
        if ( _BASE > -1) bValue = true;
        break;

      // Other Measured Well Data
      case brine.brineStandardTools._SPGR:   // Specific Gravity
        if ( _SPGR > -1) bValue = true;
        break;
      case brine.brineStandardTools._SIGMA:  // Specific Conductivity
        if ( _SIGMA > -1) bValue = true;
        break;
      case brine.brineStandardTools._PH:     // PH
        if ( _PH > -1) bValue = true;
        break;
      case brine.brineStandardTools._DEG:    // Degree F
        if ( _DEG > -1) bValue = true;
        break;
      case brine.brineStandardTools._OHM:    // Resistivity (Rw)
        if ( _OHM > -1) bValue = true;
        break;
      case brine.brineStandardTools._OHM75:  // Resistivity at 75 deg
        if ( _OHM75 > -1) bValue = true;
        break;
      case brine.brineStandardTools._OHME:   // Estimated Rw
        if ( _OHME > -1) bValue = true;
        break;

      // .. Alkali Metal
      case brine.brineStandardTools._LI:     // Lithium
        if ( _LI > -1) bValue = true;
        break;
      case brine.brineStandardTools._NA:     // Sodium
        if ( _NA > -1) bValue = true;
        break;
      case brine.brineStandardTools._K:      // Potassium
        if ( _K > -1) bValue = true;
        break;
      case brine.brineStandardTools._RB:     // Rubidium
        if ( _RB > -1) bValue = true;
        break;
      case brine.brineStandardTools._CS:     // Cesium
        if ( _CS > -1) bValue = true;
        break;
      case brine.brineStandardTools._NA_K:   // Sodium/Potasium
        if ( _NA_K > -1) bValue = true;
        break;

      // ..  Alkali Earth Metal
      case brine.brineStandardTools._BE:     // Beryllium
        if ( _BE > -1) bValue = true;
        break;
      case brine.brineStandardTools._MG:     // Magnesium
        if ( _MG > -1) bValue = true;
        break;
      case brine.brineStandardTools._CA:     // Calcium
        if ( _CA > -1) bValue = true;
        break;
      case brine.brineStandardTools._SR:     // Strontium
        if ( _SR > -1) bValue = true;
        break;
      case brine.brineStandardTools._BA:     // Barium
        if ( _BA > -1) bValue = true;
        break;

      // ..  VIB
      case brine.brineStandardTools._CR_II:  // Chromium (II)
        if ( _CR_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._CR_III: // Chromium (iii)
        if ( _CR_III > -1) bValue = true;
        break;

      // ..  VIIB
      case brine.brineStandardTools._MN_II:  // Manganese (II)
        if ( _MN_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._MN_III:  // Manganese (III)
        if ( _MN_III > -1) bValue = true;
        break;

      // ..  VIIIB
      case brine.brineStandardTools._FE_II:  // Iron (II)
        if ( _FE_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._FE_III: // Iron (III)
        if ( _FE_III > -1) bValue = true;
        break;
      case brine.brineStandardTools._CO_II:  // Cobalt (II)
        if ( _CO_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._CO_III: // Cobalt (III)
        if ( _CO_III > -1) bValue = true;
        break;
      case brine.brineStandardTools._NI_II:  // Nickel (II)
        if ( _NI_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._NI_III: // Nickel (III)
        if ( _NI_III > -1) bValue = true;
        break;

      // ..  IB
      case brine.brineStandardTools._CU_I:   // Copper (I)
        if ( _CU_I > -1) bValue = true;
        break;
      case brine.brineStandardTools._CU_II:  // Copper (II)
        if ( _CU_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._AG:     // Silver
        if ( _AG > -1) bValue = true;
        break;
      case brine.brineStandardTools._AU:     // Gold
        if ( _AU > -1) bValue = true;
        break;
      case brine.brineStandardTools._AU_III: // Gold (III)
        if ( _AU_III > -1) bValue = true;
        break;

      // ..  IIB
      case brine.brineStandardTools._ZN:     // Zinc
        if ( _ZN > -1) bValue = true;
        break;
      case brine.brineStandardTools._CD:     // Cadmium
        if ( _CD > -1) bValue = true;
        break;
      case brine.brineStandardTools._HG:    // Mercury (I)
        if ( _HG > -1) bValue = true;
        break;
      case brine.brineStandardTools._HG_II:     // Mercury (II)
        if ( _HG_II > -1) bValue = true;
        break;

      // ..  IIIA
      case brine.brineStandardTools._AL:     // Aluminium
        if ( _AL > -1) bValue = true;
        break;

      //..
      case brine.brineStandardTools._SB_III: // Antimonous(III)
        if ( _SB_III > -1) bValue = true;
        break;
      case brine.brineStandardTools._SB_V:   // Antimonic(V)
        if ( _SB_V > -1) bValue = true;
        break;
      case brine.brineStandardTools._BI_III: // Bismuthous(III)
        if ( _BI_III > -1) bValue = true;
        break;
      case brine.brineStandardTools._BI_V:   // Bismuth(V)
        if ( _BI_V > -1) bValue = true;
        break;

      // ..  IVA
      case brine.brineStandardTools._SN_II:  // Tin (II)
        if ( _SN_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._SN_IV:  // Tin (IV)
        if ( _SN_IV > -1) bValue = true;
        break;
      case brine.brineStandardTools._PB_II:  // Lead (II)
        if ( _PB_II > -1) bValue = true;
        break;
      case brine.brineStandardTools._PB_IV:  // Lead (IV)
        if ( _PB_IV > -1) bValue = true;
        break;

      // .. Other
      case brine.brineStandardTools._NH4:    // Ammonium
        if ( _NH4 > -1) bValue = true;
        break;

      // Common Anions

      // ..Halogen
      case brine.brineStandardTools._F:      // Flouride
        if ( _F > -1) bValue = true;
        break;
      case brine.brineStandardTools._CL:     // Chloride
        if ( _CL > -1) bValue = true;
        break;
      case brine.brineStandardTools._BR:     // Bromide
        if ( _BR > -1) bValue = true;
        break;
      case brine.brineStandardTools._I:      // Iodide
        if ( _I > -1) bValue = true;
        break;

      case brine.brineStandardTools._OH:     // Hydroxide
        if ( _OH > -1) bValue = true;
        break;

      case brine.brineStandardTools._BO3:    // Borate
        if ( _BO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._CO3:    // Carbonate
        if ( _CO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._HCO3:   // Bicarbonate
        if ( _HCO3 > -1) bValue = true;
        break;

      case brine.brineStandardTools._CLO:    // Hypochlorite
        if ( _CLO > -1) bValue = true;
        break;
      case brine.brineStandardTools._CLO2:   // Chlorite
        if ( _CLO2 > -1) bValue = true;
        break;
      case brine.brineStandardTools._CLO3:   // Chlorate
        if ( _CLO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._CLO4:   // Per chlorate
        if ( _CLO4 > -1) bValue = true;
        break;

      case brine.brineStandardTools._CN :    // Cyanide
        if ( _CN > -1) bValue = true;
        break;
      case brine.brineStandardTools._NCO:    // Cyanate
        if ( _NCO > -1) bValue = true;
        break;
      case brine.brineStandardTools._OCN:    // Isocyanate
        if ( _OCN > -1) bValue = true;
        break;
      case brine.brineStandardTools._SCN:    // Thicyanite
        if ( _SCN > -1) bValue = true;
        break;

      case brine.brineStandardTools._N:      // Nitride
        if ( _N > -1) bValue = true;
        break;
      case brine.brineStandardTools._N3:     // Azide
        if ( _N3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._NO2:    // Nitrite
        if ( _NO2 > -1) bValue = true;
        break;
      case brine.brineStandardTools._NO3:    // Nitrate
        if ( _NO3 > -1) bValue = true;
        break;

      case brine.brineStandardTools._CRO4:   // Chromium (II)
        if ( _CRO4 > -1) bValue = true;
        break;
      case brine.brineStandardTools._CR2O7:  // Chromium (III)
        if ( _CR2O7 > -1) bValue = true;
        break;

      case brine.brineStandardTools._MNO4:   // Permanganate
        if ( _MNO4 > -1) bValue = true;
        break;

      case brine.brineStandardTools._P:      // Phosphide
        if ( _P > -1) bValue = true;
        break;
      case brine.brineStandardTools._PO3:    // Phosphite
        if ( _PO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._PO4:    // Phosphate
        if ( _PO4 > -1) bValue = true;
        break;
      case brine.brineStandardTools._HPO4:   // Monohydrogen Phosphate
        if ( _HPO4 > -1) bValue = true;
        break;
      case brine.brineStandardTools._H2PO4:  // Dihydrogen Phosphate
        if ( _H2PO4 > -1) bValue = true;
        break;

      case brine.brineStandardTools._AS:     // Arsinide
        if ( _AS > -1) bValue = true;
        break;
      case brine.brineStandardTools._SE:     // Selenide
        if ( _SE > -1) bValue = true;
        break;

      case brine.brineStandardTools._S:      // Sulfide
        if ( _S > -1) bValue = true;
        break;
      case brine.brineStandardTools._HS:     // Hydrogen Sulfide
        if ( _HS > -1) bValue = true;
        break;
      case brine.brineStandardTools._SO3:    // Sulphite
        if ( _SO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._HSO3:   // Hydrogen Sulphite
        if ( _HSO3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._S2O3:   // Thiosulphate
        if ( _S2O3 > -1) bValue = true;
        break;
      case brine.brineStandardTools._SO4:    // Sulfate
        if ( _SO4 > -1) bValue = true;
        break;
      case brine.brineStandardTools._HSO4:   // Bisulfate
        if ( _HSO4 > -1) bValue = true;
        break;

      case brine.brineStandardTools._SOLID:  // Total Solids
        if ( _SOLID > -1) bValue = true;
        break;
      case brine.brineStandardTools._TDS:    // Total Solids Computed
        if ( _TDS > -1) bValue = true;
        break;
    }

    return (bValue);
  }

  /** Method setID(int iCurve)
   * <p> This method will set the anion & cation Identifier
   * @param  iCurve = the requested curve type
   */

  public void setID(int iCurve)
  {
	if (iCount > 0)
	{
	  for (int i=0; i<iCount; i++)
	  {
        switch (iCurve)
        {
          // Depth
          case brine.brineStandardTools._TOP:    // Top Depth
            if (stItem[i].dTOP > 0.0) _TOP = 1;
            break;
          case brine.brineStandardTools._BASE:   // Base Depth
            if (stItem[i].dBASE > 0.0) _BASE = 1;
            break;

          // Other Measured Well Data
          case brine.brineStandardTools._SPGR:   // Specific Gravity
            if (stItem[i].dSPGR > 0.0) _SPGR = 1;
            break;
          case brine.brineStandardTools._SIGMA:  // Specific Conductivity
            if (stItem[i].dSIGMA > 0.0) _SIGMA = 1;
            break;
          case brine.brineStandardTools._PH:     // PH
            if (stItem[i].dPH > 0.0) _PH = 1;
            break;
          case brine.brineStandardTools._DEG:    // Degree F
            if (stItem[i].dDEG > 0.0) _DEG = 1;
            break;
          case brine.brineStandardTools._OHM:    // Resistivity (Rw)
            if (stItem[i].dOHM > 0.0) _OHM = 1;
            break;
          case brine.brineStandardTools._OHM75:  // Resistivity at 75 deg
            if (stItem[i].dOHM75 > 0.0) _OHM75 = 1;
            break;
          case brine.brineStandardTools._OHME:   // Estimated Rw
            if (stItem[i].dOHME > 0.0) _OHME = 1;
            break;

          // .. Alkali Metal
          case brine.brineStandardTools._LI:     // Lithium
            if (stItem[i].dLI > 0.0) _LI = 1;
            break;
          case brine.brineStandardTools._NA:     // Sodium
            if (stItem[i].dNA > 0.0) _NA = 1;
            break;
          case brine.brineStandardTools._K:      // Potassium
            if (stItem[i].dK > 0.0) _K = 1;
            break;
          case brine.brineStandardTools._RB:     // Rubidium
            if (stItem[i].dRB > 0.0) _RB = 1;
            break;
          case brine.brineStandardTools._CS:     // Cesium
            if (stItem[i].dCS > 0.0) _CS = 1;
            break;
          case brine.brineStandardTools._NA_K:   // Sodium/Potasium
            if (stItem[i].dNA_K > 0.0) _NA_K = 1;
            break;

          // ..  Alkali Earth Metal
          case brine.brineStandardTools._BE:     // Beryllium
            if (stItem[i].dBE > 0.0) _BE = 1;
            break;
          case brine.brineStandardTools._MG:     // Magnesium
            if (stItem[i].dMG > 0.0) _MG = 1;
            break;
          case brine.brineStandardTools._CA:     // Calcium
            if (stItem[i].dCA > 0.0) _CA = 1;
            break;
          case brine.brineStandardTools._SR:     // Strontium
            if (stItem[i].dSR > 0.0) _SR = 1;
            break;
          case brine.brineStandardTools._BA:     // Barium
            if (stItem[i].dBA > 0.0) _BA = 1;
            break;

          // ..  VIB
          case brine.brineStandardTools._CR_II:  // Chromium (II)
            if (stItem[i].dCR_II > 0.0) _CR_II = 1;
            break;
          case brine.brineStandardTools._CR_III: // Chromium (iii)
            if (stItem[i].dCR_III > 0.0) _CR_III = 1;
            break;

          // ..  VIIB
          case brine.brineStandardTools._MN_II:  // Manganese (II)
           if (stItem[i].dMN_II > 0.0) _MN_II = 1;
            break;
          case brine.brineStandardTools._MN_III:  // Manganese (III)
            if (stItem[i].dMN_III > 0.0) _MN_III = 1;
            break;

          // ..  VIIIB
          case brine.brineStandardTools._FE_II:  // Iron (II)
            if (stItem[i].dFE_II > 0.0) _FE_II = 1;
            break;
          case brine.brineStandardTools._FE_III: // Iron (III)
            if (stItem[i].dFE_III > 0.0) _FE_III = 1;
            break;
          case brine.brineStandardTools._CO_II:  // Cobalt (II)
            if (stItem[i].dCO_II > 0.0) _CO_II = 1;
            break;
          case brine.brineStandardTools._CO_III: // Cobalt (III)
            if (stItem[i].dCO_III > 0.0) _CO_III = 1;
            break;
          case brine.brineStandardTools._NI_II:  // Nickel (II)
            if (stItem[i].dNI_II > 0.0) _NI_II = 1;
            break;
          case brine.brineStandardTools._NI_III: // Nickel (III)
            if (stItem[i].dNI_III > 0.0) _NI_III = 1;
            break;

          // ..  IB
          case brine.brineStandardTools._CU_I:   // Copper (I)
            if (stItem[i].dCU_I > 0.0) _CU_I = 1;
            break;
          case brine.brineStandardTools._CU_II:  // Copper (II)
            if (stItem[i].dCU_II > 0.0) _CU_II = 1;
            break;
          case brine.brineStandardTools._AG:     // Silver
            if (stItem[i].dAG > 0.0) _AG = 1;
            break;
          case brine.brineStandardTools._AU:     // Gold
            if (stItem[i].dAU > 0.0) _AU = 1;
            break;
          case brine.brineStandardTools._AU_III: // Gold (III)
            if (stItem[i].dAU_III > 0.0) _AU_III = 1;
            break;

          // ..  IIB
          case brine.brineStandardTools._ZN:     // Zinc
            if (stItem[i].dZN > 0.0) _ZN = 1;
            break;
          case brine.brineStandardTools._CD:     // Cadmium
            if (stItem[i].dCD > 0.0) _CD = 1;
            break;
          case brine.brineStandardTools._HG:    // Mercury (I)
            if (stItem[i].dHG > 0.0) _HG = 1;
            break;
          case brine.brineStandardTools._HG_II:     // Mercury (II)
            if (stItem[i].dHG_II > 0.0) _HG_II = 1;
            break;

          // ..  IIIA
          case brine.brineStandardTools._AL:     // Aluminium
            if (stItem[i].dAL > 0.0) _AL = 1;
            break;

          //..
          case brine.brineStandardTools._SB_III: // Antimonous(III)
            if (stItem[i].dSB_III > 0.0) _SB_III = 1;
            break;
          case brine.brineStandardTools._SB_V:   // Antimonic(V)
            if (stItem[i].dSB_V > 0.0) _SB_V = 1;
            break;
          case brine.brineStandardTools._BI_III: // Bismuthous(III)
            if (stItem[i].dBI_III > 0.0) _BI_III = 1;
            break;
          case brine.brineStandardTools._BI_V:   // Bismuth(V)
            if (stItem[i].dBI_V > 0.0) _BI_V = 1;
            break;

          // ..  IVA
          case brine.brineStandardTools._SN_II:  // Tin (II)
            if (stItem[i].dSN_II > 0.0) _SN_II = 1;
            break;
          case brine.brineStandardTools._SN_IV:  // Tin (IV)
            if (stItem[i].dSN_IV > 0.0) _SN_IV = 1;
            break;
          case brine.brineStandardTools._PB_II:  // Lead (II)
            if (stItem[i].dPB_II > 0.0) _PB_II = 1;
            break;
          case brine.brineStandardTools._PB_IV:  // Lead (IV)
            if (stItem[i].dPB_IV > 0.0) _PB_IV = 1;
            break;

          // .. Other
          case brine.brineStandardTools._NH4:    // Ammonium
            if (stItem[i].dNH4 > 0.0) _NH4 = 1;
            break;

          // Common Anions

          // ..Halogen
          case brine.brineStandardTools._F:      // Flouride
            if (stItem[i].dF > 0.0) _F = 1;
            break;
          case brine.brineStandardTools._CL:     // Chloride
            if (stItem[i].dCL > 0.0) _CL = 1;
            break;
          case brine.brineStandardTools._BR:     // Bromide
            if (stItem[i].dBR > 0.0) _BR = 1;
            break;
          case brine.brineStandardTools._I:      // Iodide
            if (stItem[i].dI > 0.0) _I = 1;
            break;

          case brine.brineStandardTools._OH:     // Hydroxide
            if (stItem[i].dOH > 0.0) _OH = 1;
            break;

          case brine.brineStandardTools._BO3:    // Borate
            if (stItem[i].dBO3 > 0.0) _BO3 = 1;
            break;
          case brine.brineStandardTools._CO3:    // Carbonate
            if (stItem[i].dCO3 > 0.0) _CO3 = 1;
            break;
          case brine.brineStandardTools._HCO3:   // Bicarbonate
            if (stItem[i].dHCO3 > 0.0) _HCO3 = 1;
            break;

          case brine.brineStandardTools._CLO:    // Hypochlorite
            if (stItem[i].dCLO > 0.0) _CLO = 1;
            break;
          case brine.brineStandardTools._CLO2:   // Chlorite
            if (stItem[i].dCLO2 > 0.0) _CLO2 = 1;
            break;
          case brine.brineStandardTools._CLO3:   // Chlorate
            if (stItem[i].dCLO3 > 0.0) _CLO3 = 1;
            break;
          case brine.brineStandardTools._CLO4:   // Per chlorate
            if (stItem[i].dCLO4 > 0.0) _CLO4 = 1;
            break;

          case brine.brineStandardTools._CN :    // Cyanide
            if (stItem[i].dCN > 0.0) _CN = 1;
            break;
          case brine.brineStandardTools._NCO:    // Cyanate
            if (stItem[i].dNCO > 0.0) _NCO = 1;
            break;
          case brine.brineStandardTools._OCN:    // Isocyanate
            if (stItem[i].dOCN > 0.0) _OCN = 1;
            break;
          case brine.brineStandardTools._SCN:    // Thicyanite
            if (stItem[i].dSCN > 0.0) _SCN = 1;
            break;

          case brine.brineStandardTools._N:      // Nitride
            if (stItem[i].dN > 0.0) _N = 1;
            break;
          case brine.brineStandardTools._N3:     // Azide
            if (stItem[i].dN3 > 0.0) _N3 = 1;
            break;
          case brine.brineStandardTools._NO2:    // Nitrite
            if (stItem[i].dNO2 > 0.0) _NO2 = 1;
            break;
          case brine.brineStandardTools._NO3:    // Nitrate
            if (stItem[i].dNO3 > 0.0) _NO3 = 1;
            break;

          case brine.brineStandardTools._CRO4:   // Chromium (II)
            if (stItem[i].dCRO4 > 0.0) _CRO4 = 1;
            break;
          case brine.brineStandardTools._CR2O7:  // Chromium (III)
            if (stItem[i].dCR2O7 > 0.0) _CR2O7 = 1;
            break;

          case brine.brineStandardTools._MNO4:   // Permanganate
            if (stItem[i].dMNO4 > 0.0) _MNO4 = 1;
            break;

          case brine.brineStandardTools._P:      // Phosphide
            if (stItem[i].dP > 0.0) _P = 1;
            break;
          case brine.brineStandardTools._PO3:    // Phosphite
            if (stItem[i].dPO3 > 0.0) _PO3 = 1;
            break;
          case brine.brineStandardTools._PO4:    // Phosphate
            if (stItem[i].dPO4 > 0.0) _PO4 = 1;
            break;
          case brine.brineStandardTools._HPO4:   // Monohydrogen Phosphate
            if (stItem[i].dHPO4 > 0.0) _HPO4 = 1;
            break;
          case brine.brineStandardTools._H2PO4:  // Dihydrogen Phosphate
            if (stItem[i].dH2PO4 > 0.0) _H2PO4 = 1;
            break;

          case brine.brineStandardTools._AS:     // Arsinide
            if (stItem[i].dAS > 0.0) _AS = 1;
            break;
          case brine.brineStandardTools._SE:     // Selenide
            if (stItem[i].dSE > 0.0) _SE = 1;
            break;

          case brine.brineStandardTools._S:      // Sulfide
            if (stItem[i].dS > 0.0) _S = 1;
            break;
          case brine.brineStandardTools._HS:     // Hydrogen Sulfide
            if (stItem[i].dHS > 0.0) _HS = 1;
            break;
          case brine.brineStandardTools._SO3:    // Sulphite
            if (stItem[i].dSO3 > 0.0) _SO3 = 1;
            break;
          case brine.brineStandardTools._HSO3:   // Hydrogen Sulphite
            if (stItem[i].dHSO3 > 0.0) _HSO3 = 1;
            break;
          case brine.brineStandardTools._S2O3:   // Thiosulphate
            if (stItem[i].dS2O3 > 0.0) _S2O3 = 1;
            break;
          case brine.brineStandardTools._SO4:    // Sulfate
            if (stItem[i].dSO4 > 0.0) _SO4 = 1;
            break;
          case brine.brineStandardTools._HSO4:   // Bisulfate
            if (stItem[i].dHSO4 > 0.0) _HSO4 = 1;
            break;

          case brine.brineStandardTools._SOLID:  // Total Solids
            if (stItem[i].dSOLID > 0.0) _SOLID = 1;
            break;
          case brine.brineStandardTools._TDS:    // Total Solids Computed
            if (stItem[i].dTDS > 0.0) _TDS = 1;
            break;
		}
  	  }
    }
  }

  /** Method getData(int iCurve)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @return data   = the data requested.
   */

  public double[] getData(int iCurve)
  {
    double data[] = null;

    data = new double[iCount];

    for (int i=0; i<iCount; i++)
    {
      switch (iCurve)
      {
        // Brine Data Identifiers

        // Depth
        case brine.brineStandardTools._TOP:    // Top Depth
          data[i] = stItem[i].dTOP;
          break;
        case brine.brineStandardTools._BASE:   // Base Depth
          data[i] = stItem[i].dBASE;
          break;

        // Other Measured Well Data
        case brine.brineStandardTools._SPGR:   // Specific Gravity
          data[i] = stItem[i].dSPGR;
          break;
        case brine.brineStandardTools._SIGMA:  // Specific Conductivity
          data[i] = stItem[i].dSIGMA;
          break;
        case brine.brineStandardTools._PH:     // PH
          data[i] = stItem[i].dPH;
          break;
        case brine.brineStandardTools._DEG:    // Degree F
          data[i] = stItem[i].dDEG;
          break;
        case brine.brineStandardTools._OHM:    // Resistivity (Rw)
          data[i] = stItem[i].dOHM;
          break;
        case brine.brineStandardTools._OHM75:  // Resistivity at 75 deg
          data[i] = stItem[i].dOHM75;
          break;
        case brine.brineStandardTools._OHME:   // Estimated Rw
          data[i] = stItem[i].dOHME;
          break;

        // .. Alkali Metal
        case brine.brineStandardTools._LI:     // Lithium
          data[i] = stItem[i].dLI;
          break;
        case brine.brineStandardTools._NA:     // Sodium
          data[i] = stItem[i].dNA;
          break;
        case brine.brineStandardTools._K:      // Potassium
          data[i] = stItem[i].dK;
          break;
        case brine.brineStandardTools._RB:     // Rubidium
          data[i] = stItem[i].dRB;
          break;
        case brine.brineStandardTools._CS:     // Cesium
          data[i] = stItem[i].dCS;
          break;
        case brine.brineStandardTools._NA_K:   // Sodium/Potasium
          data[i] = stItem[i].dNA_K;
          break;

        // ..  Alkali Earth Metal
        case brine.brineStandardTools._BE:     // Beryllium
          data[i] = stItem[i].dBE;
          break;
        case brine.brineStandardTools._MG:     // Magnesium
          data[i] = stItem[i].dMG;
          break;
        case brine.brineStandardTools._CA:     // Calcium
          data[i] = stItem[i].dCA;
          break;
        case brine.brineStandardTools._SR:     // Strontium
          data[i] = stItem[i].dSR;
          break;
        case brine.brineStandardTools._BA:     // Barium
          data[i] = stItem[i].dBA;
          break;

        // ..  VIB
        case brine.brineStandardTools._CR_II:  // Chromium (II)
          data[i] = stItem[i].dCR_II;
          break;
        case brine.brineStandardTools._CR_III: // Chromium (iii)
          data[i] = stItem[i].dCR_III;
          break;

        // ..  VIIB
        case brine.brineStandardTools._MN_II:  // Manganese (II)
          data[i] = stItem[i].dMN_II;
          break;
        case brine.brineStandardTools._MN_III:  // Manganese (III)
          data[i] = stItem[i].dMN_III;
          break;

        // ..  VIIIB
        case brine.brineStandardTools._FE_II:  // Iron (II)
          data[i] = stItem[i].dFE_II;
          break;
        case brine.brineStandardTools._FE_III: // Iron (III)
          data[i] = stItem[i].dFE_III;
          break;
        case brine.brineStandardTools._CO_II:  // Cobalt (II)
          data[i] = stItem[i].dCO_II;
          break;
        case brine.brineStandardTools._CO_III: // Cobalt (III)
          data[i] = stItem[i].dCO_III;
          break;
        case brine.brineStandardTools._NI_II:  // Nickel (II)
          data[i] = stItem[i].dNI_II;
          break;
        case brine.brineStandardTools._NI_III: // Nickel (III)
          data[i] = stItem[i].dNI_III;
          break;

        // ..  IB
        case brine.brineStandardTools._CU_I:   // Copper (I)
          data[i] = stItem[i].dCU_I;
          break;
        case brine.brineStandardTools._CU_II:  // Copper (II)
          data[i] = stItem[i].dCU_II;
          break;
        case brine.brineStandardTools._AG:     // Silver
          data[i] = stItem[i].dAG;
          break;
        case brine.brineStandardTools._AU:     // Gold
          data[i] = stItem[i].dAU;
          break;
        case brine.brineStandardTools._AU_III: // Gold
          data[i] = stItem[i].dAU_III;
          break;

        // ..  IIB
        case brine.brineStandardTools._ZN:     // Zinc
          data[i] = stItem[i].dZN;
          break;
        case brine.brineStandardTools._CD:     // Cadmium
          data[i] = stItem[i].dCD;
          break;
        case brine.brineStandardTools._HG:    // Mercury (I)
          data[i] = stItem[i].dHG;
          break;
        case brine.brineStandardTools._HG_II:     // Mercury (II)
          data[i] = stItem[i].dHG_II;
          break;

        // ..  IIIA
        case brine.brineStandardTools._AL:     // Aluminium
          data[i] = stItem[i].dAL;
          break;

        //..
        case brine.brineStandardTools._SB_III: // Antimonous(III)
          data[i] = stItem[i].dSB_III;
          break;
        case brine.brineStandardTools._SB_V:   // Antimonic(V)
          data[i] = stItem[i].dSB_V;
          break;
        case brine.brineStandardTools._BI_III: // Bismuthous(III)
          data[i] = stItem[i].dBI_III;
          break;
        case brine.brineStandardTools._BI_V:   // Bismuth(V)
          data[i] = stItem[i].dBI_V;
          break;

        // ..  IVA
        case brine.brineStandardTools._SN_II:  // Tin (II)
          data[i] = stItem[i].dSN_II;
          break;
        case brine.brineStandardTools._SN_IV:  // Tin (IV)
          data[i] = stItem[i].dSN_IV;
          break;
        case brine.brineStandardTools._PB_II:  // Lead (II)
          data[i] = stItem[i].dPB_II;
          break;
        case brine.brineStandardTools._PB_IV:  // Lead (IV)
          data[i] = stItem[i].dPB_IV;
          break;

        // .. Other
        case brine.brineStandardTools._NH4:    // Ammonium
          data[i] = stItem[i].dNH4;
          break;

        // Common Anions

        // ..Halogen
        case brine.brineStandardTools._F:      // Flouride
          data[i] = stItem[i].dF;
          break;
        case brine.brineStandardTools._CL:     // Chloride
          data[i] = stItem[i].dCL;
          break;
        case brine.brineStandardTools._BR:     // Bromide
          data[i] = stItem[i].dBR;
          break;
        case brine.brineStandardTools._I:      // Iodide
          data[i] = stItem[i].dI;
          break;

        case brine.brineStandardTools._OH:     // Hydroxide
          data[i] = stItem[i].dOH;
          break;

        case brine.brineStandardTools._BO3:    // Borate
          data[i] = stItem[i].dBO3;
          break;
        case brine.brineStandardTools._CO3:    // Carbonate
          data[i] = stItem[i].dCO3;
          break;
        case brine.brineStandardTools._HCO3:   // Bicarbonate
          data[i] = stItem[i].dHCO3;
          break;

        case brine.brineStandardTools._CLO:    // Hypochlorite
          data[i] = stItem[i].dCLO;
          break;
        case brine.brineStandardTools._CLO2:   // Chlorite
          data[i] = stItem[i].dCLO2;
          break;
        case brine.brineStandardTools._CLO3:   // Chlorate
          data[i] = stItem[i].dCLO3;
          break;
        case brine.brineStandardTools._CLO4:   // Per chlorate
          data[i] = stItem[i].dCLO4;
          break;

        case brine.brineStandardTools._CN :    // Cyanide
          data[i] = stItem[i].dCN;
          break;
        case brine.brineStandardTools._NCO:    // Cyanate
          data[i] = stItem[i].dNCO;
          break;
        case brine.brineStandardTools._OCN:    // Isocyanate
          data[i] = stItem[i].dOCN;
          break;
        case brine.brineStandardTools._SCN:    // Thicyanite
          data[i] = stItem[i].dSCN;
          break;

        case brine.brineStandardTools._N:      // Nitride
          data[i] = stItem[i].dN;
          break;
        case brine.brineStandardTools._N3:     // Azide
          data[i] = stItem[i].dN3;
          break;
        case brine.brineStandardTools._NO2:    // Nitrite
          data[i] = stItem[i].dNO2;
          break;
        case brine.brineStandardTools._NO3:    // Nitrate
          data[i] = stItem[i].dNO3;
          break;

        case brine.brineStandardTools._CRO4:   // Chromium (II)
          data[i] = stItem[i].dCRO4;
          break;
        case brine.brineStandardTools._CR2O7:  // Chromium (III)
          data[i] = stItem[i].dCR2O7;
          break;

        case brine.brineStandardTools._MNO4:   // Permanganate
          data[i] = stItem[i].dMNO4;
          break;

        case brine.brineStandardTools._P:      // Phosphide
          data[i] = stItem[i].dP;
          break;
        case brine.brineStandardTools._PO3:    // Phosphite
          data[i] = stItem[i].dPO3;
          break;
        case brine.brineStandardTools._PO4:    // Phosphate
          data[i] = stItem[i].dPO4;
          break;
        case brine.brineStandardTools._HPO4:   // Monohydrogen Phosphate
          data[i] = stItem[i].dHPO4;
          break;
        case brine.brineStandardTools._H2PO4:  // Dihydrogen Phosphate
          data[i] = stItem[i].dH2PO4;
          break;

        case brine.brineStandardTools._AS:     // Arsinide
          data[i] = stItem[i].dAS;
          break;
        case brine.brineStandardTools._SE:     // Selenide
          data[i] = stItem[i].dSE;
          break;

        case brine.brineStandardTools._S:      // Sulfide
          data[i] = stItem[i].dS;
          break;
        case brine.brineStandardTools._HS:     // Hydrogen Sulfide
          data[i] = stItem[i].dHS;
          break;
        case brine.brineStandardTools._SO3:    // Sulphite
          data[i] = stItem[i].dSO3;
          break;
        case brine.brineStandardTools._HSO3:   // Hydrogen Sulphite
          data[i] = stItem[i].dHSO3;
          break;
        case brine.brineStandardTools._S2O3:   // Thiosulphate
          data[i] = stItem[i].dS2O3;
          break;
        case brine.brineStandardTools._SO4:    // Sulfate
          data[i] = stItem[i].dSO4;
          break;
        case brine.brineStandardTools._HSO4:   // Bisulfate
          data[i] = stItem[i].dHSO4;
          break;

        case brine.brineStandardTools._SOLID:  // Total Solids
          data[i] = stItem[i].dSOLID;
          break;
        case brine.brineStandardTools._TDS:    // Total Solids Computed
          data[i] = stItem[i].dTDS;
          break;
      }
    }

    return (data);
  }

  /** Method getData(int iCurve, int iRow)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @param  iRow   = the row
   * @return data   = the data requested.
   */

  public double getData( int iCurve, int iRow )
  {
    double data = 0.0;

    switch (iCurve)
    {
        // Brine Data Identifiers

        // Depth
        case brine.brineStandardTools._TOP:    // Top Depth
          data = stItem[iRow].dTOP;
          break;
        case brine.brineStandardTools._BASE:   // Base Depth
          data = stItem[iRow].dBASE;
          break;

        // Other Measured Well Data
        case brine.brineStandardTools._SPGR:   // Specific Gravity
          data = stItem[iRow].dSPGR;
          break;
        case brine.brineStandardTools._SIGMA:  // Specific Conductivity
          data = stItem[iRow].dSIGMA;
          break;
        case brine.brineStandardTools._PH:     // PH
          data = stItem[iRow].dPH;
          break;
        case brine.brineStandardTools._DEG:    // Degree F
          data = stItem[iRow].dDEG;
          break;
        case brine.brineStandardTools._OHM:    // Resistivity (Rw)
          data = stItem[iRow].dOHM;
          break;
        case brine.brineStandardTools._OHM75:  // Resistivity at 75 deg
          data = stItem[iRow].dOHM75;
          break;
        case brine.brineStandardTools._OHME:   // Estimated Rw
          data = stItem[iRow].dOHME;
          break;

        // .. Alkali Metal
        case brine.brineStandardTools._LI:     // Lithium
          data = stItem[iRow].dLI;
          break;
        case brine.brineStandardTools._NA:     // Sodium
          data = stItem[iRow].dNA;
          break;
        case brine.brineStandardTools._K:      // Potassium
          data = stItem[iRow].dK;
          break;
        case brine.brineStandardTools._RB:     // Rubidium
          data = stItem[iRow].dRB;
          break;
        case brine.brineStandardTools._CS:     // Cesium
          data = stItem[iRow].dCS;
          break;
        case brine.brineStandardTools._NA_K:   // Sodium/Potasium
          data = stItem[iRow].dNA_K;
          break;

        // ..  Alkali Earth Metal
        case brine.brineStandardTools._BE:     // Beryllium
          data = stItem[iRow].dBE;
          break;
        case brine.brineStandardTools._MG:     // Magnesium
          data = stItem[iRow].dMG;
          break;
        case brine.brineStandardTools._CA:     // Calcium
          data = stItem[iRow].dCA;
          break;
        case brine.brineStandardTools._SR:     // Strontium
          data = stItem[iRow].dSR;
          break;
        case brine.brineStandardTools._BA:     // Barium
          data = stItem[iRow].dBA;
          break;

        // ..  VIB
        case brine.brineStandardTools._CR_II:  // Chromium (II)
          data = stItem[iRow].dCR_II;
          break;
        case brine.brineStandardTools._CR_III: // Chromium (iii)
          data = stItem[iRow].dCR_III;
          break;

        // ..  VIIB
        case brine.brineStandardTools._MN_II:  // Manganese (II)
          data = stItem[iRow].dMN_II;
          break;
        case brine.brineStandardTools._MN_III:  // Manganese (III)
          data = stItem[iRow].dMN_III;
          break;

        // ..  VIIIB
        case brine.brineStandardTools._FE_II:  // Iron (II)
          data = stItem[iRow].dFE_II;
          break;
        case brine.brineStandardTools._FE_III: // Iron (III)
          data = stItem[iRow].dFE_III;
          break;
        case brine.brineStandardTools._CO_II:  // Cobalt (II)
          data = stItem[iRow].dCO_II;
          break;
        case brine.brineStandardTools._CO_III: // Cobalt (III)
          data = stItem[iRow].dCO_III;
          break;
        case brine.brineStandardTools._NI_II:  // Nickel (II)
          data = stItem[iRow].dNI_II;
          break;
        case brine.brineStandardTools._NI_III: // Nickel (III)
          data = stItem[iRow].dNI_III;
          break;

        // ..  IB
        case brine.brineStandardTools._CU_I:   // Copper (I)
          data = stItem[iRow].dCU_I;
          break;
        case brine.brineStandardTools._CU_II:  // Copper (II)
          data = stItem[iRow].dCU_II;
          break;
        case brine.brineStandardTools._AG:     // Silver
          data = stItem[iRow].dAG;
          break;
        case brine.brineStandardTools._AU:     // Gold
          data = stItem[iRow].dAU;
          break;
        case brine.brineStandardTools._AU_III: // Gold (III)
          data = stItem[iRow].dAU_III;
          break;

        // ..  IIB
        case brine.brineStandardTools._ZN:     // Zinc
          data = stItem[iRow].dZN;
          break;
        case brine.brineStandardTools._CD:     // Cadmium
          data = stItem[iRow].dCD;
          break;
        case brine.brineStandardTools._HG:    // Mercury (I)
          data = stItem[iRow].dHG;
          break;
        case brine.brineStandardTools._HG_II:     // Mercury (II)
          data = stItem[iRow].dHG_II;
          break;

        // ..  IIIA
        case brine.brineStandardTools._AL:     // Aluminium
          data = stItem[iRow].dAL;
          break;

        //..
        case brine.brineStandardTools._SB_III: // Antimonous(III)
          data = stItem[iRow].dSB_III;
          break;
        case brine.brineStandardTools._SB_V:   // Antimonic(V)
          data = stItem[iRow].dSB_V;
          break;
        case brine.brineStandardTools._BI_III: // Bismuthous(III)
          data = stItem[iRow].dBI_III;
          break;
        case brine.brineStandardTools._BI_V:   // Bismuth(V)
          data = stItem[iRow].dBI_V;
          break;

        // ..  IVA
        case brine.brineStandardTools._SN_II:  // Tin (II)
          data = stItem[iRow].dSN_II;
          break;
        case brine.brineStandardTools._SN_IV:  // Tin (IV)
          data = stItem[iRow].dSN_IV;
          break;
        case brine.brineStandardTools._PB_II:  // Lead (II)
          data = stItem[iRow].dPB_II;
          break;
        case brine.brineStandardTools._PB_IV:  // Lead (IV)
          data = stItem[iRow].dPB_IV;
          break;

        // .. Other
        case brine.brineStandardTools._NH4:    // Ammonium
          data = stItem[iRow].dNH4;
          break;

        // Common Anions

        // ..Halogen
        case brine.brineStandardTools._F:      // Flouride
          data = stItem[iRow].dF;
          break;
        case brine.brineStandardTools._CL:     // Chloride
          data = stItem[iRow].dCL;
          break;
        case brine.brineStandardTools._BR:     // Bromide
          data = stItem[iRow].dBR;
          break;
        case brine.brineStandardTools._I:      // Iodide
          data = stItem[iRow].dI;
          break;

        case brine.brineStandardTools._OH:     // Hydroxide
          data = stItem[iRow].dOH;
          break;

        case brine.brineStandardTools._BO3:    // Borate
          data = stItem[iRow].dBO3;
          break;
        case brine.brineStandardTools._CO3:    // Carbonate
          data = stItem[iRow].dCO3;
          break;
        case brine.brineStandardTools._HCO3:   // Bicarbonate
          data = stItem[iRow].dHCO3;
          break;

        case brine.brineStandardTools._CLO:    // Hypochlorite
          data = stItem[iRow].dCLO;
          break;
        case brine.brineStandardTools._CLO2:   // Chlorite
          data = stItem[iRow].dCLO2;
          break;
        case brine.brineStandardTools._CLO3:   // Chlorate
          data = stItem[iRow].dCLO3;
          break;
        case brine.brineStandardTools._CLO4:   // Per chlorate
          data = stItem[iRow].dCLO4;
          break;

        case brine.brineStandardTools._CN :    // Cyanide
          data = stItem[iRow].dCN;
          break;
        case brine.brineStandardTools._NCO:    // Cyanate
          data = stItem[iRow].dNCO;
          break;
        case brine.brineStandardTools._OCN:    // Isocyanate
          data = stItem[iRow].dOCN;
          break;
        case brine.brineStandardTools._SCN:    // Thicyanite
          data = stItem[iRow].dSCN;
          break;

        case brine.brineStandardTools._N:      // Nitride
          data = stItem[iRow].dN;
          break;
        case brine.brineStandardTools._N3:     // Azide
          data = stItem[iRow].dN3;
          break;
        case brine.brineStandardTools._NO2:    // Nitrite
          data = stItem[iRow].dNO2;
          break;
        case brine.brineStandardTools._NO3:    // Nitrate
          data = stItem[iRow].dNO3;
          break;

        case brine.brineStandardTools._CRO4:   // Chromium (II)
          data = stItem[iRow].dCRO4;
          break;
        case brine.brineStandardTools._CR2O7:  // Chromium (III)
          data = stItem[iRow].dCR2O7;
          break;

        case brine.brineStandardTools._MNO4:   // Permanganate
          data = stItem[iRow].dMNO4;
          break;

        case brine.brineStandardTools._P:      // Phosphide
          data = stItem[iRow].dP;
          break;
        case brine.brineStandardTools._PO3:    // Phosphite
          data = stItem[iRow].dPO3;
          break;
        case brine.brineStandardTools._PO4:    // Phosphate
          data = stItem[iRow].dPO4;
          break;
        case brine.brineStandardTools._HPO4:   // Monohydrogen Phosphate
          data = stItem[iRow].dHPO4;
          break;
        case brine.brineStandardTools._H2PO4:  // Dihydrogen Phosphate
          data = stItem[iRow].dH2PO4;
          break;

        case brine.brineStandardTools._AS:     // Arsinide
          data = stItem[iRow].dAS;
          break;
        case brine.brineStandardTools._SE:     // Selenide
          data = stItem[iRow].dSE;
          break;

        case brine.brineStandardTools._S:      // Sulfide
          data = stItem[iRow].dS;
          break;
        case brine.brineStandardTools._HS:     // Hydrogen Sulfide
          data = stItem[iRow].dHS;
          break;
        case brine.brineStandardTools._SO3:    // Sulphite
          data = stItem[iRow].dSO3;
          break;
        case brine.brineStandardTools._HSO3:   // Hydrogen Sulphite
          data = stItem[iRow].dHSO3;
          break;
        case brine.brineStandardTools._S2O3:   // Thiosulphate
          data = stItem[iRow].dS2O3;
          break;
        case brine.brineStandardTools._SO4:    // Sulfate
          data = stItem[iRow].dSO4;
          break;
        case brine.brineStandardTools._HSO4:   // Bisulfate
          data = stItem[iRow].dHSO4;
          break;

        case brine.brineStandardTools._SOLID:  // Total Solids
          data = stItem[iRow].dSOLID;
          break;
        case brine.brineStandardTools._TDS:    // Total Solids Computed
          data = stItem[iRow].dTDS;
          break;
    }

    return (data);
  }

  /** Method setData(int iCurve, int iRow, double data)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @param  iRow   = the row
   * @return data   = the data requested.
   */

  public void setData( int iCurve, int iRow, double data )
  {
    switch (iCurve)
    {
        // Brine Data Identifiers

        case brine.brineStandardTools._LAT: // Latitude
          stItem[iRow].dLatitude = data;
          break;
        case brine.brineStandardTools._LONG: // Longitude
          stItem[iRow].dLongitude = data;
          break;
        case brine.brineStandardTools._KB: // Kelly Bushing
          stItem[iRow].dKB = data;
          break;
        case brine.brineStandardTools._GL: // Ground Level
          stItem[iRow].dGL = data;
          break;
        case brine.brineStandardTools._DF: // Derrick Floor
          stItem[iRow].dDF = data;
          break;
        case brine.brineStandardTools._TGT: // TGT
          stItem[iRow].dTGT = data;
          break;

        // Depth
        case brine.brineStandardTools._TOP:    // Top Depth
          stItem[iRow].dTOP = data;
          break;
        case brine.brineStandardTools._BASE:   // Base Depth
          stItem[iRow].dBASE = data;
          break;

        // Other Measured Well Data
        case brine.brineStandardTools._SPGR:   // Specific Gravity
          stItem[iRow].dSPGR = data;
          break;
        case brine.brineStandardTools._SIGMA:  // Specific Conductivity
          stItem[iRow].dSIGMA = data;
          break;
        case brine.brineStandardTools._PH:     // PH
          stItem[iRow].dPH = data;
          break;
        case brine.brineStandardTools._DEG:    // Degree F
          stItem[iRow].dDEG = data;
          break;
        case brine.brineStandardTools._OHM:    // Resistivity (Rw)
          stItem[iRow].dOHM = data;
          break;
        case brine.brineStandardTools._OHM75:  // Resistivity at 75 deg
          stItem[iRow].dOHM75 = data;
          break;
        case brine.brineStandardTools._OHME:   // Estimated Rw
          stItem[iRow].dOHME = data;
          break;

        // .. Alkali Metal
        case brine.brineStandardTools._LI:     // Lithium
          stItem[iRow].dLI = data;
          break;
        case brine.brineStandardTools._NA:     // Sodium
          stItem[iRow].dNA = data;
          break;
        case brine.brineStandardTools._K:      // Potassium
          stItem[iRow].dK = data;
          break;
        case brine.brineStandardTools._RB:     // Rubidium
          stItem[iRow].dRB = data;
          break;
        case brine.brineStandardTools._CS:     // Cesium
          stItem[iRow].dCS = data;
          break;
        case brine.brineStandardTools._NA_K:   // Sodium/Potasium
          stItem[iRow].dNA_K = data;
          break;

        // ..  Alkali Earth Metal
        case brine.brineStandardTools._BE:     // Beryllium
          stItem[iRow].dBE = data;
          break;
        case brine.brineStandardTools._MG:     // Magnesium
          stItem[iRow].dMG = data;
          break;
        case brine.brineStandardTools._CA:     // Calcium
          stItem[iRow].dCA = data;
          break;
        case brine.brineStandardTools._SR:     // Strontium
          stItem[iRow].dSR = data;
          break;
        case brine.brineStandardTools._BA:     // Barium
          stItem[iRow].dBA = data;
          break;

        // ..  VIB
        case brine.brineStandardTools._CR_II:  // Chromium (II)
          stItem[iRow].dCR_II = data;
          break;
        case brine.brineStandardTools._CR_III: // Chromium (iii)
          stItem[iRow].dCR_III = data;
          break;

        // ..  VIIB
        case brine.brineStandardTools._MN_II:  // Manganese (II)
          stItem[iRow].dMN_II = data;
          break;
        case brine.brineStandardTools._MN_III:  // Manganese (III)
          stItem[iRow].dMN_III = data;
          break;

        // ..  VIIIB
        case brine.brineStandardTools._FE_II:  // Iron (II)
          stItem[iRow].dFE_II = data;
          break;
        case brine.brineStandardTools._FE_III: // Iron (III)
          stItem[iRow].dFE_III = data;
          break;
        case brine.brineStandardTools._CO_II:  // Cobalt (II)
          stItem[iRow].dCO_II = data;
          break;
        case brine.brineStandardTools._CO_III: // Cobalt (III)
          stItem[iRow].dCO_III = data;
          break;
        case brine.brineStandardTools._NI_II:  // Nickel (II)
          stItem[iRow].dNI_II = data;
          break;
        case brine.brineStandardTools._NI_III: // Nickel (III)
          stItem[iRow].dNI_III = data;
          break;

        // ..  IB
        case brine.brineStandardTools._CU_I:   // Copper (I)
          stItem[iRow].dCU_I = data;
          break;
        case brine.brineStandardTools._CU_II:  // Copper (II)
          stItem[iRow].dCU_II = data;
          break;
        case brine.brineStandardTools._AG:     // Silver
          stItem[iRow].dAG = data;
          break;
        case brine.brineStandardTools._AU:     // Gold
          stItem[iRow].dAU = data;
          break;
        case brine.brineStandardTools._AU_III: // Gold (III)
          stItem[iRow].dAU_III = data;
          break;

        // ..  IIB
        case brine.brineStandardTools._ZN:     // Zinc
          stItem[iRow].dZN = data;
          break;
        case brine.brineStandardTools._CD:     // Cadmium
          stItem[iRow].dCD = data;
          break;
        case brine.brineStandardTools._HG:    // Mercury (I)
          stItem[iRow].dHG = data;
          break;
        case brine.brineStandardTools._HG_II:     // Mercury (II)
          stItem[iRow].dHG_II = data;
          break;

        // ..  IIIA
        case brine.brineStandardTools._AL:     // Aluminium
          stItem[iRow].dAL = data;
          break;

        //..
        case brine.brineStandardTools._SB_III: // Antimonous(III)
          stItem[iRow].dSB_III = data;
          break;
        case brine.brineStandardTools._SB_V:   // Antimonic(V)
          stItem[iRow].dSB_V = data;
          break;
        case brine.brineStandardTools._BI_III: // Bismuthous(III)
          stItem[iRow].dBI_III = data;
          break;
        case brine.brineStandardTools._BI_V:   // Bismuth(V)
          stItem[iRow].dBI_V = data;
          break;

        // ..  IVA
        case brine.brineStandardTools._SN_II:  // Tin (II)
          stItem[iRow].dSN_II = data;
          break;
        case brine.brineStandardTools._SN_IV:  // Tin (IV)
          stItem[iRow].dSN_IV = data;
          break;
        case brine.brineStandardTools._PB_II:  // Lead (II)
          stItem[iRow].dPB_II = data;
          break;
        case brine.brineStandardTools._PB_IV:  // Lead (IV)
          stItem[iRow].dPB_IV = data;
          break;

        // .. Other
        case brine.brineStandardTools._NH4:    // Ammonium
          stItem[iRow].dNH4 = data;
          break;

        // Common Anions

        // ..Halogen
        case brine.brineStandardTools._F:      // Flouride
          stItem[iRow].dF = data;
          break;
        case brine.brineStandardTools._CL:     // Chloride
          stItem[iRow].dCL = data;
          break;
        case brine.brineStandardTools._BR:     // Bromide
          stItem[iRow].dBR = data;
          break;
        case brine.brineStandardTools._I:      // Iodide
          stItem[iRow].dI = data;
          break;

        case brine.brineStandardTools._OH:     // Hydroxide
          stItem[iRow].dOH = data;
          break;

        case brine.brineStandardTools._BO3:    // Borate
          stItem[iRow].dBO3 = data;
          break;
        case brine.brineStandardTools._CO3:    // Carbonate
          stItem[iRow].dCO3 = data;
          break;
        case brine.brineStandardTools._HCO3:   // Bicarbonate
          stItem[iRow].dHCO3 = data;
          break;

        case brine.brineStandardTools._CLO:    // Hypochlorite
          stItem[iRow].dCLO = data;
          break;
        case brine.brineStandardTools._CLO2:   // Chlorite
          stItem[iRow].dCLO2 = data;
          break;
        case brine.brineStandardTools._CLO3:   // Chlorate
          stItem[iRow].dCLO3 = data;
          break;
        case brine.brineStandardTools._CLO4:   // Per chlorate
          stItem[iRow].dCLO4 = data;
          break;

        case brine.brineStandardTools._CN :    // Cyanide
          stItem[iRow].dCN = data;
          break;
        case brine.brineStandardTools._NCO:    // Cyanate
          stItem[iRow].dNCO = data;
          break;
        case brine.brineStandardTools._OCN:    // Isocyanate
          stItem[iRow].dOCN = data;
          break;
        case brine.brineStandardTools._SCN:    // Thicyanite
          stItem[iRow].dSCN = data;
          break;

        case brine.brineStandardTools._N:      // Nitride
          stItem[iRow].dN = data;
          break;
        case brine.brineStandardTools._N3:     // Azide
          stItem[iRow].dN3 = data;
          break;
        case brine.brineStandardTools._NO2:    // Nitrite
          stItem[iRow].dNO2 = data;
          break;
        case brine.brineStandardTools._NO3:    // Nitrate
          stItem[iRow].dNO3 = data;
          break;

        case brine.brineStandardTools._CRO4:   // Chromium (II)
          stItem[iRow].dCRO4 = data;
          break;
        case brine.brineStandardTools._CR2O7:  // Chromium (III)
          stItem[iRow].dCR2O7 = data;
          break;

        case brine.brineStandardTools._MNO4:   // Permanganate
          stItem[iRow].dMNO4 = data;
          break;

        case brine.brineStandardTools._P:      // Phosphide
          stItem[iRow].dP = data;
          break;
        case brine.brineStandardTools._PO3:    // Phosphite
          stItem[iRow].dPO3 = data;
          break;
        case brine.brineStandardTools._PO4:    // Phosphate
          stItem[iRow].dPO4 = data;
          break;
        case brine.brineStandardTools._HPO4:   // Monohydrogen Phosphate
          stItem[iRow].dHPO4 = data;
          break;
        case brine.brineStandardTools._H2PO4:  // Dihydrogen Phosphate
          stItem[iRow].dH2PO4 = data;
          break;

        case brine.brineStandardTools._AS:     // Arsinide
          stItem[iRow].dAS = data;
          break;
        case brine.brineStandardTools._SE:     // Selenide
          stItem[iRow].dSE = data;
          break;

        case brine.brineStandardTools._S:      // Sulfide
          stItem[iRow].dS = data;
          break;
        case brine.brineStandardTools._HS:     // Hydrogen Sulfide
          stItem[iRow].dHS = data;
          break;
        case brine.brineStandardTools._SO3:    // Sulphite
          stItem[iRow].dSO3 = data;
          break;
        case brine.brineStandardTools._HSO3:   // Hydrogen Sulphite
          stItem[iRow].dHSO3 = data;
          break;
        case brine.brineStandardTools._S2O3:   // Thiosulphate
          stItem[iRow].dS2O3 = data;
          break;
        case brine.brineStandardTools._SO4:    // Sulfate
          stItem[iRow].dSO4 = data;
          break;
        case brine.brineStandardTools._HSO4:   // Bisulfate
          stItem[iRow].dHSO4 = data;
          break;

        case brine.brineStandardTools._SOLID:  // Total Solids
          stItem[iRow].dSOLID = data;
          break;
        case brine.brineStandardTools._TDS:    // Total Solids Computed
          stItem[iRow].dTDS = data;
          break;
    }
  }

  /** Method setData(int iCurve, int iRow, String sData)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @param  iRow   = the row
   * @return sData  = the data requested.
   */

  public void setData( int iCurve, int iRow, String sData )
  {
    switch (iCurve)
    {
        // Brine Data Identifiers

        // Depth
        case brine.brineStandardTools._FORM:  // Primary Formation
          stItem[iRow].sFORM = new String(sData);
          break;
        case brine.brineStandardTools._AGE:  // Age of Formation
          stItem[iRow].sAGE = new String(sData);
          break;
        case brine.brineStandardTools._NAME: // Well Name
          stItem[iRow].sName = new String(sData);
          break;
        case brine.brineStandardTools._API: // API-Number
          stItem[iRow].sAPI = new String(sData);
          break;
        case brine.brineStandardTools._LOC: // Location
          stItem[iRow].sLocation = new String(sData);
          break;
        case brine.brineStandardTools._FIELD: // Field
          stItem[iRow].sField = new String(sData);
          break;
        case brine.brineStandardTools._COUNTY: // County
          stItem[iRow].sCounty = new String(sData);
          break;
        case brine.brineStandardTools._STATE: // State
          stItem[iRow].state = new String(sData);
          break;
        case brine.brineStandardTools._SRC: // Source
          stItem[iRow].source = new String(sData);
          break;
        case brine.brineStandardTools._DATE: // Date
          stItem[iRow].sRecovery = new String(sData);
          break;
	}
  }
}