/*
 * @Profile.java Version 1.1 02/15/2012
 *
 * Copyright (c) 2012 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

import java.net.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import javax.swing.*;

import cmn.cmnStruct;
import iqstrat.iqstratStruct;

import iqstrat.gui.iqstratStartAppPanel;

import horizon.strat.stratListStruct;
import horizon.strat.io.ReadStratUnitsXMLFile;

import horizon.bio.bioStratListStruct;
import horizon.bio.io.ReadBioStratUnitsXMLFile;

import lith.lithology.lithologySymbolsStruct;
import lith.lithology.io.ReadLithologySymbolsXMLFile;

import mask.maskSymbolsListStruct;
import mask.io.ReadMaskSymbolsXMLFile;

import parse.io.ReadParseMineralsXMLFile;
import parse.io.ReadParseTextureXMLFile;
import parse.io.ReadParseSedStructXMLFile;
import parse.io.ReadParseFossilXMLFile;
import parse.io.ReadParseColorXMLFile;
import parse.io.ReadParsePHIXMLFile;
import parse.io.ReadParseDepEnvXMLFile;
import parse.io.ReadRockColumnSymbolsXMLFile;
import parse.io.ReadGrainsTexturesXMLFile;

import horizon.env.io.ReadDepEnvXMLFile;

import rock.color.rockColorChartListStruct;
import rock.color.io.ReadColorChartXML;

/** Class Profile
 *  <p> This Class is the starting point for the Profile Web App.
 *
 *  @version 1.1 02/15/2012
 *  @author  John R. Victorine
 */

public class Profile extends Applet
{
  // Runtime object

  private Runtime r = Runtime.getRuntime();

  // Web Application is runing Stand Alone Indicator

  private boolean isStandalone = false;

  private String  sKID            = new String("0"); // LAS KID
  private String  sKID_Standalone = new String("0");

  // LAS Viewer Global Variables

  private iqstratStruct stStruct = new iqstratStruct();
  private cmnStruct     stCMN    = new cmnStruct();
  private String        sPath    = new String(".");

  private String        sGEMINI  = new String("");

  // Array of LAS 3.0 File Data to read into program

  private boolean       bLAS[]   = { true,   // Version Information ( NO Data Sections)
                                     true,   // Well Information    ( NO Data Sections)
                                     true,   // Log Data Section
                                     true,   // Core Data Section
                                     false,  // Drilling Data Section
                                     false,  // Inclinometry Data Section
                                     true,   // Tops Data Section
                                     false,  // Test Data Section
                                     true,   // Perforation Data Section
                                     true,   // Control Data Section
                                     true,   // Selected Log Curves  ( NO Data Sections)
                                     true,   // Selected Core Curves ( NO Data Sections)
                                     true,   // Sequence Stratigraphy
                                     false,  // PfEFFER Flow Units
                                     false,  // PfEFFER Table Data ( READ ONLY )
                                     true,   // Geologist report Data
                                     true,   // Gamma API Shale Limits
                                     true,   // Rock Images Data
                                     true,   // Brine Data Data
                                     true,   // Bio Stratigraphy Data
                                     true};  // Depositional Environment Data

  /** getParameter()
   *  <p> This method wil get a parameter value
   *  @param key = The Name of the Parameter
   *  @param def = The Object Type of the Parameter
   *  @return Parameter
   */

  public String getParameter(String key, String def)
  {
    return (isStandalone ? System.getProperty(key, def) :
            (getParameter(key) != null ? getParameter(key) : def));
  }

  /** Constructor Profile()
   *  <p> This is the Constructor for this class.
   */

  public Profile() {  }

  /** Method init()
   *  <p> This Method will retrieve the data values from the HTML or the main()
   *      arguments and run the Stratigraphic Data Collection and Biostratigraphy Profile
   *      Viewer Module.
   */

  public void init()
  {
    try
    {
      sKID = this.getParameter("sKID", sKID_Standalone);
      if (sKID.equals("undefined"))
        sKID = new String("0");
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }

    try
    {
      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Method jbInit()
   *  <p> This method will launch the Field LAS Catalog.
   * @throws Exception
   */

  private void jbInit() throws Exception
  {
    iqstratStartAppPanel panel = null;

    // initialize Global LAS Data Structure

    stCMN.bStandalone   = isStandalone;
    stStruct.depthStart = 0.0;
    stStruct.depthEnd   = 2000.0;

    // Retrieve KGS Image Icon for printing

    if (!isStandalone)
    {
      stCMN.urlDirCodeBase = getCodeBase();
      stCMN.appletContext  = getAppletContext();
      stCMN.pIcon          = getImage(getCodeBase(), cmn.cmnStruct.ICON);
      stCMN.pIcon2         = getImage(getCodeBase(), cmn.cmnStruct.ICON2);
    }
    else
    {
      URL url = null;

      stCMN.urlDirCodeBase =  getClass().getClassLoader().getResource(".");

      ClassLoader cl = this.getClass().getClassLoader();
      sPath = cl.getResource("images").toString();
      stCMN.sImagePath = new String(sPath);

      try
      {
        url = new URL(sPath+"\\kgsSm.png");
        stCMN.pIcon = Toolkit.getDefaultToolkit().getImage(url);
        url = new URL(sPath+"\\kgsSm2.png");
        stCMN.pIcon2 = Toolkit.getDefaultToolkit().getImage(url);
      }
      catch (java.net.MalformedURLException e)
      {
        e.printStackTrace();
        String sMessage = new String("Profile(): \n" +
                              e.getMessage() + "\n" +
                              "Couldn't create image: " + sPath+"\\kgsSm.gif");
        JOptionPane.showMessageDialog((Component) null,
                                      sMessage,
                                      "ERROR",
                                      JOptionPane.ERROR_MESSAGE);
      }
    }

    // Get the Path of the XML Files stored in JAR File.

    ClassLoader cl = this.getClass().getClassLoader();
    sPath = cl.getResource("data").toString();
    stCMN.sPath = new String(sPath);

    sPath = cl.getResource("images").toString();
    stCMN.sImagePath = new String(sPath);
    stCMN.sGEMINI    = new String(sGEMINI);
    stStruct.bLAS    = bLAS;

    stCMN.cl = cl;

    stStruct.stCMN = stCMN;

    // Read in Lookup Data

    readICSUnits();                // Read in the ISC Stratigraphic Units
    readKGSUnits();                // Read Kansas Stratigraphic Units

    readDepEnvironment();          // Read the Depositional Environment Thesaurus

    readLithologySymbols();        // Read the Lithology Symbols Data
    readParseMinerals();           // Read the Common Minerals Data

    readPorositySymbols();         // Read the Porosity Types Symbols Data
    readFossilSymbols();           // Read the Fossil Symbols Data
    readSedStructureSymbols();     // Read the Sedimentary Structure Symbols Data

    readParseTextureSymbols();     // Read the Texture Thesaurus XML File
    readParsePorosityThesaurus();  // Read the Porosity Types Thesaurus XML File
    readParseFossilsThesaurus();   // Read the Fossil Thesaurus XML FIle
    readParseSedStructSymbols();   // Read the Sedimentary Structure Thesaurus
    readParseColors();             // Read the Colors XML Thesaurus File

    readRockColumn();              // Read Rock Column Lithology XML File
    readParseGrainsTextures();     // Read Rock Column Texture/Grain XML File

    readBioStratigraphicUnits();   // Read Fossil by their Scientific Names
    readKSBioStratigraphicUnits(); // Read Kansas Fossils by their Scientific Names

    readRockColorChart();          // Read Rock Color Chart
    readSoilColorChart();          // Read Soil Color Chart

    // Display the LAS Viewer Start

    if (!isStandalone)
    {
      panel = new iqstratStartAppPanel( null, stStruct );

      this.setLayout(new BorderLayout());
      this.add(panel, BorderLayout.CENTER);
    }
    else
    {
      JFrame frame = new JFrame();

      frame.setTitle("SELECT");
      panel = new iqstratStartAppPanel( frame, stStruct );
      frame.getContentPane().add(panel, null);

      // Set the Size and Display the Frame

      frame.setSize(300, 350);
      Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
      frame.setLocation((d.width - frame.getSize().width)/2,
                        (d.height - frame.getSize().height)/2);

      frame.setResizable(false);
      frame.setVisible(true);
    }
  }

  /* ======================================================================= *
   * ------------------------ READ DATA XML METHODS ------------------------ *
   * ======================================================================= */

  /** Method readICSUnits()
   * <p> This method will read in the International Committee on Stratigraphy
   *     Stratigraphic Units.
   */

  private void readICSUnits()
  {
    ReadStratUnitsXMLFile pData =
        new ReadStratUnitsXMLFile(horizon.strat.io.ReadStratUnitsXMLFile.URL);
    stStruct.stICS = pData.Process( stStruct.stCMN.sPath + "/ISC.xml" );

//horizon.strat.stratUtility.print(stStruct.stICS);
  }

  /** Method readKGSUnits()
   * <p> This method will read in the KGS Stratigraphic Units.
   */

  private void readKGSUnits()
  {
    ReadStratUnitsXMLFile pData =
        new ReadStratUnitsXMLFile(horizon.strat.io.ReadStratUnitsXMLFile.URL);
    stStruct.stKGS = pData.Process( stStruct.stCMN.sPath + "/kansas.xml" );
    stStruct.stKGS = horizon.strat.stratUtility.addICSArray( stStruct.stKGS,
                                                             stStruct.stICS );
//horizon.strat.stratUtility.print(stStruct.stKGS);
  }

  /** Method readLithologySymbols()
   * <p> This method will read in the Lithology Plot Symbols
   */

  private void readLithologySymbols()
  {
    ReadLithologySymbolsXMLFile pData =
        new ReadLithologySymbolsXMLFile(
          lith.lithology.io.ReadLithologySymbolsXMLFile.URL);

    stStruct.stSymbols = pData.Process( stStruct.stCMN.sPath + "/lithology.xml" );
//lith.lithology.lithologySymbolsUtility.print(stStruct.stSymbols);
  }

  /** Method readParseMinerals()
   * <p> This method will read in the Common Minerals Data
   */

  private void readParseMinerals()
  {
    ReadParseMineralsXMLFile pData =
        new ReadParseMineralsXMLFile( parse.io.ReadParseMineralsXMLFile.URL );

    stStruct.stMinerals = pData.Process( stStruct.stCMN.sPath + "/minerals.xml" );

//parse.parseMineralsUtility.print( stStruct.stMinerals );
  }

  /** Method readParseTextureSymbols()
   * <p> This method will read in the Texture Type Plot Symbols
   */

  private void readParseTextureSymbols()
  {
    ReadParseTextureXMLFile pData =
        new ReadParseTextureXMLFile( parse.io.ReadParseTextureXMLFile.URL );

    stStruct.stParseTexture = pData.Process( stStruct.stCMN.sPath + "/textures.xml" );

    stStruct.stParseTexture = parse.parseTextureUtility.setRhomaaUmaa(
        stStruct.stMinerals, stStruct.stParseTexture );
//parse.parseTextureUtility.print(stStruct.stParseTexture);
  }

  /** Method readDepEnvironment()
   * <p> This method will read in the Depositional Environment Thesaurus
   */

  private void readDepEnvironment()
  {
    ReadDepEnvXMLFile pData =
        new ReadDepEnvXMLFile( horizon.env.io.ReadDepEnvXMLFile.URL );

    stStruct.stDepEnv =
        pData.Process( stStruct.stCMN.sPath + "/kansas_env.xml" );

//horizon.env.envUtility.print( stStruct.stDepEnv );
  }

  /** Method readPorositySymbols()
   * <p> This method will read in the Porosity Type Plot Symbols
   */

  private void readPorositySymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stPorosity = pData.Process( stStruct.stCMN.sPath + "/porosity_types.xml" );

//mask.maskSymbolsUtility.print(stStruct.stPorosity);
  }

  /** Method readParsePorosityThesaurus()
   * <p> This method will read in the Porosity Types Thesaurus
   */

  private void readParsePorosityThesaurus()
  {
    ReadParsePHIXMLFile pData =
        new ReadParsePHIXMLFile( parse.io.ReadParsePHIXMLFile.URL );

    stStruct.stParsePorosity =
        pData.Process( stStruct.stCMN.sPath + "/porosity.xml" );

//parse.parsePHIUtility.print( stStruct.stParsePorosity );
  }

  /** Method readFossilSymbols()
   * <p> This method will read in the Fossil Plot Symbols
   */

  private void readFossilSymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stFossils = pData.Process( stStruct.stCMN.sPath + "/fossils.xml" );
//mask.maskSymbolsUtility.print(stStruct.stFossils);
  }

  /** Method readParseFossilsThesaurus()
   * <p> This method will read in the Fossil Thesaurus
   */

  private void readParseFossilsThesaurus()
  {
    ReadParseFossilXMLFile pData =
        new ReadParseFossilXMLFile( parse.io.ReadParseFossilXMLFile.URL );

    stStruct.stParseFossils =
        pData.Process( stStruct.stCMN.sPath + "/fossils_th.xml" );

//parse.parseFossilUtility.print( stStruct.stParseFossils );
  }

  /** Method readSedStructureSymbols()
   * <p> This method will read in the Sedimentary Structure Plot Symbols
   */

  private void readSedStructureSymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stSedStruct = pData.Process( stStruct.stCMN.sPath + "/sed_structure.xml" );
    stStruct.stSedStruct = mask.maskSymbolsUtility.setHierarchy(
        mask.maskSymbolsListStruct._SYMBOLS, stStruct.stSedStruct );

//mask.maskSymbolsUtility.print(stStruct.stSedStruct);
  }

  /** Method readParseSedStructSymbols()
   * <p> This method will read in the Texture Type Plot Symbols
   */

  private void readParseSedStructSymbols()
  {
    ReadParseSedStructXMLFile pData =
        new ReadParseSedStructXMLFile( parse.io.ReadParseSedStructXMLFile.URL );

    stStruct.stParseSedStruct =
        pData.Process( stStruct.stCMN.sPath + "/sed_structures.xml" );

//parse.parseSedStructUtility.print(stStruct.stParseSedStruct);
  }

  /** Method readBioStratigraphicUnits()
   * <p> This method will read in the bio-stratigraphy fossil names
   */

  private void readBioStratigraphicUnits()
  {
    ReadBioStratUnitsXMLFile pData =
        new ReadBioStratUnitsXMLFile(
            horizon.bio.io.ReadBioStratUnitsXMLFile.URL );

    stStruct.stBio = pData.Process( stStruct.stCMN.sPath + "/fossils_bio.xml",
                                    stStruct.stParseFossils,
                                    stStruct.stICS );

//horizon.bio.bioStratUtility.print( stStruct.stBio );
  }

  /** Method readKSBioStratigraphicUnits()
   * <p> This method will read in the Kansas bio-stratigraphy fossil names
   */

  private void readKSBioStratigraphicUnits()
  {
    ReadBioStratUnitsXMLFile pData =
        new ReadBioStratUnitsXMLFile(
            horizon.bio.io.ReadBioStratUnitsXMLFile.URL );

    stStruct.stKSBio = pData.Process( stStruct.stCMN.sPath + "/kansas_bio.xml",
                                      stStruct.stParseFossils,
                                      stStruct.stICS );

//horizon.bio.bioStratUtility.print( stStruct.stKSBio );
  }

  /** Method readParseColors()
   * <p> This method will read in the Colors
   */

  private void readParseColors()
  {
    ReadParseColorXMLFile pData =
        new ReadParseColorXMLFile( parse.io.ReadParseColorXMLFile.URL );

    stStruct.stParseColors =
        pData.Process( stStruct.stCMN.sPath + "/colors.xml" );

//parse.parseColorUtility.print( stStruct.stParseColors );
  }

  /** Method readRockColorChart()
   * <p> This method will read in the rock color chart
   */

  private void readRockColorChart()
  {
    ReadColorChartXML pData =
        new ReadColorChartXML(
            rock.color.io.ReadColorChartXML.URL );

    stStruct.stRockColor =
        pData.Process( stStruct.stCMN.sPath + "/rock_color_catalog.xml" );

//rock.color.rockColorChartUtility.print( stStruct.stRockColor );
  }

  /** Method readSoilColorChart()
   * <p> This method will read in the soil color chart
   */

  private void readSoilColorChart()
  {
    ReadColorChartXML pData =
        new ReadColorChartXML(
            rock.color.io.ReadColorChartXML.URL );

    stStruct.stSoilColor =
        pData.Process( stStruct.stCMN.sPath + "/soil_color_catalog.xml" );

//rock.color.rockColorChartUtility.print( stStruct.stSoilColor );
  }

  /** Method readRockColumn()
   * <p> This method will read in the Rock Column Lithology Data
   */

  private void readRockColumn()
  {
    ReadRockColumnSymbolsXMLFile pData =
        new ReadRockColumnSymbolsXMLFile(
            parse.io.ReadRockColumnSymbolsXMLFile.URL );

    stStruct.stParseRock =
        pData.Process( stStruct.stCMN.sPath + "/rock_lithology.xml" );

//parse.parseRockColumnSymbolsUtility.print( stStruct.stParseRock );
  }

  /** Method readParseGrainsTextures()
   * <p> This method will read in the Grain & Texture Thesaurus
   */

  private void readParseGrainsTextures()
  {
    ReadGrainsTexturesXMLFile pData =
        new ReadGrainsTexturesXMLFile( parse.io.ReadGrainsTexturesXMLFile.URL );

    stStruct.stParseGrain =
        pData.Process( stStruct.stCMN.sPath + "/grains.xml");
//parse.parseGrainsTextureUtility.print( stStruct.stParseGrain );
  }

  /** Method readICSUnits()
   * <p> This method will read in the International Committee on Stratigraphy
   *     Stratigraphic Units.
   *

  private void readICSUnits()
  {
    ReadStratUnitsXMLFile pData =
        new ReadStratUnitsXMLFile(horizon.strat.io.ReadStratUnitsXMLFile.URL);
    stStruct.stICS = pData.Process( cmn.cmnStruct.ISC_XML );

//horizon.strat.stratUtility.print(stStruct.stICS);
  }
*/
  /** Method readKGSUnits()
   * <p> This method will read in the KGS Stratigraphic Units.
   *

  private void readKGSUnits()
  {
    ReadStratUnitsXMLFile pData =
        new ReadStratUnitsXMLFile(horizon.strat.io.ReadStratUnitsXMLFile.URL);
    stStruct.stKGS = pData.Process( cmn.cmnStruct.KANSAS_XML );
    stStruct.stKGS = horizon.strat.stratUtility.addICSArray( stStruct.stKGS,
                                                             stStruct.stICS );
//horizon.strat.stratUtility.print(stStruct.stKGS);
  }
*/
  /** Method readLithologySymbols()
   * <p> This method will read in the Lithology Plot Symbols
   *

  private void readLithologySymbols()
  {
    ReadLithologySymbolsXMLFile pData =
        new ReadLithologySymbolsXMLFile(
          lith.lithology.io.ReadLithologySymbolsXMLFile.URL);

    stStruct.stSymbols = pData.Process( cmn.cmnStruct.LITHOLOGY_XML );
//lith.lithology.lithologySymbolsUtility.print(stStruct.stSymbols);
  }
*/
  /** Method readParseMinerals()
   * <p> This method will read in the Common Minerals Data
   *

  private void readParseMinerals()
  {
    ReadParseMineralsXMLFile pData =
        new ReadParseMineralsXMLFile( parse.io.ReadParseMineralsXMLFile.URL );

    stStruct.stMinerals = pData.Process( cmn.cmnStruct.MINERALS_XML );

//parse.parseMineralsUtility.print( stStruct.stMinerals );
  }
*/
  /** Method readParseTextureSymbols()
   * <p> This method will read in the Texture Type Plot Symbols
   *

  private void readParseTextureSymbols()
  {
    ReadParseTextureXMLFile pData =
        new ReadParseTextureXMLFile( parse.io.ReadParseTextureXMLFile.URL );

    stStruct.stParseTexture = pData.Process( cmn.cmnStruct.TEXTURES_XML );

    stStruct.stParseTexture = parse.parseTextureUtility.setRhomaaUmaa(
        stStruct.stMinerals, stStruct.stParseTexture );
//parse.parseTextureUtility.print(stStruct.stParseTexture);
  }
*/
  /** Method readDepEnvironment()
   * <p> This method will read in the Depositional Environment Thesaurus
   *

  private void readDepEnvironment()
  {
    ReadDepEnvXMLFile pData =
        new ReadDepEnvXMLFile( horizon.env.io.ReadDepEnvXMLFile.URL );

    stStruct.stDepEnv =
        pData.Process( cmn.cmnStruct.KANSAS_ENV_XML);

//horizon.env.envUtility.print( stStruct.stDepEnv );
  }
*/
  /** Method readPorositySymbols()
   * <p> This method will read in the Porosity Type Plot Symbols
   *

  private void readPorositySymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stPorosity = pData.Process( cmn.cmnStruct.POROSITY_TYPES_XML );

//mask.maskSymbolsUtility.print(stStruct.stPorosity);
  }
*/
  /** Method readParsePorosityThesaurus()
   * <p> This method will read in the Porosity Types Thesaurus
   *

  private void readParsePorosityThesaurus()
  {
    ReadParsePHIXMLFile pData =
        new ReadParsePHIXMLFile( parse.io.ReadParsePHIXMLFile.URL );

    stStruct.stParsePorosity =
        pData.Process( cmn.cmnStruct.POROSITY_XML );

//parse.parsePHIUtility.print( stStruct.stParsePorosity );
  }
*/
  /** Method readFossilSymbols()
   * <p> This method will read in the Fossil Plot Symbols
   *

  private void readFossilSymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stFossils = pData.Process( cmn.cmnStruct.FOSSILS_XML );
//mask.maskSymbolsUtility.print(stStruct.stFossils);
  }
*/
  /** Method readParseFossilsThesaurus()
   * <p> This method will read in the Fossil Thesaurus
   *

  private void readParseFossilsThesaurus()
  {
    ReadParseFossilXMLFile pData =
        new ReadParseFossilXMLFile( parse.io.ReadParseFossilXMLFile.URL );

    stStruct.stParseFossils =
        pData.Process( cmn.cmnStruct.FOSSILS_TH_XML );

//parse.parseFossilUtility.print( stStruct.stParseFossils );
  }
*/
  /** Method readSedStructureSymbols()
   * <p> This method will read in the Sedimentary Structure Plot Symbols
   *

  private void readSedStructureSymbols()
  {
    ReadMaskSymbolsXMLFile pData =
        new ReadMaskSymbolsXMLFile( mask.io.ReadMaskSymbolsXMLFile.URL );

    stStruct.stSedStruct = pData.Process( cmn.cmnStruct.SED_STRUCTURE_XML );
    stStruct.stSedStruct = mask.maskSymbolsUtility.setHierarchy(
        mask.maskSymbolsListStruct._SYMBOLS, stStruct.stSedStruct );

//mask.maskSymbolsUtility.print(stStruct.stSedStruct);
  }
*/
  /** Method readParseSedStructSymbols()
   * <p> This method will read in the Texture Type Plot Symbols
   *

  private void readParseSedStructSymbols()
  {
    ReadParseSedStructXMLFile pData =
        new ReadParseSedStructXMLFile( parse.io.ReadParseSedStructXMLFile.URL );

    stStruct.stParseSedStruct =
        pData.Process( cmn.cmnStruct.SED_STRUCTURES_XML );

//parse.parseSedStructUtility.print(stStruct.stParseSedStruct);
  }
*/
  /** Method readBioStratigraphicUnits()
   * <p> This method will read in the bio-stratigraphy fossil names
   *

  private void readBioStratigraphicUnits()
  {
    ReadBioStratUnitsXMLFile pData =
        new ReadBioStratUnitsXMLFile(
            horizon.bio.io.ReadBioStratUnitsXMLFile.URL );

    stStruct.stBio = pData.Process( cmn.cmnStruct.FOSSILS_BIO_XML,
                                    stStruct.stParseFossils,
                                    stStruct.stICS );

//horizon.bio.bioStratUtility.print( stStruct.stBio );
  }
*/
  /** Method readKSBioStratigraphicUnits()
   * <p> This method will read in the Kansas bio-stratigraphy fossil names
   *

  private void readKSBioStratigraphicUnits()
  {
    ReadBioStratUnitsXMLFile pData =
        new ReadBioStratUnitsXMLFile(
            horizon.bio.io.ReadBioStratUnitsXMLFile.URL );

    stStruct.stKSBio = pData.Process( cmn.cmnStruct.KANSAS_BIO_XML,
                                      stStruct.stParseFossils,
                                      stStruct.stICS );

//horizon.bio.bioStratUtility.print( stStruct.stKSBio );
  }
*/
  /** Method readParseColors()
   * <p> This method will read in the Colors
   *

  private void readParseColors()
  {
    ReadParseColorXMLFile pData =
        new ReadParseColorXMLFile( parse.io.ReadParseColorXMLFile.URL );

    stStruct.stParseColors =
        pData.Process( cmn.cmnStruct.COLORS_XML );

//parse.parseColorUtility.print( stStruct.stParseColors );
  }
*/
  /** Method readRockColorChart()
   * <p> This method will read in the rock color chart
   *

  private void readRockColorChart()
  {
    ReadColorChartXML pData =
        new ReadColorChartXML(
            rock.color.io.ReadColorChartXML.URL );

    stStruct.stRockColor =
        pData.Process( cmn.cmnStruct.ROCK_COLOR_CATALOG_XML );

//rock.color.rockColorChartUtility.print( stStruct.stRockColor );
  }
*/
  /** Method readSoilColorChart()
   * <p> This method will read in the soil color chart
   *

  private void readSoilColorChart()
  {
    ReadColorChartXML pData =
        new ReadColorChartXML(
            rock.color.io.ReadColorChartXML.URL );

    stStruct.stSoilColor =
        pData.Process( cmn.cmnStruct.SOIL_COLOR_CATALOG_XML );

//rock.color.rockColorChartUtility.print( stStruct.stSoilColor );
  }
*/
  /** Method readRockColumn()
   * <p> This method will read in the Rock Column Lithology Data
   *

  private void readRockColumn()
  {
    ReadRockColumnSymbolsXMLFile pData =
        new ReadRockColumnSymbolsXMLFile(
            parse.io.ReadRockColumnSymbolsXMLFile.URL );

    stStruct.stParseRock =
        pData.Process( cmn.cmnStruct.ROCK_LITHOLOGY_XML );

//parse.parseRockColumnSymbolsUtility.print( stStruct.stParseRock );
  }
*/
  /* ======================================================================= *
   * --------------------------- APPLET METHODS ---------------------------- *
   * ======================================================================= */

  /** Method start()
   *  <p> This method will start the applet
   */

  public void start() { }

  /** Method stop()
   *  <p> This method will stop the applet
   */

  public void stop() { }

  /** Method destroy()
   *  <p> This method will destroy the applet
   */

  public void destroy() { }

  /** Method getAppletInfo()
   *  <p> This method will return Applet information
   *  @return - Biostratigraphy Profile Plot
   */

  public String getAppletInfo() { return ("LAS IO Applet"); }

  /** Method getParameterInfo()
   *  <p> This method will get the parameter info
   *  @return pinfo - Array of Parameters Expected.
   */

  public String[][] getParameterInfo()
  {
    String[][] pinfo = { {"sKID",  "String", "User Primary Key"}, };

    return pinfo;
  }

  /** Method main()
   *  <p> This method will allow this Module to run either as an Application or
   *      an Applet.
   *  @param args = The Arguments passed in to query for the production data
   */

  public static void main(String[] args)
  {
    Profile applet = new Profile();

    applet.isStandalone = true;
    applet.sKID_Standalone = new String("0");
    if (args.length > 0)
      applet.sKID_Standalone = new String(args[0]);

    applet.init();
    applet.start();
  }
}


/*
 *  @version 1.1 02/15/2012
 *  @author  John Victorine
 */
